#!/usr/local/bin/perl

@ARPTYPE = ("REQUEST", "REPLY");
@TCPTYPE = ("tcp", "ack");
@MACTYPE = ("RTS", "CTS", "ACK");
@TORATYPE = ("BEACON", "HELLO", "HELLO_ACK", "QUERY", "UPDATE", "CLEAR");
@DSRTYPE = ("REQUEST", "REPLY", "ERROR");
@CBRTYPE = ("UNIQUE");

$MACTYPE_RTS = hex("1b");
$MACTYPE_CTS = hex("1c");
$MACTYPE_ACK = hex("1d");
$MACTYPE_DATA = hex("20");

$infile = 0;			# input file
$time = 0;			# length of the simulation
$line = 0;			# line number of input file

$PACKET_SIZE = 1460;		# Data bytes in a TCP Packet
$MAX_ROUTE_LENGTH = 10;		# Maximum Route Length
$MAX_NODES = 0;			# Number of nodes
$TOTALS = 0;			# Number of nodes + 1;

$OP_TX = 0;	# transmit
$OP_RX = 1;	# receive
$OP_SD = 2;	# sender drop
$OP_RD = 3;	# receiver drop
$OPT_RTLEN = 4;	# differences in optimal vs actual route lengths
$ACT_RTLEN = 5;	# array of actual route lengths
$GOD_RTLEN = 6;	# array of optimal path len when pkt sent

$CNT_P = 0;	# packet count
$CNT_B = 1;	# byte count
$CNT_R = 2;	# MAC layer retransmit count
$CNT_S = 3;	# TCPsequence number count
$CNT_F = 4;	# Packets forwarded
$CNT_L = 5;	# Count > MAX_ROUTE_LENGTH routes

$dsr_hdr_bytes = 0;  # how many SR bytes used in data pkt headers?

# ======================================================================
# Initialization Routines
# ======================================================================
sub init_type {

	$PKT = shift(@_);

	foreach $P (@_) {

		foreach $OP ($OP_TX, $OP_RX, $OP_SD, $OP_RD) {
			$PKT->{$P}->[$OP]->[$CNT_P] = 0;
			$PKT->{$P}->[$OP]->[$CNT_B] = 0;
			$PKT->{$P}->[$OP]->[$CNT_R] = 0;
			$PKT->{$P}->[$OP]->[$CNT_S] = 0;
			$PKT->{$P}->[$OP]->[$CNT_F] = 0;
		}
	}

	foreach $OP ($OP_TX, $OP_RX, $OP_SD, $OP_RD) {

		$PKT->[$TOTALS]->[$OP]->[$CNT_P] = 0;
		$PKT->[$TOTALS]->[$OP]->[$CNT_B] = 0;
		$PKT->[$TOTALS]->[$OP]->[$CNT_R] = 0;
		$PKT->[$TOTALS]->[$OP]->[$CNT_S] = 0;
		$PKT->[$TOTALS]->[$OP]->[$CNT_F] = 0;
	}
}


sub init_tcp {

	for($src = 1; $src <= $MAX_NODES; $src += 1) {
		for($dst = 1; $dst <= $MAX_NODES; $dst += 1) {
			foreach $pkt (@TCPTYPE) {
				foreach $op ($OP_TX, $OP_RX, $OP_SD, $OP_RD) {
					TCP->[$src]->[$dst]->{$pkt}->[$op]->[$CNT_P] = 0;
					TCP->[$src]->[$dst]->{$pkt}->[$op]->[$CNT_B] = 0;
					TCP->[$src]->[$dst]->{$pkt}->[$op]->[$CNT_R] = 0;
					TCP->[$src]->[$dst]->{$pkt}->[$op]->[$CNT_S] = 0;
					TCP->[$src]->[$dst]->{$pkt}->[$op]->[$CNT_F] = 0;
				}
				#
				# Initialize route length counters.
				#
				for($i = 0; $i < $MAX_ROUTE_LENGTH; $i += 1) {
					TCP->[$src]->[$dst]->{$pkt}->[$OPT_RTLEN]->[$i] = 0;
				}
				for($i = 1; $i <= $MAX_ROUTE_LENGTH; $i += 1) {
					TCP->[$src]->[$dst]->{$pkt}->[$ACT_RTLEN]->[$i] = 0;
				}
			}
		}
	}
}

# ======================================================================
# Output Routines
# ======================================================================
sub print_totals {

	$A = shift(@_);
	$B = shift(@_);
	$C = shift(@_);


	if ($B eq "DSR" && $C eq "TOTALS") {
	    printf("\t\tDSR bytes includes %d bytes in TCP headers\n",
		   $dsr_hdr_bytes);
	    $A->[$OP_TX]->[$CNT_B] += $dsr_hdr_bytes;
	}

	printf("\t%s %s Transmitted\n", $B, $C);
	printf("\t\tPackets:       %d\n", $A->[$OP_TX]->[$CNT_P]);
	printf("\t\tBytes:         %d\n", $A->[$OP_TX]->[$CNT_B]);
	printf("\t\tPackets Rxmt:  %d\n", $A->[$OP_TX]->[$CNT_R]);
	printf("\t\tBitRate:       %f KB/sec, %f bps\n",
		$A->[$OP_TX]->[$CNT_B] / (1024 * $time),
		$A->[$OP_TX]->[$CNT_B] * 8 / $time);

	if ($B eq "DSR" && $C eq "TOTALS") {
	    $A->[$OP_TX]->[$CNT_B] -= $dsr_hdr_bytes;
	    # prevent double counting the bytes later on
	}

	printf("\t%s %s Received\n", $B, $C);
	printf("\t\tPackets:       %d\n", $A->[$OP_RX]->[$CNT_P]);
	printf("\t\tBytes:         %d\n", $A->[$OP_RX]->[$CNT_B]);
	printf("\t\tPackets Rxmt   %d\n", $A->[$OP_RX]->[$CNT_R]);
	printf("\t\tBitRate:       %f KB/sec, %f bps\n",
		$A->[$OP_RX]->[$CNT_B] / (1024 * $time),
		$A->[$OP_RX]->[$CNT_B] * 8 / $time);

	printf("\t%s %s Sender Drops\n", $B, $C);
	printf("\t\tPackets:       %d\n", $A->[$OP_SD]->[$CNT_P]);
	printf("\t\tBytes:         %d\n", $A->[$OP_SD]->[$CNT_B]);
	printf("\t\tPackets Rxmt:  %d\n", $A->[$OP_SD]->[$CNT_R]);

	printf("\t%s %s Receiver Drops\n", $B, $C);
	printf("\t\tPackets:       %d\n", $A->[$OP_RD]->[$CNT_P]);
	printf("\t\tBytes:         %d\n", $A->[$OP_RD]->[$CNT_B]);
	printf("\t\tPackets Rxmt:  %d\n", $A->[$OP_RD]->[$CNT_R]);

	#
	#  If we did not TRY TO SEND or SEND any
	#  packets, then don't bother with the rest.
	#
	if($A->[$OP_SD]->[$CNT_P] + 
	   $A->[$OP_TX]->[$CNT_P] == 0) {
		return;
	}

	$pktmiss = $A->[$OP_TX]->[$CNT_P] -
		  ($A->[$OP_RX]->[$CNT_P] +
                   $A->[$OP_RD]->[$CNT_P]);

	$pktdrop = ($A->[$OP_SD]->[$CNT_P] +
		    $A->[$OP_RD]->[$CNT_P]) /
		   ($A->[$OP_SD]->[$CNT_P] +
		    $A->[$OP_TX]->[$CNT_P]);

	printf("\n\t%s %s Missing:               %d\n", $B, $C, $pktmiss);
	printf("\t%s %s %% Drops:               %3.2f\n", $B, $C, $pktdrop * 100);
}

sub show_totals {

	$PKT = shift(@_);

	printf("\n%s Packets ========================================\n", $PKT);

	if($PKT eq "TOTAL") {
		print_totals($PKT, $PKT, "");
	}
	else {
		print_totals($PKT->[$TOTALS], $PKT, "TOTALS");
	}

	if(! @_) {
		return;
	}

	#
	# Individual Packet Formats
	#
	$tcnt = $PKT->[$TOTALS]->[$OP_TX]->[$CNT_P];
	while(@_) {
		$P = shift(@_);

		$tcnt -= $PKT->{$P}->[$OP_TX]->[$CNT_P];

		printf("\n\t%s %s Packets\n", $PKT, $P);

		print_totals($PKT->{$P}, $PKT, $P);
	}

	if($tcnt != 0) {
		print stderr "Error in sent totals.\n";
		# exit 1;
	}
}

# ======================================================================
# ARP Packet Processing Routines
# ======================================================================
sub process_arp {
	$OP = shift(@_);		# send, receive, drop
	$RXMT = shift(@_);		# retransmission (YES/NO)
	$SIZE = shift(@_);		# packet size
	$LINE = shift(@_);		# line from log file

	if($LINE =~ / ------- \[(w+) (\d+)\/(\d+) (\d+)\/(\d+)\]/o) {
		$pkt = $1;

		if( $pkt != "REQUEST" && $pkt != "REPLY" ) {
			print stderr "Invalid ARP packet.\n";
			exit 1;
		}
		ARP->{$pkt}->[$OP]->[$CNT_P] += 1;
		ARP->{$pkt}->[$OP]->[$CNT_R] += $RXMT;
		ARP->{$pkt}->[$OP]->[$CNT_B] += $SIZE;
	}
	else {
		print stderr "ARP Logging error\n$LINE\n";
		exit 1;
	}

	ARP->[$TOTALS]->[$OP]->[$CNT_P] += 1;
	ARP->[$TOTALS]->[$OP]->[$CNT_R] += $RXMT;
	ARP->[$TOTALS]->[$OP]->[$CNT_B] += $SIZE;
}

# ======================================================================
# TCP Packet Processing Routines
# ======================================================================
sub process_tcp {
	$OP = shift(@_);		# send, receive, drop
	$RXMT = shift(@_);		# retransmission (YES/NO)
	$SIZE = shift(@_);		# packet size
	$LINE = shift(@_);		# line from log file

	if($LINE =~ / (\d+) (\D+) \d+ ------- \[(\d+):(\d+) (\d+):(\d+) \d+\] \[(\d+) \d+\] (\d+) (\d+)/o) {
		$node = $1;
		$pkt = $2;
		$src_ipaddr = $3;
		$src_ipport = $4;
		$dst_ipaddr = $5;
		$dst_ipport = $6;
		$tcp_seqno = $7;
		$rt_len = $8;			# length of route
		$opt_rt_len = $9;		# optimal length

		if( $pkt ne "tcp" && $pkt ne "ack" ) {
			print stderr "Invalid TCP packet.\n";
			exit 1;
		}

		#
		#  Avoid counting TCP packets multiple times if they
		#  are forwarded across multiples hops.
		#
		if($node != $src_ipaddr && $node != $dst_ipaddr) {
			#
			#  Count packets that you forward on behalf of
			#  other nodes - index by destination.  Don't
			#  count MAC retransmission attempts.
			#
			if($OP == $OP_TX && $RXMT == 0) {
				TCP->[$node]->[$dst_ipaddr]->{$pkt}->[$OP]->[$CNT_F] += 1;
			}
			return;
		}

		if($node == $dst_ipaddr) {
			$subnode = $src_ipaddr;
		}
		elsif ($node == $src_ipaddr) {
			$subnode = $dst_ipaddr;
		}

		TCP->[$node]->[$subnode]->{$pkt}->[$OP]->[$CNT_P] += 1;
		TCP->[$node]->[$subnode]->{$pkt}->[$OP]->[$CNT_B] += $SIZE;
		TCP->[$node]->[$subnode]->{$pkt}->[$OP]->[$CNT_R] += $RXMT;

		#
		#  If this is a packet reception, log the difference
		#  between the optimal path length and the actual length.
		#
		if($OP == $OP_RX) {

			if($rt_len <= 0 ) {
				print stderr "$LINE";
				print stderr "Actual RT Length $rt_len\n";
				# exit 1;
			}

			$len = $rt_len - $opt_rt_len;
			if($len < 0) {
				print stderr "$LINE";
				print stderr "Actual RT Length < Optimal.\n";
				$len = 0;
				# exit 1;
			}

			if($len > $MAX_ROUTE_LENGTH) {
				print "$LINE";
				print "Actual RT Length > Optimal + $MAX_ROUTE_LENGTH.\n";
				TCP->[$node]->[$subnode]->{$pkt}->[$OPT_RTLEN]->[$MAX_ROUTE_LENGTH + 1]->[$CNT_P] += 1;
				TCP->[$node]->[$subnode]->{$pkt}->[$OPT_RTLEN]->[$MAX_ROUTE_LENGTH + 1]->[$CNT_L] += $len;
			}
			else {
				TCP->[$node]->[$subnode]->{$pkt}->[$OPT_RTLEN]->[$len] += 1;
			}

			if($rtlen > $MAX_ROUTE_LENGTH) {
				TCP->[$node]->[$subnode]->{$pkt}->[$ACT_RTLEN]->[$MAX_ROUTE_LENGTH + 1]->[$CNT_P] += 1;
				TCP->[$node]->[$subnode]->{$pkt}->[$ACT_RTLEN]->[$MAX_ROUTE_LENGTH + 1]->[$CNT_L] += $len;
			}
			else {
				TCP->[$node]->[$subnode]->{$pkt}->[$ACT_RTLEN]->[$rt_len] += 1;
			}
		}

		if(TCP->[$node]->[$subnode]->{$pkt}->[$OP]->[$CNT_S] < $tcp_seqno) {
			TCP->[$node]->[$subnode]->{$pkt}->[$OP]->[$CNT_S] = $tcp_seqno;
		}
		if($OP == $OP_TX) {
		    $dsr_hdr_bytes += $sr_bytes;
		}
	}
	else {
		print stderr "TCP Logging error\n$LINE\n";
		exit 1;
	}

	TCP->[$TOTALS]->[$OP]->[$CNT_P] += 1;
	TCP->[$TOTALS]->[$OP]->[$CNT_B] += $SIZE;
	TCP->[$TOTALS]->[$OP]->[$CNT_R] += $RXMT;
}

sub show_cbr_totals {
    show_totals(CBR, @CBRTYPE);

    printf("\nDistributions:\n");    

    $sum = 0;
    printf("\tOptimal Path Len on sending\n");
    for($i = 1; $i <= $MAX_ROUTE_LENGTH; $i += 1) {
	printf("\t\t Optimal Len %d : %d\n",$i,
	       CBR->[$TOTALS]->[$GOD_RTLEN]->[$i]);
	$sum = $sum + CBR->[$TOTALS]->[$GOD_RTLEN]->[$i];
    }
    printf("\t\t Optimal Len Inf : %d\n",
	   CBR->[$TOTALS]->[$GOD_RTLEN]->[$MAX_ROUTE_LENGTH + 1]->[$CNT_P]);
    printf("\t\t Optimal Len Sum %d\n",$sum);

    $sum = 0;
    printf("\tActual Path Len on receiving\n");
    for($i = 1; $i <= $MAX_ROUTE_LENGTH; $i += 1) {
	printf("\t\t Actual Len %d : %d\n",$i,
	       CBR->[$TOTALS]->[$ACT_RTLEN]->[$i]);
	$sum = $sum + CBR->[$TOTALS]->[$ACT_RTLEN]->[$i];
    }
    printf("\t\t Actual Len >%d : %d\n", $MAX_ROUTE_LENGTH,
	   CBR->[$TOTALS]->[$GOD_RTLEN]->[$MAX_ROUTE_LENGTH + 1]->[$CNT_P]);
    printf("\t\t Actual Len Sum %d\n",$sum);

    $sum = 0;
    printf("\tActual - Optimal Path Len on receiving\n");
    for($i = 0; $i <= $MAX_ROUTE_LENGTH; $i += 1) {
	printf("\t\t Actual - Optimal Len %d : %d\n",$i,
	       CBR->[$TOTALS]->[$OPT_RTLEN]->[$i]);
	$usm = $sum + CBR->[$TOTALS]->[$OPT_RTLEN]->[$i];
    }
    printf("\t\t Actual - Optimal Len >%d : %d\n", $MAX_ROUTE_LENGTH,
	   CBR->[$TOTALS]->[$OPT_RTLEN]->[$MAX_ROUTE_LENGTH + 1]);
    printf("\t\t Actual  - Optimal Len Sum %d\n",$sum);

}

sub show_tcp_totals {

	$PKT = TCP;

	printf("\n%s Packets ========================================\n", $PKT);

	print_totals($PKT->[$TOTALS], $PKT, "TOTALS");

	#
	# Individual Nodes
	#
	for($src = 1; $src <= $MAX_NODES; $src += 1) {

		printf("Node: %d\n", $src);

		for($dst = 1; $dst <= $MAX_NODES; $dst += 1) {

			if($src == $dst) {
				next;
			}
		
			foreach $pkt (@TCPTYPE) {

				#
				# Outgoing Packets
				#
				if(TCP->[$src]->[$dst]->{$pkt}->[$OP_TX]->[$CNT_P]) {
					#
					# (1) packets transmitted
					# (2) highest sequence number transmitted
					# (3) Raw Throughput
					# (4) TCP Goodput
					#
					printf("%2d %2d --- %s  TX: %4d %4d %6.2f KB/sec  %6.2f KB/sec\n",
						$src, $dst, $pkt,
						TCP->[$src]->[$dst]->{$pkt}->[$OP_TX]->[$CNT_P],
						TCP->[$src]->[$dst]->{$pkt}->[$OP_TX]->[$CNT_S],
						TCP->[$src]->[$dst]->{$pkt}->[$OP_TX]->[$CNT_B] / (1024 * $time),
						TCP->[$src]->[$dst]->{$pkt}->[$OP_TX]->[$CNT_S] * $PACKET_SIZE / (1024 * $time));
				}

				#
				# Incoming Packets
				#
				if(TCP->[$src]->[$dst]->{$pkt}->[$OP_RX]->[$CNT_P] > 0) {
					#
					# (1) packets received
					# (2) highest sequence number received
					# (3) Raw Throughput
					# (4) TCP Goodput
					#
					printf("%2d %2d --- %s  RX: %4d %4d %6.2f KB/sec  %6.2f KB/sec\n",
						$src, $dst, $pkt,
						TCP->[$src]->[$dst]->{$pkt}->[$OP_RX]->[$CNT_P],
						TCP->[$src]->[$dst]->{$pkt}->[$OP_RX]->[$CNT_S],
						TCP->[$src]->[$dst]->{$pkt}->[$OP_RX]->[$CNT_B]/ (1024 * $time),
						TCP->[$src]->[$dst]->{$pkt}->[$OP_RX]->[$CNT_S] * $PACKET_SIZE / (1024 * $time));
					#
					#  Distribution of Route Lengths
					#
					for($i = 1; $i <= $MAX_ROUTE_LENGTH; $i += 1) {
						printf("%2d %2d --- %s  Route Length: %d, Packet Count: %d\n",
							$src, $dst, $pkt, $i, TCP->[$src]->[$dst]->{$pkt}->[$ACT_RTLEN]->[$i]);
					}
					#
					# Larger than MAX ROUTE LENGTHS
					#
					printf("%2d %2d --- %s  Route Length: %d, Packet Count: %d\n",
						$src, $dst, $pkt,
						TCP->[$src]->[$dst]->{$pkt}->[$ACT_RTLEN]->[$MAX_ROUTE_LENGTH+1]->[$CNT_L],
						TCP->[$src]->[$dst]->{$pkt}->[$ACT_RTLEN]->[$MAX_ROUTE_LENGTH+1]->[$CNT_P]);
					#
					# Actual-Optimal Route Lengths
					#
					for($i = 0; $i < $MAX_ROUTE_LENGTH; $i += 1) {
						printf("%2d %2d --- %s  Actual-Optimal: %d, Packet Count: %d\n",
							$src, $dst, $pkt, $i, TCP->[$src]->[$dst]->{$pkt}->[$OPT_RTLEN]->[$i]);
					}
					#
					# Larger than MAX OPTIMAL ROUTE LENGTHS
					#
					printf("%2d %2d --- %s  Actual-Optimal: %d, Packet Count: %d\n",
						$src, $dst, $pkt,
						TCP->[$src]->[$dst]->{$pkt}->[$OPT_RTLEN]->[$MAX_ROUTE_LENGTH+1]->[$CNT_L],
						TCP->[$src]->[$dst]->{$pkt}->[$OPT_RTLEN]->[$MAX_ROUTE_LENGTH+1]->[$CNT_P]);

				}

				#
				#  Forwarding Packets
				#
				if(TCP->[$src]->[$dst]->{$pkt}->[$OP_TX]->[$CNT_F] > 0) {
					printf("  %2d --- forwarded %5d %s packets\n",
						$dst, TCP->[$src]->[$dst]->{$pkt}->[$OP_TX]->[$CNT_F], $pkt);
				}

				next;

				printf("\tDest: %d --- %s\n", $dst, $pkt);

				printf("\t\t\tTX: %4d %4d %6.2f KB/sec\n",
					TCP->[$src]->[$dst]->{$pkt}->[$OP_TX]->[$CNT_P],
					TCP->[$src]->[$dst]->{$pkt}->[$OP_TX]->[$CNT_S],
					TCP->[$src]->[$dst]->{$pkt}->[$OP_TX]->[$CNT_B]/ (1024 * $time));

				printf("\t\t\tRX: %4d %4d %6.2f KB/sec\n",
					TCP->[$src]->[$dst]->{$pkt}->[$OP_RX]->[$CNT_P],
					TCP->[$src]->[$dst]->{$pkt}->[$OP_RX]->[$CNT_S],
					TCP->[$src]->[$dst]->{$pkt}->[$OP_RX]->[$CNT_B]/ (1024 * $time));

				printf("\t\t\tSD: %d, RD: %d\n",
					TCP->[$src]->[$dst]->{$pkt}->[$OP_SD]->[$CNT_P],
					TCP->[$src]->[$dst]->{$pkt}->[$OP_RD]->[$CNT_P]);

			}
		}
	}
	printf("\n");
}


# ======================================================================
# CBR Packet Processing Routines
# ======================================================================
#s 5.73300 [20 42 2b 29 800] 41 cbr 1060 ------- 0 [41:2 43:0] 20 22 [0 0x8 0]
sub process_cbr {
	$OP = shift(@_);		# send, receive, drop
	$RXMT = shift(@_);		# retransmission (YES/NO)
	$SIZE = shift(@_);		# packet size
	$LINE = shift(@_);		# line from log file

	if($LINE =~ /^.*\] (\d+) cbr .*\[(\d+):(\d+) (\d+):(\d+) \d+\] \[\d+\] (\d+) (\d+)/o) {
	    $node = $1;
	    $src_ipaddr = $2;
	    $src_ipport = $3;
	    $dst_ipaddr = $4;
	    $dst_ipport = $5;
	    $rt_len = $6;
	    $opt_rt_len = $7;		# optimal length
#		$sr_bytes = hex($7);		# bytes in sr hdr if using DSR

	    $kind = "UNIQUE";
	    if($OP == $OP_TX && $node == $src_ipaddr && $rl_len == 0) {
		$dsr_hdr_bytes += $sr_bytes;
		CBR->{$kind}->[$OP]->[$CNT_P] += 1;
		CBR->{$kind}->[$OP]->[$CNT_R] += $RXMT;
		CBR->{$kind}->[$OP]->[$CNT_B] += $SIZE;

		if ($opt_rt_len > $MAX_ROUTE_LENGTH) {
		    CBR->[$TOTALS]->[$GOD_RTLEN]->[$MAX_ROUTE_LENGTH + 1]->[$CNT_P] += 1;
		} else {
		    CBR->[$TOTALS]->[$GOD_RTLEN]->[$opt_rt_len] += 1;
		}
	    } elsif($OP == $OP_RX && $node == $dst_ipaddr) {
		CBR->{$kind}->[$OP]->[$CNT_P] += 1;
		CBR->{$kind}->[$OP]->[$CNT_R] += $RXMT;
		CBR->{$kind}->[$OP]->[$CNT_B] += $SIZE;
		
		if($rt_len <= 0 ) {
		    print stderr "$LINE";
		    print stderr "Actual RT Length $rt_len\n";
		    exit 1;
		}
		
		$len = $rt_len - $opt_rt_len;
		if($len < 0) {
		    print stderr "$LINE";
		    print stderr "Actual RT Length < Optimal.\n";
		    $len = 0;
		} elsif($len > $MAX_ROUTE_LENGTH) {
		    print "$LINE";
		    print "Actual RT Length > Optimal + $MAX_ROUTE_LENGTH.\n";
		    CBR->[$TOTALS]->[$OPT_RTLEN]->[$MAX_ROUTE_LENGTH + 1]->[$CNT_P] += 1;
		    CBR->[$TOTALS]->[$OPT_RTLEN]->[$MAX_ROUTE_LENGTH + 1]->[$CNT_L] += $len;
		}
		else {
		    CBR->[$TOTALS]->[$OPT_RTLEN]->[$len] += 1;
		}
		
		if($rtlen > $MAX_ROUTE_LENGTH) {
		    CBR->[$TOTALS]->[$ACT_RTLEN]->[$MAX_ROUTE_LENGTH + 1]->[$CNT_P] += 1;
		    CBR->[$TOTALS]->[$ACT_RTLEN]->[$MAX_ROUTE_LENGTH + 1]->[$CNT_L] += $len;
		}
		else {
		    CBR->[$TOTALS]->[$ACT_RTLEN]->[$rt_len] += 1;
		}
	    }
	} else {
	    print stderr "CBR Logging error\n$LINE\n";
	    exit 1;
	}
	
	CBR->[$TOTALS]->[$OP]->[$CNT_P] += 1;
	CBR->[$TOTALS]->[$OP]->[$CNT_R] += $RXMT;
	CBR->[$TOTALS]->[$OP]->[$CNT_B] += $SIZE;
}

# ======================================================================
# MAC Packet Processing Routines
# ======================================================================
sub process_msg {
        $OP = shift(@_);                # send, receive, drop
        $RXMT = shift(@_);              # retransmission (YES/NO)
        $SIZE = shift(@_);              # packet size
        $LINE = shift(@_);              # line from log file

        MSG->[$TOTALS]->[$OP]->[$CNT_P] += 1;
        MSG->[$TOTALS]->[$OP]->[$CNT_R] += $RXMT;
        MSG->[$TOTALS]->[$OP]->[$CNT_B] += $SIZE;
}

# ======================================================================
# MAC Packet Processing Routines
# ======================================================================
sub process_mac {
	$OP = shift(@_);		# send, receive, drop
	$RXMT = shift(@_);		# retransmission (YES/NO)
	$SIZE = shift(@_);		# packet size
	$LINE = shift(@_);		# line from log file

	if($LINE =~ /\[(\w+) (\w+) (\w+) (\w+) (\w+)\]/o) {
		$t = hex($1) & 255;

		if($t == $MACTYPE_RTS) {
			$pkt = "RTS";
		}
		elsif($t == $MACTYPE_CTS) {
			$pkt = "CTS";
		}
		elsif($t == $MACTYPE_ACK) {
			$pkt = "ACK";
		}
		else {
			print stderr "Invalid MAC packet.\n";
			exit 1;
		}

		MAC->{$pkt}->[$OP]->[$CNT_P] += 1;
		MAC->{$pkt}->[$OP]->[$CNT_R] += $RXMT;
		MAC->{$pkt}->[$OP]->[$CNT_B] += $SIZE;
	}
	else {
		print stderr "MAC Logging error\n$LINE\n";
		exit 1;
	}

	MAC->[$TOTALS]->[$OP]->[$CNT_P] += 1;
	MAC->[$TOTALS]->[$OP]->[$CNT_R] += $RXMT;
	MAC->[$TOTALS]->[$OP]->[$CNT_B] += $SIZE;
}


# ======================================================================
# Main Procedure
# ======================================================================
if($#ARGV != 1) {
	print stderr "\nusage: $0 <input file> <num nodes>\n\n";
	exit 1;
}

if(! open $infile, $ARGV[0]) {
	print stderr "Could not open $ARGV[0]\n";
	exit 1;
}

$MAX_NODES = $ARGV[1];
$TOTALS = $MAX_NODES + 1;

#
# Initialize Data Structures
#
init_type(MAC, @MACTYPE);
init_type(CBR, @CBRTYPE);
init_type(ARP, @ARPTYPE);
init_type(DSR, @DSRTYPE);
init_type(TORA, @TORATYPE);
init_type(TCP, @TCPTYPE); init_tcp();

while(<$infile>) {

	$line += 1;

	if(/^[rsfD] (\d+).\d+ \D+ \D+ \[(\w+) \w+ (\w+) (\w+) (\w+)\] (\d+) (\D+) (\d+)/o) {
		$time = $1;
		$mac_type = hex($2);
		$mac_dst = hex($3);
		$mac_src = hex($4);
		$ether_type = hex($5);
		$node = $6;
		$pkttype = $7;
		$pktsize = $8;

		$retransmit = 0;

		if($mac_type > 255) {
			$retransmit = 1;
			$mac_type &= 255;
		}

		if(/^s / || /^f /) {
			$opcode = $OP_TX;
		}
		elsif(/^r /) {
			$opcode = $OP_RX;
		}
		elsif(/^D / || /^d /) {
			if($mac_src == $node) {
				$opcode = $OP_SD;
			}
			else {
				$opcode = $OP_RD;
			}
		}
		else {
			print stderr "Invalid OPERATION\n$_\n";
			exit 1;
		}

		if($mac_type == $MACTYPE_RTS ||
		   $mac_type == $MACTYPE_CTS ||
		   $mac_type == $MACTYPE_ACK) {
			process_mac($opcode, $retransmit, $pktsize, $_);
		}
		elsif($mac_type == $MACTYPE_DATA || $mac_type == 0) {
			if($pkttype eq "ARP") {
#				process_arp($opcode, $retransmit, $pktsize, $_);
			}
			elsif($pkttype eq "tcp" ||
			      $pkttype eq "ack") {
				process_tcp($opcode, $retransmit, $pktsize, $_);
			}
			elsif($pkttype eq "TORA") {
#				process_tora($opcode, $retransmit, $pktsize, $_);
			}
			elsif($pkttype eq "cbr") {
				process_cbr($opcode, $retransmit, $pktsize, $_);
			}
			elsif($pkttype eq "DSR") {
#				process_dsr($opcode, $retransmit, $pktsize, $_);
			}
			elsif($pkttype eq "message") {
			    process_msg($opcode, $retransmit, $pktsize, $_);
			}	
			else {
			    print stderr "Invalid packet type $pkttype\n";
			    exit 1;
			}
		}
		else {
			print stderr "Invalid packet type $pkttype\n";
			exit 1;
		}
	}
	#
	# Mobility Logging
	#
	elsif(/^M /o) {
	}
	#
	# TORA Logging
	#
	elsif(/^T /o) {
	}
	#
	# DSR Logging
	#
	elsif(/^S/o) {
	}
	#
	# GOD Logging
	#
	elsif(/^G/o) {
	}
	else {
		print stderr "Error, $line: '$_'\n";
		exit 1;
	}
}

$time += 1;		# accounts for fractional part


#
# Compute Totals for the Simulation
#
foreach $PKT (MAC, ARP, DSR, TORA, TCP, CBR) {
	foreach $OP ($OP_TX, $OP_RX, $OP_SD, $OP_RD) {
		TOTAL->[$OP]->[$CNT_P] += $PKT->[$TOTALS]->[$OP]->[$CNT_P];
		TOTAL->[$OP]->[$CNT_B] += $PKT->[$TOTALS]->[$OP]->[$CNT_B];
		TOTAL->[$OP]->[$CNT_R] += $PKT->[$TOTALS]->[$OP]->[$CNT_R];
	}
}

show_totals(MAC, @MACTYPE);
show_totals(MSG, @MSGTYPE);
show_cbr_totals();
show_tcp_totals();

printf("\nSimulation Time: %d\n", $time);
show_totals(TOTAL);

