/*
 * Copyright (c) 1994 Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *	This product includes software developed by the Computer Systems
 *	Engineering Group at Lawrence Berkeley Laboratory.
 * 4. Neither the name of the University nor of the Laboratory may be used
 *    to endorse or promote products derived from this software without
 *    specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * @(#) $Header: /home/cr1/benjie/cvsroot/ns/ns-span/scheduler.cc,v 1.1.1.1 2001/11/15 19:43:13 benjie Exp $
 */

#ifndef lint
static const char rcsid[] =
    "@(#) $Header: /home/cr1/benjie/cvsroot/ns/ns-span/scheduler.cc,v 1.1.1.1 2001/11/15 19:43:13 benjie Exp $ (LBL)";
#endif

#include <stdlib.h>
#include "config.h"
#include "scheduler.h"
#include <stdio.h>

#ifdef MONARCH
#include <assert.h>
#define DEBUG_INSERTIONS
#endif

#ifdef __FreeBSD__
#include <machine/cpufunc.h>
#include <machine/perfmon.h>
#include <fcntl.h>
#include <unistd.h>
#endif /* __FreeBSD__ */

#include "ostream.h"

#ifdef MEMDEBUG_SIMULATIONS
#include "mem-trace.h"
#endif

Scheduler* Scheduler::instance_;
int Scheduler::uid_;

Scheduler::Scheduler() : clock_(0.0), time_floor_(0.0), halted_(0)
{
	instance_ = this;
}

/*
 * Schedule an event delay time units into the future.
 * The event will be dispatched to the specified handler.
 * We use a relative time to avoid the problem of scheduling
 * something in the past.
 */
void Scheduler::schedule(Handler* h, Event* e, double delay,
			 const char *description = 0)
{ 
    if (description)
	strcpy(e->description_, description);
#ifdef MONARCH
        if (delay < 0.0)
	    assert(0);
#endif
	e->uid_ = uid_++;
	e->handler_ = h;
	double t = clock_ + delay;
	e->time_ = t;
	insert(e);
}

void
Scheduler::run()
{
        instance_ = this;
	Event *p;
	while ((p = deque()) && !halted_) {
		dispatch(p);
	}
}

/*
 * dispatch a single simulator event by setting the system
 * virtul clock to the event's timestamp and calling its handler.
 * Note that the event may have side effects of placing other items
 * in the scheduling queue
 */

void
Scheduler::dispatch(Event* p, double t)
{
	if (t < clock_)
		fprintf(stderr, "ns: scheduler going backwards in \
time from %f to %f for event %s.\n", clock_, t, 
			(p->description_[0] ? p->description_ : "[unknown]"));

//  	// the time of event dispatches must monotoniclly increasing -dam
//  	if (t < time_floor_)
//  	    assert(0);
//	time_floor_ = t;

	clock_ = t;
	p->uid_ = -p->uid_;	// being dispatched
	p->handler_->handle(p);	// dispatch
}

void
Scheduler::dispatch(Event* p)
{
	dispatch(p, p->time_);
}

class AtEvent : public Event {
public:
	char* proc_;
};

class AtHandler : public Handler {
public:
	void handle(Event* event);
} at_handler;

void AtHandler::handle(Event* e)
{
	AtEvent* at = (AtEvent*)e;
	Tcl::instance().eval(at->proc_);
	delete[] at->proc_;
	delete at;
}

int Scheduler::command(int argc, const char*const* argv)
{
	Tcl& tcl = Tcl::instance();
	if (argc == 2) {
		if (strcmp(argv[1], "run") == 0) {
			/* set global to 0 before calling object reset methods */
			reset();	// sets clock to zero
			run();
			return (TCL_OK);
		} else if (strcmp(argv[1], "now") == 0) {
			sprintf(tcl.buffer(), "%.17g", clock());
			tcl.result(tcl.buffer());
			return (TCL_OK);
		} else if (strcmp(argv[1], "resume") == 0) {
			halted_ = 0;
			run();
			return (TCL_OK);
		} else if (strcmp(argv[1], "halt") == 0) {
			halted_ = 1;
			return (TCL_OK);

                } else if (strcmp(argv[1], "clearMemTrace") == 0) {
#ifdef MEMDEBUG_SIMULATIONS
                        extern MemTrace *globalMemTrace;
                        if (globalMemTrace)
                                globalMemTrace->diff("Sim.");
#endif
                        return (TCL_OK);
		} else if (strcmp(argv[1], "is-running") == 0) {
			sprintf(tcl.buffer(), "%d", !halted_);
			return (TCL_OK);
		}
	} else if (argc == 3) {
		if (strcmp(argv[1], "at") == 0 ||
		    strcmp(argv[1], "cancel") == 0) {
			Event* p = lookup(atoi(argv[2]));
			if (p != 0) {
				/*XXX make sure it really is an atevent*/
				cancel(p);
				AtEvent* ae = (AtEvent*)p;
				delete[] ae->proc_;
				delete ae;
			}
		} else if (strcmp(argv[1], "at-now") == 0) {
                        const char* proc = argv[2];

			// "at [$ns now]" may not work because of tcl's 
			// string number resolution
                        AtEvent* e = new AtEvent;
                        int n = strlen(proc);
                        e->proc_ = new char[n + 1];
                        strcpy(e->proc_, proc);
                        schedule(&at_handler, e, 0);
                        sprintf(tcl.buffer(), "%u", e->uid_);
                        tcl.result(tcl.buffer());
                }
		return (TCL_OK);
	} else if (argc == 4) {
		if (strcmp(argv[1], "at") == 0) {
			double t = atof(argv[2]);
			const char* proc = argv[3];
	
			AtEvent* e = new AtEvent;
			int n = strlen(proc);
			e->proc_ = new char[n + 1];
			strcpy(e->proc_, proc);
			double delay = t - clock();
			if (delay < 0) {
				tcl.result("can't schedule command in past");
				return (TCL_ERROR);
			}
			schedule(&at_handler, e, delay);
			sprintf(tcl.buffer(), "%u", e->uid_);
			tcl.result(tcl.buffer());
			return (TCL_OK);
		}
	}
	return (TclObject::command(argc, argv));
}

void
Scheduler::dumpq()
{
	Event *p;

	printf("Contents of scheduler queue (events) [cur time: %f]---\n",
		clock());
	while ((p = deque()) != NULL) {
		printf("t:%f uid: %d handler: %p\n",
			p->time_, p->uid_, p->handler_);
	}
}

#if defined(MONARCH) && defined(DEBUG_INSERTIONS)
#include <cmu/debug.h>
HashSet event_check("event_check");
int event_count = 0;
#endif

/* XXX need to implement a calendar queue as well */

class ListScheduler : public Scheduler {
public:
	inline ListScheduler() : queue_(0) {}
	virtual void run();
	virtual void cancel(Event*);
	virtual void insert(Event*);
	virtual Event* lookup(int uid);
	virtual Event* deque();		// next event (removes from q)
protected:
	Event* queue_;
};

static class ListSchedulerClass : public TclClass {
public:
	ListSchedulerClass() : TclClass("Scheduler/List") {}
	TclObject* create(int /* argc */, const char*const* /* argv */) {
		return (new ListScheduler);
	}
} class_list_sched;


void ListScheduler::insert(Event* e)
{
	double t = e->time_;
	Event** p;

#if defined(MONARCH) && defined(DEBUG_INSERTIONS)
	assert(t >= time_floor_); // can't ask for time to go backwards
	struct hb ** hbp = event_check.find(e);
	if (0 != hbp) {
	    printf("scheduler.cc:284\n");//DEBUG
	    abort();
	}
	event_check.insert(e);
	event_count++;
#endif
	for (p = &queue_; *p != 0; p = &(*p)->next_)
	  {
	    // check for dupes
	    if (*p == e) {
	    printf("scheduler.cc:294\n");//DEBUG
		abort();
	    }

		if (t < (*p)->time_)
			break;
	  }
	e->next_ = *p;
	*p = e;
}

/*
 * Cancel an event.  It is an error to call this routine
 * when the event is not actually in the queue.  The caller
 * must free the event if necessary; this routine only removes
 * it from the scheduler queue.
 */
void ListScheduler::cancel(Event* e)
{
	Event** p;
	for (p = &queue_; *p != e; p = &(*p)->next_)
	    if (*p == 0) {
	    printf("scheduler.cc:316\n");//DEBUG
			abort();
	    }

	*p = (*p)->next_;
	e->uid_ = - e->uid_;
#if defined(MONARCH) && defined(DEBUG_INSERTIONS)
	struct hb ** hbp = event_check.find(e);
	assert(hbp);
	event_check.remove(hbp);
#endif
}

Event* 
ListScheduler::lookup(int uid)
{
	Event* p;
	for (p = queue_; p != 0; p = p->next_)
		if (p->uid_ == uid)
			break;
	return (p);
}

Event*
ListScheduler::deque()
{ 
	Event* e = queue_;
	if (e) {
		queue_ = e->next_;

#if defined(MONARCH) && defined(DEBUG_INSERTIONS)
		struct hb ** hbp = event_check.find(e);
		assert(hbp);
		event_check.remove(hbp);      
#endif
	}
	return (e);
}

void 
ListScheduler::run()
{ 
	Scheduler::run();

#if defined(MONARCH) && defined(DEBUG_INSERTIONS)
	printf("  Num events: %d   Max scheduler queue: %d\n",
	       event_count, event_check.max_hb);
#endif
}

#include "heap.h"

class HeapScheduler : public Scheduler {
public:
	inline HeapScheduler() { hp_ = new Heap; } 
	virtual void cancel(Event* e) {
		if (e->uid_ <= 0)
			return;
		e->uid_ = - e->uid_;
		hp_->heap_delete((void*) e);
	}
	virtual void insert(Event* e) {
		hp_->heap_insert(e->time_, (void*) e);
	}
	virtual Event* lookup(int uid);
	virtual Event* deque();
protected:
	Heap* hp_;
};

static class HeapSchedulerClass : public TclClass {
public:
	HeapSchedulerClass() : TclClass("Scheduler/Heap") {}
	TclObject* create(int /* argc */, const char*const* /* argv */) {
		return (new HeapScheduler);
	}
} class_heap_sched;

Event* HeapScheduler::lookup(int uid)
{
        Event* p;
	
	for (p = (Event*) hp_->heap_iter_init(); p;
	     p = (Event*) hp_->heap_iter())
		if (p->uid_ == uid)
			break;
	return p;
}

Event*
HeapScheduler::deque()
{
	return ((Event*) hp_->heap_extract_min());
}

#ifndef MONARCH
/* Note: be advised that we believe this calendar scheduler has serious 
   bugs in it.  On complicated simulations with many events the scheduler 
   will occasionally *drop* events.  They go in, but they don't come out.  
   We don't recommend using this scheduler.

  The CMU Monarch Project -dam 8/5/98 */


/*
 * Calendar queue scheduler contributed by
 * David Wetherall <djw@juniper.lcs.mit.edu>
 * March 18, 1997.
 *
 * A calendar queue basically hashes events into buckets based on
 * arrival time.
 */

class CalendarScheduler : public Scheduler {
public:
	CalendarScheduler();
	virtual ~CalendarScheduler();
	virtual void run();
	virtual void cancel(Event*);
	virtual void insert(Event*);
	virtual Event* lookup(int uid);

protected:
	int resizeenabled_;
	double width_;
	double oneonwidth_;
	double buckettop_;
	int nbuckets_;
	int buckbits_;
	int lastbucket_;
	int top_threshold_;
	int bot_threshold_;
	Event** buckets_;
	int qsize_;

	virtual Event* deque();
	virtual void reinit(int nbuck, double bwidth, double start);
	virtual void resize(int newsize);
	virtual double newwidth();
};

static class CalendarSchedulerClass : public TclClass {
public:
	CalendarSchedulerClass() : TclClass("Scheduler/Calendar") {}
	TclObject* create(int /* argc */, const char*const* /* argv */) {
		return (new CalendarScheduler);
	}
} class_calendar_sched;

CalendarScheduler::CalendarScheduler() {
	reinit(2, 1.0, 0.0);
	resizeenabled_ = 1;
}

CalendarScheduler::~CalendarScheduler() {
	delete [] buckets_;
	qsize_ = 0;
}

void CalendarScheduler::insert(Event* e)
{
	// bucket number and address
	int i = (int)(((long)(e->time_ * oneonwidth_)) & buckbits_); 
	Event** p = buckets_ + i;

#ifdef MONARCH
	// check for dupes
	for( ; *p ; p = &(*p)->next_) {
		if(*p == e) {
			fprintf(stderr, "INSERTING DUPES (%x)\n", (int)e);
			exit(1);
		}
	}
	p = buckets_ + i;
#endif

	// insert event in stable time sorted order
	while ((*p != NULL) && (e->time_ >= (*p)->time_))
		p = &(*p)->next_;
  
	e->next_ = *p;
	*p = e;
  
	if (++qsize_ > top_threshold_) resize(2*nbuckets_);
}

Event* CalendarScheduler::deque()
{
	if (qsize_ == 0) return NULL;
	int i = lastbucket_;

	// check for an event this `year'
	do {
		Event* e = buckets_[i];
		if ((e != NULL) && (e->time_ < buckettop_)) {
			buckets_[i] = e->next_;
			lastbucket_ = i;
			clock_ = e->time_;
			if (--qsize_ < bot_threshold_) resize(nbuckets_/2);
			return e;
		} else {
			if (++i == nbuckets_) i = 0;
			buckettop_ += width_;
		}
	} while (i != lastbucket_);

	// or direct search for the minimum event
	int pos = 0;
	Event* min;
	do { min =  buckets_[pos++]; } while (min == NULL); pos--;

	for (int k = pos+1; k < nbuckets_; k++) {
		Event* e = buckets_[k];
		if ((e != NULL) && (e->time_ < min->time_)) {
			min = e; pos = k;
		}
	}
  
	// adjust year and resume
	lastbucket_ = pos;
	clock_ = min->time_;
	long n = (long)(clock_ * oneonwidth_);
	buckettop_ = (n + 1) * width_ + 0.5 * width_;

	return deque();
}

void CalendarScheduler::reinit(int nbuck, double bwidth, double start)
{
	buckets_ = new Event*[nbuck];
	memset(buckets_, 0, sizeof(Event*)*nbuck);
	width_ = bwidth;
	oneonwidth_ = 1.0 / width_;
	nbuckets_ = nbuck;
	buckbits_ = nbuck-1;
	qsize_ = 0;
	clock_ = start;
	long n = (long)(start * oneonwidth_);
	lastbucket_ = n % nbuck;
	buckettop_ = (n + 1) * width_ + 0.5 * width_;
	bot_threshold_ = nbuck / 2 - 2;
	top_threshold_ = 2 * nbuck;
}

void CalendarScheduler::resize(int newsize)
{
	if (!resizeenabled_) return;
	resizeenabled_ = 0;
	double bwidth = newwidth();
	Event** oldb = buckets_;
	int oldn = nbuckets_;

	// copy events to new buckets
	reinit(newsize, bwidth, clock_);
	for (int i = oldn-1; i >= 0; i--) {
		Event* e = oldb[i];
		while (e != NULL) {
			Event* en = e->next_;
			insert(e);
			e = en;
		}
	}
	delete [] oldb;
	resizeenabled_ = 1;
}

#define MIN_WIDTH (1.0e-6)
#define MAX_HOLD  25

double CalendarScheduler::newwidth()
{
	static Event* hold[MAX_HOLD];
	int nsamples;
  
	if (qsize_ < 2) return 1.0;
	if (qsize_ < 5) nsamples = qsize_;
	else nsamples = 5 + qsize_ / 10;
	if (nsamples > MAX_HOLD) nsamples = MAX_HOLD;

	// dequue and requeue sample events to gauge width
	double olp = clock_;
	double olt = buckettop_;
	int olb = lastbucket_;
	for (int i = 0; i < nsamples; i++) hold[i] = deque();
	for (int j = nsamples-1; j >= 0; j--) insert(hold[j]);
	clock_ = olp;
	buckettop_ = olt;
	lastbucket_ = olb;

	// try to work out average cluster separation
	double asep = (hold[nsamples-1]->time_ - hold[0]->time_) / (nsamples-1);
	double asep2 = 0.0;
	double min = (clock_ + 1.0) * MIN_WIDTH; 
	int count = 0;
  
	for (int k = 1; k < nsamples; k++) {
		double diff = hold[k]->time_ - hold[k-1]->time_;
		if (diff < 2.0*asep) { asep2 += diff; count++; }
	}
  
	// but don't let things get too small for numerical stability
	double nw = count ? 3.0*(asep2/count) : asep;
	if (nw < min) nw = min;
	return nw;
}

/*
 * Cancel an event.  It is an error to call this routine
 * when the event is not actually in the queue.  The caller
 * must free the event if necessary; this routine only removes
 * it from the scheduler queue.
 */
void CalendarScheduler::cancel(Event* e)
{
	int i = (int)(((long)(e->time_ * oneonwidth_)) & buckbits_);

	for (Event** p = buckets_ + i; (*p) != NULL; p = &(*p)->next_)
		if ((*p) == e) {
			(*p) = (*p)->next_;
			e->uid_ = - e->uid_;
			qsize_--;
			return;
		}
	printf("scheduler.cc:639\n");//DEBUG
	abort();
}

Event* CalendarScheduler::lookup(int uid)
{
	for (int i = 0; i < nbuckets_; i++)
		for (Event* p = buckets_[i]; p != NULL; p = p->next_)
			if (p->uid_== uid) return p;
	return NULL;
}

void CalendarScheduler::run()
{ 
	Event* p;
	/*XXX*/
	instance_ = this;
	while (((p = deque()) != NULL) && !halted_) {
		p->uid_ = - p->uid_;
		p->handler_->handle(p);
	}
}
#endif // ndef MONARCH

#ifndef WIN32
#include <sys/time.h>
#endif

/*
 * Really should instance the list/calendar/heap discipline
 * inside a RealTimeScheduler or VirtualTimeScheduler
 */

#ifdef notyet
class RealTimeScheduler : public CalendarScheduler {
#endif

class RealTimeScheduler : public ListScheduler {
public:
        RealTimeScheduler();
        virtual void run();
        double start() const { return start_; }
protected:
	virtual void dispatch(Event*, double);	// exec event, set clock_
	inline virtual void dispatch(Event *e) {Scheduler::dispatch(e);}
        void sync() { clock_ = tod(); }
        int rwait(double);      // sleep
        double tod();
        double slop_;   // allowed drift between real-time and virt time
        double start_;  // starting time
};

static class RealTimeSchedulerClass : public TclClass {
public:
        RealTimeSchedulerClass() : TclClass("Scheduler/RealTime") {}
        TclObject* create(int /* argc */, const char*const* /* argv */) {
                return (new RealTimeScheduler);
        }
} class_realtime_sched;

RealTimeScheduler::RealTimeScheduler() : start_(0.0)
{
        start_ = tod();
        bind("maxslop_", &slop_);
}


double
RealTimeScheduler::tod()
{
        timeval tv;
        gettimeofday(&tv, 0);
        double s = tv.tv_sec;
        s += (1e-6 * tv.tv_usec);
        return (s - start_);
}

// XXX not used?
// static void nullTimer(ClientData)
// {
// }

/*
 * dispatch a single simulator event by setting the system
 * virtul clock to the event's timestamp and calling its handler.
 * Note that the event may have side effects of placing other items
 * in the scheduling queue
 */
// these are global to make debugging easier. should be instance var
// of RealTimeSched -dam
int time_skew_cnt = 0;
double time_skew_sum = 0.0;
double time_skew_max = 0.0;

/* statics (histogram) of time-lags.  511 bins for 0-2 seconds, 512-th
 * for longer than 2 seconds.  so 512 in total.
 */
#define MAX_LAG_BOUND 2
#define BIN_NUM 512
#define BIN_SIZE (2.0/511.0)

int time_lag_his[BIN_NUM];
int emulation_event_counts = 0;

#ifdef __FreeBSD__
quad_t hz_cnt_start;
quad_t hz_cnt_next;
struct pmc_tstamp ts;	/* used to obtain clock freq */
double current_time;
#endif

static void
write_to_file(char *filename, int hisbuf[], int size)
{
	assert(size <= BIN_NUM);
	FILE *hisf = fopen(filename, "w+");
	fprintf(hisf, "%% 0-2 seconds for first 511 bins, larger than 2 fall into 512-th bin. 513-th row: max, 514-th row: sum, 515-th count, 516-th: total event counts\n");
	for (int i = 0; i < size; i++) {
		fprintf(hisf, "%d\n", hisbuf[i]);
	}

	// break the programming rules. --ke
	fprintf(hisf, "%f\n", time_skew_max);
	fprintf(hisf, "%f\n", time_skew_sum);
	fprintf(hisf, "%d\n", time_skew_cnt);
	fprintf(hisf, "%d\n", emulation_event_counts);
	fclose(hisf);
}

void
RealTimeScheduler::dispatch(Event* p, double t)
{
#ifdef __FreeBSD__
	if(t < current_time) {
		double lag = (current_time - p->time_) +
			(hz_cnt_next - hz_cnt_start) / ts.pmct_rate;
		/*
		 * handle case where counter wraps
		 */
		if(lag < 0) {
			lag = current_time - p->time_;
		}
#else
	if (t < clock_) {
		double lag = clock_ - t;
#endif /* __FreeBSD__ */

		if (lag > 2) {
			time_lag_his[BIN_NUM - 1] ++;
		} else {
			int index = (int)(lag/BIN_SIZE);
#ifdef __FreeBSD__
			assert(index >= 0 && index < BIN_NUM);
#endif
			time_lag_his[index] ++;
		}
		time_skew_cnt++;
		time_skew_sum += lag;			
		if (lag > time_skew_max) time_skew_max = lag;
	}

	emulation_event_counts ++;
	// the time of event dispatches must monotoniclly increasing -dam
	assert(t >= time_floor_);
	time_floor_ = t;

	clock_ = t;
	p->uid_ = -p->uid_;	// being dispatched
	p->handler_->handle(p);	// dispatch
}

void RealTimeScheduler::run()
{ 
        Event *p;
        double now;
#ifdef __FreeBSD__
	int fd;

	fd = open(_PATH_PERFMON, O_RDWR, 0);
        if (fd < 0) {
                perror("open " _PATH_PERFMON);
                exit(1);
        }

        if (ioctl(fd, PMIOTSTAMP, &ts) < 0) {
		perror("ioctl (PMIOTSTAMP)");
		exit(1);
	}

	close(fd);

	/*
	 * Convert rate to Hz
	 */
	ts.pmct_rate *= (int) 1e6;

#endif /* __FreeBSD__ */


        /*XXX*/
        instance_ = this;

	bzero(time_lag_his, sizeof(time_lag_his));

        while (!halted_) {
                now = tod();
		if ((clock_ - now) > slop_) {
			fprintf(stderr, "RealTimeScheduler: warning: slop %f exceeded limit %f\n",
                                (clock_ - now), slop_); 
                }

                //
                // first handle any "old events"
                //

#ifdef __FreeBSD__
		current_time = now;
		hz_cnt_start = hz_cnt_next = rdtsc();
#endif
                while ((p = deque()) != NULL && (p->time_ <= now)) {
                        dispatch(p);
#ifdef __FreeBSD__
			hz_cnt_next = rdtsc();	
#endif
                }

                //
                // now handle a "future event", if there is one
                //
                if (p != NULL) {
                        int rval = rwait(p->time_);
                        if (rval < 0) {
                                fprintf(stderr, "RTScheduler: wait problem\n");
                                abort();
                        } else if (rval == 0) {
                                //
                                // proper time to dispatch sim event... do so
                                //

				/* the following line of code is broken.

				dispatch(p, clock_);

				   consider that that q has events at
				   1 2 3 usecs.  rwait returns when clock is
				   at 3 usec (it overruns the deadline by a
				   bit), the first event is dispatched at
				   clock = 3, then the second one is dispatched
				   at clock = 2
				   Hence, time moves backwards which violates
				   the contract for time -dam 4/5/99
				*/
				dispatch(p);
                        } else {
                                //
                                // there was a simulator event which fired, and
                                // may have added something to the queue, which
                                // could cause our event p to not be the next,
                                // so put p back into the event queue and cont
                                //
                                insert(p);
                        }
                        continue;
                }

                //
                // no sim events to handle at all, check with tcl
                //
                sync();
                Tcl_DoOneEvent(TCL_DONT_WAIT);
        }

	fprintf(stderr,"skew cnt %d  sum %f  ave %f  max %f\n", time_skew_cnt,
		time_skew_sum, time_skew_sum / time_skew_cnt, time_skew_max);

	fprintf(stderr, "total events %d\n", emulation_event_counts);

	write_to_file("timelag_histogram", time_lag_his, BIN_NUM);
        return; // we reach here only if halted
}

/*
 * wait until the specified amount has elapsed, or a tcl event has happened,
 * whichever comes first.  Return 1 if a tcl event happened, 0 if the
 * deadline has been reached, or -1 on error (shouldn't happen).
 */

int
RealTimeScheduler::rwait(double deadline)
{
        while (1) {
                sync();
                if (Tcl_DoOneEvent(TCL_DONT_WAIT) == 1)
                        return (1);
                if (deadline <= tod())
                        return 0;
        }
        return -1;
}
