#include "capacityagent.h"
#define START_JITTER 0.5

/* 
   1. to run this agent:
   ns run.tcl 
   -seed 1.0 
   -rp testmac.tcl 
   -sc scen/rand-100-1000x1000-1 
   -x 1000 
   -y 1000 
   -pktsize 100 
   -stop 300 
   -tr /tmp/out.tr

   to analyze the trace file:
   use totals.pl in the current directory

   2. set the following variables according to what scenarios you are simulating:
   test_scenario (RANDOM -- random graph, will use shortest path routing
		  CHAIN,GRID,GRID_2, fixed type of graph, static routing)
   cbr (false -- source nodes send packets asap, i.e.greedy mode
        true -- use CBR connection file for traffic)
   3. the directory scen contains the scenario files, use
    generate-random-scen.pl to generate random graph
*/
#define max(x,y) (x>y?x:y)

static class CapacityAgentClass : public TclClass
{
    public:
    CapacityAgentClass():TclClass("Agent/CapacityAgent")
    {
    }
    TclObject* create(int, const char *const *)
    {
	return (new CapacityAgent());
    }
}class_CapacityAgent;

void
RandSendTimer::expire(Event *) { 
    assert(a_);

    a_->sendPkt();
  
    double next_time = Random::uniform(1) * 2 * t_;
    resched(next_time);
}    

CapacityAgent::CapacityAgent() : Agent(PT_CBR)
{
    test_scenario = RANDOM;
    i_am_transmitter = false;
    total_nodes = 0;
    packet_size = 0;
    packet_rate = -1.0;
    ifq = NULL;
    cbr = false;
    snd_timer = 0;

    total_pkt_sent = 0;
    total_pkt_length = 0;

    //get the square root of A -- the area of the entire network, or the longest distance in the entire network
    bind("off_ll_", &off_ll_);
    bind("off_mac_", &off_mac_);
    bind("off_ip_", &off_ip_);
}

CapacityAgent::~CapacityAgent()
{
    fprintf(stderr,"Don't do this! \n");
    exit(-1);
}

void
CapacityAgent::Terminate()
{
  //    char buffer[100];
    /*
    int len = 0;

       for (int i=0;i<mac->num_interfere_nodes;i++) {
	len += sprintf(buffer+len,"%d ",mac->interfere_nodes[i]);
    }*/
    /*
    mac_log_entry *mylog;
    mylog = &(God::instance()->maclog[myaddr_]);
    trace("Stimebreakdown _%d_ tx %.2f, rx %.2f, backoff %.2f(%.2f), defer %.2f, idle %.2f(%.2f)",myaddr_,mylog->tx,mylog->rx,mylog->backoff,mylog->useless_backoff,mylog->defering,mylog->idle,mylog->true_idle);

    trace("Sstopped _%d_ total bftime %.2f, total_rx %d, total wasted time %.2f total true idle time %.2f",myaddr_,mac->bftime(), mac->total_retransmit,mac->total_wasted_time,mac->total_true_idle_time);
    */
    ifq->Terminate();
    if (test_scenario == RANDOM) {
	double avg_path_len = total_pkt_length/total_pkt_sent;
	trace("Spathavg _%d_ %.2f",myaddr_,avg_path_len);
    }
}

void
CapacityAgent::Start()
{

    double maxx = God::instance()->T->upperX();
    double maxy = God::instance()->T->upperY();

    mynode->getLoc(&x,&y,&z);

    sqrt_A = x*x+y*y;

    double new_dist = (x-maxx)*(x-maxx) + (y-maxy)*(y-maxy);
    sqrt_A = max(sqrt_A,new_dist);

    new_dist = x*x + (maxy-y)*(maxy-y);
    sqrt_A = max(sqrt_A,new_dist);

    new_dist = (maxx-x)*(maxx-x) + y*y;
    sqrt_A = max(sqrt_A,new_dist);

    sqrt_A = sqrt(sqrt_A);

    //start maclog
    if (myaddr_ == 1) {
	//God::instance()->start_maclog();
    }

    if (cbr) return;

    int myaddr_mod;
    int myaddr_div;

    switch (test_scenario) {
	case ONE_CELL:
	    i_am_transmitter = true;
	    break;
	case CHAIN:
	    if (myaddr_ == 1) i_am_transmitter = true;
	    break;
	case GRID:
	    if ((myaddr_ % chain_length ) == 1) {
		i_am_transmitter = true;
	    }
	    break;
	case GRID_2:
	    myaddr_mod = myaddr_ % chain_length;
	    myaddr_div = (myaddr_ - myaddr_mod)/chain_length;

	    if ((myaddr_mod == 1) && (myaddr_div>0) && (myaddr_div < (chain_length-1))) {
		i_am_transmitter = true;
	    }
	    //the number of top-to-bottom connections are 2 less than left-to-right ones to avoid having a node as sender/receiver for two streams
	    if ((myaddr_ < chain_length) && (myaddr_ > 1)) {  
		i_am_transmitter = true;
	    }
	    break;
	case RANDOM:
	    if (myaddr_ <= (int) (fraction_sending * (double) total_nodes)) {
	      i_am_transmitter = true;
	    }
	    break;
	default:
	    printf("impossible error!\n");
	    exit(1);
    }

    if (i_am_transmitter) {
	if ((!cbr) && (packet_rate < 0.001) && (test_scenario<RANDOM)) {
	    //load the queue with 30 packets
	    for (int i=0;i<30;i++) {
		sendOnePacket(Random::uniform(START_JITTER));
	    }
	}else{
	    sendOnePacket(Random::uniform(START_JITTER));
	    snd_timer = new RandSendTimer(this, 1 / packet_rate); 
	    snd_timer->sched(Random::uniform(1) * 2 / packet_rate);
	}
    }

}

void
CapacityAgent::sendPkt()
{

    if (test_scenario == RANDOM) {
        sendOnePacket(0);
	return;
    }
    double u; //a uniform random variable
    double dist; //dist of the destination node that I want to send pkt to
    switch (test_scenario) {
	case POWER_NEG2_ALPHA:
	    u = Random::uniform();
	    dist = sqrt_A * EPISILONG;
	    dist = dist / (sqrt_A - sqrt_A*u + u*EPISILONG);
	    //pick some node around that location
	    sendOnePacketToDst(0,God::instance()->getNodeDstAway(x,y,dist));
	    break;
	case POWER_NEG3_ALPHA:
	    break;
	case EXP_1_ALPHA:
	    break;
	default:
	    printf("fatal error!\n");
	    exit(2);
    }
}

void
CapacityAgent::sendOnePacketToDst(double delay, nsaddr_t dst)
{
    //start to send a single packet first
    Packet *pkt = allocpkt();
    hdr_cmn *cmnh = (hdr_cmn*)pkt->access(off_cmn_);
    hdr_ip *iph= (hdr_ip*)pkt->access(off_ip_);
    //double time = Scheduler::instance().clock();

    cmnh->xmit_failure_ = Capacity_XmitFailureCallback;
    cmnh->xmit_failure_data_ = (void *)this;

    iph->src() = myaddr_;
    iph->dst() = dst;
    cmnh->next_hop() = (God::instance())->shortestPathNextHop(myaddr_,iph->dst());

    if (!cmnh->next_hop()) {
	printf("_%d_ error: no path to %d\n",myaddr_,iph->dst());
	free(pkt);
	return;
    }

    total_pkt_sent++;
    total_pkt_length += (God::instance())->shortestPathLength(myaddr_,iph->dst());

    mac->hdr_dst((char*) HDR_MAC(pkt), cmnh->next_hop());
    mac->hdr_src((char*) HDR_MAC(pkt), myaddr_);

    cmnh->size() = packet_size;
    Scheduler::instance().schedule(ll,pkt,delay);
}


void
CapacityAgent::sendOnePacket(double delay)
{
    if (test_scenario >= RANDOM && cbr)
      assert(0);

    //start to send a single packet first
    Packet *pkt = allocpkt();
    hdr_cmn *cmnh = (hdr_cmn*)pkt->access(off_cmn_);
    hdr_ip *iph= (hdr_ip*)pkt->access(off_ip_);
    //double time = Scheduler::instance().clock();

    cmnh->xmit_success_ = Capacity_XmitSuccessCallback;
    cmnh->xmit_success_data_ = (void *)this;

    cmnh->xmit_failure_ = Capacity_XmitFailureCallback;
    cmnh->xmit_failure_data_ = (void *)this;

    iph->src() = myaddr_;

    //now randomly pick a destination
    switch (test_scenario) {
	case ONE_CELL:
	    cmnh->next_hop() = (int)(Random::uniform() * total_nodes);
	    if (!cmnh->next_hop()) cmnh->next_hop() = 1; //not sure if uniform() returns 0
	    iph->dst() = cmnh->next_hop();
	    break;
	case CHAIN:
	    assert(myaddr_ ==1);
	    assert(chain_length);
	    iph->dst() = chain_length;
	    cmnh->next_hop() = myaddr_ + 1;
	    break;
	case GRID:
	    assert(chain_length);
	    iph->dst() = myaddr_ + chain_length - 1;
	    cmnh->next_hop() = myaddr_ +1;
	    break;
	case GRID_2:
	    if ((myaddr_ % chain_length) == 1) {
		iph->dst() = myaddr_ + chain_length -1;
		cmnh->next_hop() = myaddr_ + 1;
	    }else if ((myaddr_ > 1) && (myaddr_ < chain_length)) {
		iph->dst() = myaddr_ + chain_length * (chain_length -1);
	    	cmnh->next_hop() = myaddr_ + chain_length;
	    }else{
		printf("no route! that should not happen!\n");
		exit(1);
	    }
	    break;
	case RANDOM:
	    do{
		iph->dst() = (int)(Random::uniform() * total_nodes);
	    }while ((iph->dst()==myaddr_) || (iph->dst() == 0));

	    cmnh->next_hop() = (God::instance())->shortestPathNextHop(myaddr_,iph->dst());

	    
	    if (!cmnh->next_hop()) {
		printf("_%d_ error: no path to %d\n",myaddr_,iph->dst());
		free(pkt);
		return;
	    }

	    total_pkt_sent++;
	    total_pkt_length += (God::instance())->shortestPathLength(myaddr_,iph->dst());
    }

    mac->hdr_dst((char*) HDR_MAC(pkt), cmnh->next_hop());
    mac->hdr_src((char*) HDR_MAC(pkt), myaddr_);

    cmnh->size() = packet_size;

    Scheduler::instance().schedule(ll,pkt,delay);
}

void
CapacityAgent::trace(char* fmt, ...)
{
    va_list ap;
        
    if (!logtarget) return;
    va_start(ap, fmt);
    vsprintf(logtarget->buffer(), fmt, ap);
    logtarget->dump();
    va_end(ap);
}

int
CapacityAgent::command(int argc, const char*const* argv)
{

  TclObject *obj;  

  if (argc == 2) 
    {
      if (strcasecmp(argv[1], "reset") == 0)
	{
	  Terminate();
	  return Agent::command(argc, argv);
	}

      if (strcasecmp(argv[1],"start") == 0) {
	  Start();
	  return TCL_OK;
      }

    }
  else if(argc == 3) 
    {
      if (strcasecmp(argv[1], "ip-addr") == 0) 
	{
	 myaddr_ = atoi(argv[2]);
	 return TCL_OK;
	} 
       

      if (strcasecmp(argv[1],"total-nodes") == 0)
      {
	  total_nodes = atoi(argv[2]);
	  return TCL_OK;
      }

      if (strcasecmp(argv[1],"packet-size") == 0)
      {
	  packet_size = atoi(argv[2]);
	  return TCL_OK;
      }

      if (strcasecmp(argv[1],"packet-rate") == 0)
      {
	  packet_rate = atof(argv[2]);
	  return TCL_OK;
      }
      
      if (strcasecmp(argv[1],"offered-load") == 0)
      {
	  if (packet_rate > 0.001) abort();
	  assert(packet_size!=0);
	  //offered load is given in Mbps
	  double load = atof(argv[2]);
	  packet_rate = (load*1000000)/(packet_size * 8);
	  return TCL_OK;
      }

      if (strcasecmp(argv[1],"fraction-sending") == 0)
      {
	  fraction_sending = atof(argv[2]);
	  assert(0 <= fraction_sending && fraction_sending <= 1);
	  return TCL_OK;
      }
      if (strcasecmp(argv[1],"chain-length") == 0)
      {
	  chain_length = atoi(argv[2]);
	  return TCL_OK;
      }

      
      if( (obj = TclObject::lookup(argv[2])) == 0) 
	{
	  fprintf(stderr, "CapacityAgent: %s lookup of %s failed\n", argv[1],
		  argv[2]);
	  return TCL_ERROR;
	}
	

      if (strcasecmp(argv[1], "log-target") == 0) 
	{
	  logtarget = (Trace*) obj;
	  return TCL_OK;
	}

      if (strcasecmp(argv[1], "add-ifq") == 0) 
	{
	  ifq = (PriQueue*) obj;
	  return TCL_OK;
	}

      if (strcasecmp(argv[1],"mobile-node") == 0) {
	  mynode = ((MobileNode *)obj);
	  return TCL_OK;
      }

    }
  else if (argc == 4)
    {
      if (strcasecmp(argv[1], "add-ll") == 0) 
	{
	  if( (obj = TclObject::lookup(argv[2])) == 0) {
	    fprintf(stderr, "CapacityAgent: %s lookup of %s failed\n", argv[1],
		    argv[2]);
	    return TCL_ERROR;
	  }
	  ll= (NsObject *) obj;

	  if( (obj = TclObject::lookup(argv[3])) == 0) {
	    fprintf(stderr, "CapacityAgent: %s lookup of %s failed\n", argv[1],
		    argv[2]);
	    return TCL_ERROR;
	  }
	  mac = (Mac802_11 *) obj;
	  return TCL_OK;
	}
    }
  return Agent::command(argc, argv);
}

void
CapacityAgent::recv(Packet *pkt, Handler *)
{

    hdr_ip *iph = (hdr_ip *) pkt->access (off_ip_);
    hdr_cmn *cmnh = (hdr_cmn *) pkt->access (off_cmn_);
    Scheduler & s = Scheduler::instance();

    if (iph->src() == myaddr_) {
	cmnh->size() = packet_size;
    }else if (!cbr) {
	assert(cmnh->size() == packet_size);
    }

    if (test_scenario == ONE_CELL) {
	assert(myaddr_ == iph->dst());
	assert(cmnh->size() == packet_size);
	target_->recv(pkt, (Handler*)0);
    } else {

	if (myaddr_ != iph->dst()) {

	    if (test_scenario == RANDOM) {
		cmnh->next_hop() = (God::instance())->shortestPathNextHop(myaddr_,iph->dst());
		if (cmnh->next_hop()==0) {
		    printf("no route! this should not happen!\n");
		    exit(2);
		}

	    }else {//CHAIN,GRID, or GRID_2

		int myaddr_mod = myaddr_ % chain_length;
		int dst_mod = iph->dst() % chain_length;
		int myaddr_div = (myaddr_ - myaddr_mod)/chain_length;
		int dst_div = (iph->dst() - dst_mod)/chain_length - 1;

		if ((myaddr_ % chain_length) == (iph->dst() % chain_length)) {
		    cmnh->next_hop() = myaddr_ + chain_length;
		}else if ((iph->dst() % chain_length == 0) && (myaddr_div == dst_div)) {
		    cmnh->next_hop() = myaddr_ + 1;
		}else {
		    printf("no route! this should not happen!!!!\n");
		    exit(2);
		} 
	    }
	    cmnh->xmit_success_ = NULL;
	    cmnh->xmit_success_data_ = NULL;

	    cmnh->xmit_failure_ = Capacity_XmitFailureCallback;
	    cmnh->xmit_failure_data_ = (void *)this;

	    mac->hdr_dst((char*) HDR_MAC(pkt), cmnh->next_hop());
	    mac->hdr_src((char*) HDR_MAC(pkt), myaddr_);
	    s.schedule(ll,pkt,0);
	}else {
	    target_->recv(pkt, (Handler*)0);
	    trace("Scapacityagent: _%d_ got cbr pkt from _%d_",myaddr_,iph->src());
	}
    } 

}

void
CapacityAgent::xmitSucceeded(Packet *pkt)
{
    //as soon as packet has been successfully sent, schedule another one
    if ((packet_rate<0.001) && (!cbr) && i_am_transmitter && (test_scenario<RANDOM)) 
      sendOnePacket(0.0);
}

void
CapacityAgent::xmitFailed(Packet *pkt)
{
    hdr_cmn *cmnh = (hdr_cmn *) pkt->access (off_cmn_);
    hdr_ip *iph= (hdr_ip*)pkt->access(off_ip_);

    assert(God::instance()->isNeighbor(myaddr_,cmnh->next_hop()));

    drop(pkt,DROP_RTR_NO_ROUTE);

    if ((!cbr) && (packet_rate <0.001) && (myaddr_ == iph->src()) && (ifq->prq_length() < 10) && (test_scenario<RANDOM)) {
	sendOnePacket(0.0);
    }
}
 
void
Capacity_XmitSuccessCallback(Packet *pkt, void *data)
{
    CapacityAgent *a = (CapacityAgent *)data; // cast of trust
    a->xmitSucceeded(pkt);
}

void
Capacity_XmitFailureCallback(Packet *pkt, void *data)
{
    CapacityAgent *a = (CapacityAgent *)data; // cast of trust
    a->xmitFailed(pkt);
}

