/*
 * Copyright (c) 1994-1997 Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *	This product includes software developed by the Computer Systems
 *	Engineering Group at Lawrence Berkeley Laboratory.
 * 4. Neither the name of the University nor of the Laboratory may be used
 *    to endorse or promote products derived from this software without
 *    specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#ifndef lint
static const char rcsid[] =
    "@(#) $Header: /home/cr1/benjie/cvsroot/ns/ns-span/cmu/ntr-workload.cc,v 1.1.1.1 2001/11/15 19:43:16 benjie Exp $ (LBL)";
#endif

#include <stdarg.h>

#include "rtp.h"
#include "tclcl.h"
#include "packet.h"
#include "random.h"

#include <ip.h>

#include "ntr-workload.h"

static class NTRWKLDClass : public TclClass {
public:
	NTRWKLDClass() : TclClass("Agent/NTRWKLD") {}
	TclObject* create(int, const char*const*) {
		return (new NTRWKLD_Agent());
	}
} class_NTRWKLD;

NTRWKLD_Agent::NTRWKLD_Agent() : Agent(PT_NTRWKLD), running_(0), seqno_(-1), timer_(this)
{
	bind_time("interval_", &interval_);
	bind("maxTests_", &maxTests_);
	bind("bigPacketSize_", &bigPacketSize_);
	bind("smallPacketSize_", &smallPacketSize_);
	bind("off_ip_", &off_ip_);

	logtarget_ = 0;
}

void 
NTRWKLD_Agent::start()
{
	running_ = 1;
	testNumber_ = 0;
	sendRequest(smallPacketSize_);
	timer_.resched(interval_);
}

void 
NTRWKLD_Agent::stop()
{
	timer_.cancel();
	running_ = 0;
}


void
NTRWKLD_Agent::trace(char* fmt, ...)
{
  va_list ap;
  
  if (!logtarget_) return;

  va_start(ap, fmt);
  vsprintf(logtarget_->buffer(), fmt, ap);
  logtarget_->dump();
  va_end(ap);
}

void
NTRWKLD_Agent::recv(Packet* p, Handler*)
{
	hdr_cmn *cmh =  (hdr_cmn*)p->access(off_cmn_);
	struct ping_data *pd = (struct ping_data *) p->accessdata();
	assert(p->datalen() == sizeof(struct ping_data));


	if (ECHO_REQUEST == pd->type) {
		reflect(p);
	} else if (ECHO_REPLY == pd->type) {
		if (!running_) {
			fprintf(stderr,"%s: dropping REPLY rcvd when not running\n",
				__FUNCTION__);
			drop(p);
			return;
		}

		if (cmh->size() == bigPacketSize_) {
			logReply(p);
			drop(p); // no drop reason required
		} else if (cmh->size() == smallPacketSize_) {
			logReply(p);

			if ((Scheduler::instance().clock() - pd->timestamp )
			    <= 10 * interval_) {			  
				// send the back to back large pings once small ping reply recv'd
				// so long as small ping isn't too old
				sendRequest(bigPacketSize_); 
				sendRequest(bigPacketSize_);
			}

			drop(p); // no drop reason required
		} else {
			fprintf(stderr,"DFU: unexpected reply pkt size %d\n",cmh->size());
			abort();
		}			
	} else {
		fprintf(stderr, "DFU: unknown ntr workload packet type %d\n", pd->type);
		abort();
	}
}

void
NTRWKLD_Agent::logReply(Packet *p)
{
	hdr_cmn *cmh =  (hdr_cmn*)p->access(off_cmn_);
	hdr_ip *iph = (hdr_ip*)p->access(off_ip_);
	struct ping_data *pd = (struct ping_data *) p->accessdata();
	assert(pd->type == ECHO_REPLY);
	
	trace("P %.9f _%d_ REPLY %d  %d %d  %d %.9f",
	      Scheduler::instance().clock(), iph->dst(), cmh->size(),
	       iph->dst(), iph->src(), pd->seqno,
	      Scheduler::instance().clock() - pd->timestamp);	
}

void
NTRWKLD_Agent::logRequest(Packet *p)
{
	hdr_cmn *cmh =  (hdr_cmn*)p->access(off_cmn_);
	hdr_ip *iph = (hdr_ip*)p->access(off_ip_);
	struct ping_data *pd = (struct ping_data *) p->accessdata();
	assert(pd->type == ECHO_REQUEST);
	
	trace("P %.9f _%d_ REQUEST %d  %d %d  %d",
	      Scheduler::instance().clock(), iph->src(), cmh->size(),
	       iph->src(), iph->dst(), pd->seqno);
}

void
NTRWKLD_Agent::reflect(Packet *p)
{
	hdr_ip* iph = (hdr_ip*)p->access(off_ip_);

	// flip addrs and ports
	nsaddr_t taddr;
	taddr = iph->src();
	iph->src() = iph->dst();
	iph->dst() = taddr;
	
	u_int16_t tport;
	tport = iph->sport();
	iph->sport() = iph->dport();
	iph->dport() = tport;

	// set the type
	struct ping_data *pd = (struct ping_data *) p->accessdata();
	assert(ECHO_REQUEST == pd->type);
	pd->type = ECHO_REPLY;

	// assign the packet a new uid
	hdr_cmn* ch = (hdr_cmn*)p->access(off_cmn_);
	ch->uid() = uidcnt_++;

	target_->recv(p); 	// ship it
}


void 
NTRWKLD_Agent::timeout(int)
{
	if (running_) {

		if (maxTests_ > 0 && testNumber_ >= maxTests_) {
			running_ = 0;
			return;
		}

		testNumber_++;
		sendRequest(smallPacketSize_);
		timer_.resched(interval_);
	}
}

void 
NTRWKLD_Agent::sendRequest(int size)
{

	Packet* p = allocpkt(sizeof(struct ping_data));
	struct ping_data *pd = (struct ping_data *) p->accessdata();
	hdr_cmn *cmh =  (hdr_cmn*)p->access(off_cmn_);

	cmh->size() = size;
	pd->type = ECHO_REQUEST;
	pd->seqno = ++seqno_;
	pd->timestamp = Scheduler::instance().clock();
	
	logRequest(p);
	target_->recv(p);
}

/*
 * $ntrwkld start
 * $ntrwkld stop
 */
int 
NTRWKLD_Agent::command(int argc, const char*const* argv)
{
        TclObject *obj;

	if (argc == 2) {

		if (strcmp(argv[1], "start") == 0) {
			start();
			return (TCL_OK);
		}
		if (strcmp(argv[1], "stop") == 0) {
			stop();
			return (TCL_OK);
		}

	}  else if(argc == 3) {
		
		if ((obj = TclObject::lookup(argv[2])) == 0)  {
			fprintf(stderr, "%s: %s lookup of %s failed\n", __FUNCTION__, 
				argv[1], argv[2]);
			return TCL_ERROR;
		}

		if (strcasecmp(argv[1], "log-target") == 0)  {
			logtarget_ = (Trace*) obj;
			return (TCL_OK);
		}
	}
	return (Agent::command(argc, argv));
}

void 
NTRWKLD_Timer::expire(Event *) {
	a_->timeout(0);
}
