/* -*- c++ -*-
   net-if.h
   $Id: net-if.h,v 1.1.1.1 2001/11/15 19:43:16 benjie Exp $

   the super class of all network interfaces

   ======================================================================
   Network Interface

	The hardware that actually transmitts the packet onto the channel
	with a certain power, modulation scheme, varing amounts of FEC,
	etc.

	No packets are "buffered" at this layer as the decision to send
	has already been made and the packet is on its way to the "Channel".

	This interface "stamps" each packet with its properties so that they
	will be "known" at the receiver.  This allows the receiver to compute
	the receive power level, etc.

*/

#ifndef __net_if_h__
#define __net_if_h__

#include <assert.h>
#include <cmu/list.h>
#include <cmu/energy-model.h>

class NetIf;
LIST_HEAD(if_head, NetIf);

#include <cmu/channel.h>

class NetIf : public NsObject {

public:
  NetIf();
  
  void recv(Packet *p, Handler *h);
  
  virtual void xmitPacket(Packet *p) = 0;
  // send packet out to channel

  virtual int recvPacket(Packet *p, double *RxPr) = 0;
  // determine if pkt can be recvd.
  // rtns 1 if yes, 0 otherwise
  // *RxPr set to power of incoming packet

  inline double  txtime(Packet *p) const;
  inline double  txtime(int bytes) const {return bytes * 8.0 / Rb;}
  virtual double	bittime() const { return 1/Rb; }

  virtual void PowerSave(bool) = 0;
  
  MobileNode*	node(void) const { return node_; }

  virtual void    dump(void) const;
  
  // each network interface is on two lists
  NetIf*		nextnode(void) const { return node_link.le_next; }
  NetIf*		nextchnl(void) const { return chnl_link.le_next; }
  
  inline void insertnode(MobileNode* node, struct if_head* head) {
    LIST_INSERT_HEAD(head, this, node_link);
    node_ = node;
  }
  inline void insertchnl(Channel *chnl, struct if_head *head) {
    LIST_INSERT_HEAD(head, this, chnl_link);
    channel_ = chnl;
  }
  
protected:
  void		drop(Packet *p);
  int		command(int argc, const char*const* argv);
  
  int index;
  MobileNode *node_;		// Node using this interface
  
  /* ============================================================
     Physical Layer State
     ============================================================ */
  Propagation *propagation_;	// Propagation Model
  Modulation *modulation;	// Modulation Scheme

  double Rb;			// bit rate
  
  NsObject        *recvtarget_; // usually mac layer
  Channel         *channel_;    // the channel for output
  
private:
  inline int	initialized() {
    return (node_ && recvtarget_ && channel_ && propagation_);
  }

  /*
   * A list of all "network interfaces" on a given channel.
   * Note: a node may have multiple interfaces, each of which
   * is on a different channel.
   */
  LIST_ENTRY(NetIf) chnl_link;
  
  /*
   * A list of all "network interfaces" for a given mobile node.
   * Each interface is assoicated with exactly one mobile node
   * and one channel.
   */
  LIST_ENTRY(NetIf) node_link;
};

inline double
NetIf::txtime(Packet *p) const
{
	hdr_cmn *hdr = HDR_CMN(p);
	return hdr->size() * 8.0 / Rb;
}

#endif /* __net_if_h__ */


