#include <delay.h>
#include <connector.h>
#include <packet.h>
#include <random.h>
 
// #define DEBUG
#include <debug.h>
#include <arp.h>
#include <ll.h>
#include <mac.h>
#include <mac-timers.h>
#include <mac-802_11.h> 

/*
 * Force timers to expire on slottime boundries.
 */
// #define USE_SLOT_TIME

#define ROUND_TIME()	\
	{								\
		assert(slottime);					\
		double rmd = remainder(s.clock() + rtime, slottime);	\
		if(rmd > 0.0)						\
			rtime += (slottime - rmd);			\
			assert(rtime > 0.0);				\
		else							\
			rtime += (-rmd);				\
	}


/* ======================================================================
   Timers
   ====================================================================== */

void
MacTimer::start(double time, const char *description=0)
{
	Scheduler &s = Scheduler::instance();
	
	if (busy_)
	    assert(0);

	busy_ = 1;
	paused_ = 0;
	stime = s.clock();
	rtime = time;
	if (rtime < 0.0)
	    assert(0);

	s.schedule(this, &intr, rtime, description);
}

void
MacTimer::stop(void)
{
	Scheduler &s = Scheduler::instance();

	if (!busy_)
	    assert(0);

	if(paused_ == 0)
		s.cancel(&intr);

	busy_ = 0;
	paused_ = 0;
	stime = 0.0;
	rtime = 0.0;
}

/* ======================================================================
   Defer Timer
   ====================================================================== */
void
DeferTimer::start(double time, const char *description = 0)
{
	Scheduler &s = Scheduler::instance();

	assert(busy_ == 0);

	busy_ = 1;
	paused_ = 0;
	stime = s.clock();
	rtime = time;
#ifdef USE_SLOT_TIME
	ROUND_TIME();
#endif
	assert(rtime >= 0.0);

	s.schedule(this, &intr, rtime, description);
}


void    
DeferTimer::handle(Event *)
{       
	busy_ = 0;
	paused_ = 0;
	stime = 0.0;
	rtime = 0.0;

	mac->deferHandler();
}

void
DeferTimer::pause()
{
	Scheduler &s = Scheduler::instance();
	assert(busy_ && ! paused_);
	paused_ = 1;
	s.cancel(&intr);
}


/* ======================================================================
   NAV Timer
   ====================================================================== */
void    
NavTimer::handle(Event *)
{       
	busy_ = 0;
	paused_ = 0;
	stime = 0.0;
	rtime = 0.0;

	mac->navHandler();
}


/* ======================================================================
   Receive Timer
   ====================================================================== */
void    
RxTimer::handle(Event *)
{       
	busy_ = 0;
	paused_ = 0;
	stime = 0.0;
	rtime = 0.0;

	mac->recvHandler();
}


/* ======================================================================
   Send Timer
   ====================================================================== */
void    
TxTimer::handle(Event *)
{       
	busy_ = 0;
	paused_ = 0;
	stime = 0.0;
	rtime = 0.0;

	mac->sendHandler();
}


/* ======================================================================
   Interface Timer
   ====================================================================== */
void
IFTimer::handle(Event *)
{
	busy_ = 0;
	paused_ = 0;
	stime = 0.0;
	rtime = 0.0;

	mac->txHandler();
}


/* ======================================================================
   Backoff Timer
   ====================================================================== */
void
BackoffInterruptTimer::handle(Event *e)
{
    mac->handleBackoffInterrupt();
    Scheduler::instance().schedule(this,e,slottime);
}

void
BackoffTimer::handle(Event *)
{
	busy_ = 0;
	paused_ = 0;
	stime = 0.0;
	rtime = 0.0;
	difs_wait = 0.0;

	Scheduler::instance().cancel(&interrupt_event);

	mac->backoffHandler();
}

void BackoffTimer::stop(void) {
    Scheduler::instance().cancel(&interrupt_event);
    MacTimer::stop();    
}


void
BackoffTimer::start(int cw, double correction,
		    int idle, const char *description)
{
	Scheduler &s = Scheduler::instance();

	assert(busy_ == 0);

	busy_ = 1;
	paused_ = 0;
	stime = s.clock();
	rtime = correction + (Random::random() % cw) * mac->phymib->SlotTime;

	//jinyang
	total_bftime += rtime;

#ifdef USE_SLOT_TIME
	ROUND_TIME();
#endif
	difs_wait = 0.0;

	if(idle == 0)
		paused_ = 1;
	else {
		assert(rtime >= 0.0);
		s.schedule(this, &intr, rtime, description);
		s.schedule(interrupt_timer, &interrupt_event, 0.0);
	}
}


void
BackoffTimer::pause()
{
	Scheduler &s = Scheduler::instance();
	int slots = (int) ((s.clock() - (stime + difs_wait)) / mac->phymib->SlotTime);
	if(slots < 0)
		slots = 0;
	assert(busy_ && ! paused_);

	paused_ = 1;
	double tmp_time = rtime; //jinyang -- added for debugging
	rtime -= (slots * mac->phymib->SlotTime);
	/* -begin modified by Jinyang */
	assert(rtime >= -0.001);
	if (rtime < 0.0) {
	    rtime = 0.0;
	}
	/* -end */

	difs_wait = 0.0;

	s.cancel(&intr);
	s.cancel(&interrupt_event);
}


void
BackoffTimer::resume(double difs)
{
	Scheduler &s = Scheduler::instance();

	assert(busy_ && paused_);

	paused_ = 0;
	stime = s.clock();

	/*
	 * The media should be idle for DIFS time before we start
	 * decrementing the counter, so I add difs time in here.
	 */
	difs_wait = difs;

#ifdef USE_SLOT_TIME
	ROUND_TIME();
#endif
	assert(rtime + difs_wait >= 0.0);
	s.schedule(this, &intr, rtime + difs_wait);
	s.schedule(interrupt_timer, &interrupt_event,slottime);
}


/* ======================================================================
   TBTT Timer
   ====================================================================== */
void TBTTTimer::handle(Event *)
{       
	busy_ = 0;
	paused_ = 0;
	stime = 0.0;
	rtime = 0.0;

	mac->tbttHandler();
}

/* ======================================================================
   Beacon Timer
   ====================================================================== */
void BeaconTimer::handle(Event *)
{       
	busy_ = 0;
	paused_ = 0;
	stime = 0.0;
	rtime = 0.0;

	mac->beaconHandler();
}

/* ======================================================================
   ATIM Start Timer
   ====================================================================== */
void AdvEndTimer::handle(Event *)
{       
	busy_ = 0;
	paused_ = 0;
	stime = 0.0;
	rtime = 0.0;

	mac->advEndHandler();
}

/* ======================================================================
   ATIM End Timer
   ====================================================================== */
void ATIMEndTimer::handle(Event *)
{       
	busy_ = 0;
	paused_ = 0;
	stime = 0.0;
	rtime = 0.0;

	mac->atimEndHandler();
}
