/* mac-null.cc
   $Id: mac-null.cc,v 1.1.1.1 2001/11/15 19:43:16 benjie Exp $

   implement a mac layer that IGNORES collisions.  This layer allows the 
   simultaneous receipt of multiple packets.   

   If the variable no_delay_send_ is set to true
     The layer delays packet 1 tx time on receipt, but sends back to back 
     packets immeadiately one after the other.  Together, the effect is as
     if the link had infinite BW, and the txtime is reassigned to propagation
     delay.
   else, the txtime is used to pace the sending of packets
   
   */

#include <delay.h>
#include <connector.h>
#include <packet.h>
#include <random.h>

// #define DEBUG

#include <cmu/debug.h>
#include <cmu/arp.h>
#include <cmu/ll.h> 
#include <cmu/mac.h>
#include <cmu/mac-802_11.h>
#include <cmu/cmu-trace.h>
#include <cmu/mac-null.h>

#define RANGE 250.0

/* ======================================================================
   Mac Class Functions
   ====================================================================== */
MacNULL::MacNULL(): send_timer_(this), recv_timer_(this), 
	upcall_(0), no_delay_send_(1) //no_delay_send_ set to true -jinyang
{
	sta_seqno = 1;
	cache_node_count = 0;

	bind("off_arp_", &off_arp_);
	//bind("no_delay_send_", &no_delay_send_);

	fprintf(stderr,"WARNING: null MAC layer selected.  ignoring collisions\n");	
}


/* ======================================================================
   Packet Headers Routines
   ====================================================================== */
	
inline MacAddr
MacNULL::hdr_dst(void* hdr, MacAddr dst)
{
	struct hdr_macnull *mh = (struct hdr_macnull*) hdr;

	if(dst)
		STORE4BYTE(&dst,mh->dh_da);

	return ETHER_ADDR(mh->dh_da);
}

inline MacAddr
MacNULL::hdr_src(void* hdr, MacAddr src)
{
	struct hdr_macnull *mh = (struct hdr_macnull*) hdr;

	if(src)
		STORE4BYTE(&src,mh->dh_sa);

	return ETHER_ADDR(mh->dh_sa);
}

inline int 
MacNULL::hdr_type(void* hdr, u_int16_t type)
{
	struct hdr_macnull *mh = (struct hdr_macnull*) hdr;

	if(type)
	        SET_ETHER_TYPE(mh->dh_body, type);

	return GET_ETHER_TYPE(mh->dh_body);
}


int
MacNULL::command(int argc, const char*const* argv)
{
	if (argc == 3) {

		if (strcmp(argv[1], "log-target") == 0) {
			logtarget_ = (NsObject*) TclObject::lookup(argv[2]);
			if(logtarget_ == 0)
				return TCL_ERROR;
			return TCL_OK;
		}

		if(strcmp(argv[1], "nodes") == 0) {
			cache_node_count = atoi(argv[2]);
			return TCL_OK;
			
		}
	}
	return Mac::command(argc, argv);
}


/* ======================================================================
   Debugging Routines
   ====================================================================== */
void
MacNULL::trace_pkt(Packet *p) {
	struct hdr_cmn *ch = HDR_CMN(p);

	fprintf(stderr, "\t_%d_ %s %d\n",		
		index, packet_names[ch->ptype()], ch->size());
}

void
MacNULL::dump(char *fname)
{
	fprintf(stderr,
		"\n%s --- (INDEX: %d, time: %2.9f)\n",
		fname, index, Scheduler::instance().clock());
}



/* ======================================================================
   Incoming Packet Routines
   ====================================================================== */

void
MacNULL::recv(Packet *p, Handler *h)
{
	struct hdr_cmn *hdr = HDR_CMN(p);
	hdr_macnull* mh = (hdr_macnull*)p->access(off_mac_);

	/*
	 * Sanity Check
	 */
	assert(initialized());

	/*
	 *  Handle outgoing packets.
	 */
	if(h) {
		// give target a copy, so we can still use p
                sendtarget_->recv(p->copy(), h);

		if (no_delay_send_) {
			// tell IFQ to send down the next packet immeadiately
			// NOTE:  see comments at top of file
			upcall_ = h;
			sendComplete(p);
		} else {
			assert(upcall_ == 0);
			upcall_ = h;
			send_timer_.schedule(netif_->txtime(p),p);
		}
		return;
        }

	/*
	 *  Handle incoming packets.
	 *
	 *  We just received the 1st bit of a packet on the network
	 *  interface.
	 *
	 */

	if (hdr_dst(mh) !=  index &&
	    hdr_dst(mh) !=  MAC_BROADCAST) {
		/*
		 *  We don't want to log this event, so we just free
		 *  the packet instead of calling the drop routine.
		 */
		Packet::free(p);
		goto done;
	}

	if (hdr->error() ) {
		discard(p, DROP_MAC_PACKET_ERROR);
		goto done;
	}

	/*
	 * reception of this packet will finish in
	 * txtime seconds.  schedule completion then
	 */
	recv_timer_.schedule(netif_->txtime(p), p);

done:
	return;
}


void 
MacNULL::discard(Packet *p, const char* why)
{
	hdr_macnull * mh = (hdr_macnull *)p->access(off_mac_);

	if (hdr_dst(mh) == index)
		drop(p, why);
	else
		Packet::free(p);
}

void
MacNULL::send(Packet *, Handler *)
{
	fprintf(stderr,"DFU: shouldn't ever reach here %s\n",
		__PRETTY_FUNCTION__);
	abort();
}

void
MacNULL::sendComplete(Packet *p)
{
	hdr_macnull* mh = (hdr_macnull*)p->access(off_mac_);

	// provide a mac layer failure indication if needed.
	// this is completely bogus, but this is a test-only MAC
	// layer anyway
	if (hdr_dst(mh) != MAC_BROADCAST) {
	        /* commented out by jinyang, directly use god data to compute distance
		if (hdr_src(mh) != hdr_dst(mh)
		    && God::instance()->hops(hdr_src(mh), hdr_dst(mh)) == 0) {
			fprintf(stderr,"Mac/NULL mac layer requires that scenario file have\n"
				"  god data added.  see calcdest or setdest\n");
			abort();
		}
		if (God::instance()->hops(hdr_src(mh), hdr_dst(mh)) > 1) {
			// tell the callback the send operation failed  
			hdr_cmn *ch = HDR_CMN(p);
			if (ch->xmit_failure_) {
				ch->xmit_reason_ = XMIT_REASON_RTS;
				ch->xmit_failure_(p->copy(), ch->xmit_failure_data_);
			}
		}
	       */

	       //the following is added by jinyang
	       double my_x, my_y, my_z;
	       double his_x, his_y, his_z;
	       nsaddr_t dst = hdr_dst(mh);
	       (God::instance()->nodelist[hdr_src(mh)])->getLoc(&my_x,&my_y,&my_z);
	       (God::instance()->nodelist[dst])->getLoc(&his_x,&his_y,&his_z);
	       double dist = sqrt((my_x-his_x)*(my_x-his_x) + (my_y - his_y) * (my_y - his_y));
	       if ((dist > RANGE) || (!God::instance()->isAwake(dst))) { //the send probably has failured
		    hdr_cmn *ch = HDR_CMN(p);
		    if (ch->xmit_failure_) {
			ch->xmit_reason_ = XMIT_REASON_RTS;
			ch->xmit_failure_(p->copy(), ch->xmit_failure_data_);
	    	    }
		}
	       //above is added by jinyang

	}
	
	Packet::free(p); // no drop reason needed

	// have the upper layer (probably the IFQ) unblock itself now
	assert(upcall_ != 0);
	Handler *h = upcall_;
	upcall_ = 0;
	h->handle((Event *) 0);
}

void
MacNULL::recvComplete(Packet *p)
{
	p->incoming = 1;

	if (tap) {
		tap->tap(p);
	}

	recvtarget_->recv(p, (Handler *) 0);
}

void
MacNULLTimer::schedule(double t, Packet *p)
{
	Scheduler& s = Scheduler::instance();
	assert(busy_ == 0);
	s.schedule(this, p, t);
	busy_ = 1;
}

void 
MacNULLSend::handle(Event *e)
{
	Packet *p = (Packet *)e;  // cast of trust
	busy_ = 0;
	mac_->sendComplete(p);
}

void
MacNULLRecv::schedule(double t, Packet *p)
{
	Scheduler& s = Scheduler::instance();
	busy_++;
	s.schedule(this, p, t);
}

void 
MacNULLRecv::handle(Event *e)
{
	Packet *p = (Packet *)e;  // cast of trust
	busy_--;
	mac_->recvComplete(p);
}
