/* -*- c++ -*-
   mac-802_11.h

   $Id: mac-802_11.h,v 1.2 2001/12/11 18:19:25 benjie Exp $
   */
#ifndef __mac_80211_h__
#define __mac_80211_h__

#define RTS_RETRY_COUNTER 1

#include <cmu/mac.h>
#include <cmu/mac-timers.h>
#include <cmu/tsftimer.h>
#include <cmu/marshall.h>
#include <cmu/energy-model.h>
#include <cmu/node.h>
#include <cmu/setdest/sys/queue.h>
#include <cmu/sharedmedia.h>

// delay through the PHY components.  XXXXXX todo (minor): this number
// is a complete guess!  need an educated guess!
const double PHY_DELAY= 0.0001; 

#define DROP_MAC_BUFFER_FULL "BUF"

#define PKT_BUF_SIZ 10

#define GET_ETHER_TYPE(x)	GET2BYTE((x))
#define SET_ETHER_TYPE(x,y)	{u_int16_t t = (y); STORE2BYTE(&t,x);}
#define HDR_MAC11(p)	 ((struct hdr_mac802_11*)(p)->access(off_mac_))
#define HDR_BEACON(p)       ((struct hdr_beacon *)(p)->access(off_beacon_))

/* ======================================================================
   Frame Formats
   ====================================================================== */

#define	MAC_ProtocolVersion	0x00

#define MAC_Type_Management	0x00
#define MAC_Type_Control	0x01
#define MAC_Type_Data		0x02
#define MAC_Type_Reserved	0x03

#define MAC_Subtype_ATIM        0x09
#define MAC_Subtype_Beacon      0x0A
#define MAC_Subtype_RTS		0x0B
#define MAC_Subtype_CTS		0x0C
#define MAC_Subtype_ACK		0x0D
#define MAC_Subtype_Data	0x00

struct frame_control {
	u_char		fc_subtype		: 4;
	u_char		fc_type			: 2;
	u_char		fc_protocol_version	: 2;

	u_char		fc_order		: 1;
	u_char		fc_wep			: 1;
	u_char		fc_more_data		: 1;
	u_char		fc_pwr_mgt		: 1;
	u_char		fc_retry		: 1;
	u_char		fc_more_frag		: 1;
	u_char		fc_from_ds		: 1;
	u_char		fc_to_ds		: 1;
};

struct rts_frame {
	struct frame_control	rf_fc;
	u_int16_t		rf_duration;
	u_char			rf_ra[ETHER_ADDR_LEN];
	u_char			rf_ta[ETHER_ADDR_LEN];
	u_char			rf_fcs[ETHER_FCS_LEN];
};

struct cts_frame {
	struct frame_control	cf_fc;
	u_int16_t		cf_duration;
	u_char			cf_ra[ETHER_ADDR_LEN];
	u_char			cf_fcs[ETHER_FCS_LEN];
};

struct ack_frame {
	struct frame_control	af_fc;
	u_int16_t		af_duration;
	u_char			af_ra[ETHER_ADDR_LEN];
	u_char			af_fcs[ETHER_FCS_LEN];
};

struct hdr_mac802_11 {
	struct frame_control	dh_fc;
	u_int16_t		dh_duration;
	u_char			dh_da[ETHER_ADDR_LEN];
	u_char			dh_sa[ETHER_ADDR_LEN];
	u_char			dh_bssid[ETHER_ADDR_LEN];
	u_int16_t		dh_scontrol;
	u_char			dh_body[0];
};


/* ======================================================================
   Definitions
   ====================================================================== */
#define ETHER_HDR_LEN11				\
	((phymib->PreambleLength >> 3) +	\
	 (phymib->PLCPHeaderLength >> 3) +	\
	 sizeof(struct hdr_mac802_11) +		\
	 ETHER_FCS_LEN)

#define ETHER_RTS_LEN				\
	((phymib->PreambleLength >> 3) +        \
         (phymib->PLCPHeaderLength >> 3) +      \
	 sizeof(struct rts_frame))

#define ETHER_CTS_LEN				\
        ((phymib->PreambleLength >> 3) +        \
         (phymib->PLCPHeaderLength >> 3) +      \
         sizeof(struct cts_frame))

#define ETHER_ACK_LEN				\
        ((phymib->PreambleLength >> 3) +        \
         (phymib->PLCPHeaderLength >> 3) +      \
	 sizeof(struct ack_frame))

#define ETHER_BEACON_LEN(bh)			  \
        ((phymib->PreambleLength >> 3) +          \
         (phymib->PLCPHeaderLength >> 3) +        \
         sizeof(struct hdr_mac802_11) +           \
	 sizeof(u_int64_t) +                      \
	 sizeof(u_int16_t) +                      \
	 sizeof(struct cap_info) +                \
	 ssid_size(&bh->bh_bb.bb_ssid) +          \
	 supported_rates_size(&bh->bh_bb.bb_sr) + \
	 sizeof(struct fh_param_set) +            \
	 sizeof(u_int8_t) +                       \
	 sizeof(struct cf_param_set) +            \
	 sizeof(struct ibss_param_set) +          \
	 tim_size(&bh->bh_bb.bb_tim) +            \
	 (sizeof(u_char) * ETHER_FCS_LEN))
	 
	 
#define	RTS_Time	(netif_->txtime(ETHER_RTS_LEN))
#define CTS_Time	(netif_->txtime(ETHER_CTS_LEN))
#define ACK_Time	(netif_->txtime(ETHER_ACK_LEN))
#define DATA_Time(len)	(netif_->txtime((len)))
#define BEACON_Time(bh) (netif_->txtime(ETHER_BEACON_LEN(bh)))

/*
 *  IEEE 802.11 Spec, section 9.2.5.7
 *	- After transmitting an RTS, a node waits CTSTimeout
 *	  seconds for a CTS.
 *
 *  IEEE 802.11 Spec, section 9.2.8
 *	- After transmitting DATA, a node waits ACKTimeout
 *	  seconds for an ACK.
 *
 *  IEEE 802.11 Spec, section 9.2.5.4
 *	- After hearing an RTS, a node waits NAVTimeout seconds
 *	  before resetting its NAV.  I've coined the variable
 *	  NAVTimeout.
 *
 */
#define CTSTimeout	((RTS_Time + CTS_Time) + 2 * sifs)
#define ACKTimeout(len)	(DATA_Time(len) + ACK_Time + sifs + difs)
#define NAVTimeout	(2 * phymib->SIFSTime + CTS_Time + 2 * phymib->SlotTime)



#define RTS_DURATION(pkt)	\
	usec(sifs + CTS_Time + sifs + TX_Time(pkt) + sifs + ACK_Time)

#define CTS_DURATION(d)		\
	usec((d * 1e-6) - (CTS_Time + sifs))

#define DATA_DURATION()		\
	usec(ACK_Time + sifs)

#define ACK_DURATION()	0x00		// we're not doing fragments now

/*
 * IEEE 802.11 Spec, section 15.3.2
 *	- default values for the DSSS PHY MIB
 */
#define DSSS_CWMin			31
#define DSSS_CWMax			1023
#define DSSS_SlotTime			0.000020	// 20us
#define	DSSS_CCATime			0.000015	// 15us
#define DSSS_RxTxTurnaroundTime		0.000005	// 5us
#define DSSS_SIFSTime			0.000010	// 10us
#define DSSS_PreambleLength		144		// 144 bits
#define DSSS_PLCPHeaderLength		48		// 48 bits

class PHY_MIB {
public:
	u_int32_t	CWMin;
	u_int32_t	CWMax;
	double		SlotTime;
	double		CCATime;
	double		RxTxTurnaroundTime;
	double		SIFSTime;
	u_int32_t	PreambleLength;
	u_int32_t	PLCPHeaderLength;
};


/*
 * IEEE 802.11 Spec, section 11.4.4.2
 *      - default values for the MAC Attributes
 */
#define MAC_RTSThreshold		3000	  // bytes
#define MAC_ATIMRetryLimit		3	  // ATIM retries
#define MAC_RTSRetryLimit		10	  // RTS retry limits
#define MAC_ShortRetryLimit		7	  // retransmittions
#define MAC_LongRetryLimit		4	  // retransmissions
#define MAC_FragmentationThreshold	2346	  // bytes
#define MAC_MaxTransmitMSDULifetime	512	  // time units
#define MAC_MaxReceiveLifetime		512	  // time units
const double MAC_TBTTPreWake=           0.00001;  // 10 us
	
class MAC_MIB {
public:
	//		MACAddress;
	//		GroupAddresses;
	u_int32_t	RTSThreshold;
	u_int32_t	ShortRetryLimit;
	u_int32_t	LongRetryLimit;
	u_int32_t	FragmentationThreshold;
	u_int32_t	MaxTransmitMSDULifetime;
	u_int32_t	MaxReceiveLifetime;
	//		ManufacturerID;
	//		ProductID;

	u_int32_t	TransmittedFragmentCount;
	u_int32_t	MulticastTransmittedFrameCount;
	u_int32_t	FailedCount;	
	u_int32_t	RetryCount;
	u_int32_t	MultipleRetryCount;
	u_int32_t	FrameDuplicateCount;
	u_int32_t	RTSSuccessCount;
	u_int32_t	RTSFailureCount;
	u_int32_t	ACKFailureCount;
	u_int32_t	ReceivedFragmentCount;
	u_int32_t	MulticastReceivedFrameCount;
	u_int32_t	FCSErrorCount;
        double          BeaconPeriod;
        double          ATIMWindow;
        double          TBTTPreWake;
	double		AdvWindow;
};


/* ======================================================================
   The following destination class is used for duplicate detection.
   ====================================================================== */
class Host {
public:
	LIST_ENTRY(Host) link;
	u_int32_t	index;
	u_int32_t	seqno;
};

class buffer_entry;

/* ======================================================================
   The actual 802.11 MAC class.
   ====================================================================== */
class Mac802_11 : public Mac {
	friend class DeferTimer;
	friend class BackoffTimer;
	friend class IFTimer;
	friend class NavTimer;
	friend class RxTimer;
	friend class TxTimer;
        friend class TBTTTimer;
        friend class BeaconTimer;
        friend class AdvEndTimer;
        friend class ATIMEndTimer;
        friend class buffer_entry;

        friend void SharedMedia::set_state(power_t); 
public:
	Mac802_11(PHY_MIB* p, MAC_MIB *m);
        ~Mac802_11();
	void		recv(Packet *p, Handler *h);
	inline MacAddr	hdr_dst(void* hdr, u_int32_t dst = 0);
	inline MacAddr	hdr_src(void* hdr, u_int32_t src = 0);
	inline int	hdr_type(void* hdr, u_int16_t type = 0);

	//jinyang
	double	bftime();
	int i_am_idle();
	void handleBackoffInterrupt();

//protected: jinyang made all public
	void	backoffHandler(void);
	void	deferHandler(void);
	void	navHandler(void);
	void	recvHandler(void);
	void	sendHandler(void);
	void	txHandler(void);
        void    tbttHandler(void);
        void    beaconHandler(void);
        void    advEndHandler(void);
        void    atimEndHandler(void);

//private: jinyang made all public
	int		command(int argc, const char*const* argv);
	void		Terminate(void); /* called at end of simulation */

	/*
	 * Called by the timers.
	 */
	void		recv_timer(void);
	void		send_timer(void);
	int		check_pktCTRL();
	int		check_pktRTS();
	int		check_pktTx();
        int             check_pktBEACON();
        int             check_pktATIM();

	/*
	 * Packet Transmission Functions.
	 */
	void	send(Packet *p, Handler *h);
	void 	sendRTS(int dst);
	void	sendCTS(int dst, double duration);
	void	sendACK(int dst);
	void	sendDATA(Packet *p);
	void	sendBEACON();       
	void	RetransmitRTS();
	void	RetransmitDATA();
        void    RetransmitATIM();
    
        inline void TRANSMIT(Packet *p, double t);


	/*
	 * Packet Reception Functions.
	 */
	void	recvRTS(Packet *p);
	void	recvCTS(Packet *p);
	void	recvACK(Packet *p);
	void	recvDATA(Packet *p);
        void    recvBEACON(Packet *p);
        void    recvATIM(Packet *p);

        // these two functions scan the buffer and copy the appropriate
        // packets into pktTx/pktRTS/pktATIM.
        void            atim_scan();
        bool            tx_scan();

	void		capture(Packet *p);
	void		collision(Packet *p);
	void		discard(Packet *p, const char* why);
	void		rx_resume(void);
	void		tx_resume(void);
	void 		goto_sleep();

	// scan and reclaim packets
	Packet *	reclaim_packet(int dst);

	inline int	is_idle(void);

	/*
	 * Debugging Functions.
	 */
	void		trace_pkt(Packet *p);
	void		dump(char* fname);

        int             initialized_;
	inline int initialized() {
		return (phymib && macmib && cache && logtarget_ && 
			initialized_ && Mac::initialized());
	}
        void init();
	void mac_log(Packet *p) {
		logtarget_->recv(p, (Handler*) 0);
	}
	inline double TX_Time(Packet *p) {
		struct hdr_cmn *ch = HDR_CMN(p);
		double t = DATA_Time(ch->size());

		if(t < 0.0) {
			drop(p, "XXX");
			exit(1);
		}
		return t;
	}

	inline void inc_cw() {
		cw = (cw << 1) + 1;
		if(cw > phymib->CWMax)
			cw = phymib->CWMax;
	}

	inline void rst_cw() { cw = phymib->CWMin; }

	inline void set_nav(u_int16_t us) {
		double now = Scheduler::instance().clock();
		double t = us * 1e-6;

		if((now + t) > nav) {
			nav = now + t;
			if(mhNav.busy())
				mhNav.stop();
			mhNav.start(t);
		}
	}

	inline u_int16_t usec(double t) {
		u_int16_t us = (u_int16_t) (t *= 1e6);
		if(us < t)
			us++;
		return us;
	}
public:
	int total_retransmit;
	double total_wasted_time;
	double idle_start;
	double total_true_idle_time;

//protected: jinyang made all public
	PHY_MIB		*phymib;
	MAC_MIB		*macmib;

private:
        // (SPAN) assume that routers are eternally awake
        bool span_fastroute_;   

        // offset of beacon header
        int off_beacon_;
	// offset of span header
        int off_span_;

        EnergyModel *energy_model_;
        MobileNode *node_;
        Handler *upcall_;
        CIRCLEQ_HEAD(buffer_head, buffer_entry) buf_head_;
	void    upcall();
        
        // have we {sent, received} an ATIM before?
        bool first_atim_, recvd_atim_;
	unsigned has_packet_tx_;
	unsigned recvd_bcast_atim_;
	unsigned recvd_ucast_atim_;
	unsigned initial_wake_count_;

	bool span_opt_;
        bool psm_mode_;
	bool got_beacon_; 

        // address we last sent an ATIM to
	u_char	last_atim_da_[ETHER_ADDR_LEN]; 

//private: jinyang made all public
public:
	/*
	 * Mac Timers
	 */
	IFTimer		mhIF;		// interface timer
	NavTimer	mhNav;		// NAV timer
	RxTimer		mhRecv;		// incoming packets
	TxTimer		mhSend;		// outgoing packets

	DeferTimer	mhDefer;	// defer timer
	BackoffTimer	mhBackoff;	// backoff timer
        TBTTTimer       mhTBTT;         // TBTT timer, schedule start of ATIM
        BeaconTimer     mhBeacon;       // beacon timer
        AdvEndTimer     mhAdvEnd;       // end of adv window
        ATIMEndTimer    mhEndATIM;      // ATIM end of window
    
        TSFTimer        tsfTimer; 

	/* ============================================================
	   Internal MAC State
	   ============================================================ */
	double		nav;		// Network Allocation Vector

	MacState	rx_state;	// incoming state (MAC_RECV or MAC_IDLE)
	MacState	tx_state;	// outgoint state
#if 1
	int		tx_active;	// transmitter is ACTIVE
#endif

        double          first_bit_time; // time first bit of pkt was recvd
	Packet		*pktRTS;	// outgoing RTS packet
	Packet		*pktCTRL;	// outgoing non-RTS packet
        Packet          *pktBEACON;     // outgoing Beacon packet
        Packet          *pktATIM;       // outgoing ATIM packet
        buffer_entry    *entryATIM;     // outgoing ATIM packet's buf entry

	u_int32_t	cw;		// Contention Window
	u_int32_t	ssrc;		// STA Short Retry Count
	u_int32_t	slrc;		// STA Long Retry Count
	u_int32_t       atimrc;         // ATIM Retry Count
	u_int32_t	rtsrc;		// RTS retry counter
	double		sifs;		// Short Interface Space
	double		pifs;		// PCF Interframe Space
	double		difs;		// DCF Interframe Space
	double		eifs;		// Extended Interframe Space
	double		tx_sifs;
	double		tx_pifs;
	double		tx_difs;
	int             tx_opt;         // Optimistic tx

	int		min_frame_len;
#if 0
	NsObject*	logtarget_;
#endif
	/* ============================================================
	   Duplicate Detection state
	   ============================================================ */
	u_int16_t	sta_seqno;	// next seqno that I'll use
	int		cache_node_count;
	Host		*cache;
};

/*
 * An entry in the Packet buffer
 */
class buffer_entry {
public:
    buffer_entry(u_int32_t addr, bool sent_atim,
		   bool recvd_ack, bool sent_data, int age) :
	packet_(0), addr_(addr), sent_atim_(sent_atim),
	recvd_ack_(recvd_ack), sent_data_(sent_data), 
	pwr_mgt_(true), pwr_mgt_updated_(false), age_(age) {}

    buffer_entry(Packet *packet, bool sent_atim,
		   bool recvd_ack, bool sent_data, int age) :
	packet_(packet), addr_(0), sent_atim_(sent_atim),
	recvd_ack_(recvd_ack), sent_data_(sent_data), 
	pwr_mgt_(true), pwr_mgt_updated_(false), age_(age) {}
    
    Packet *packet_;
    u_int32_t addr_;
    bool sent_atim_;
    bool recvd_ack_;
    bool sent_data_;
    bool pwr_mgt_;
    bool pwr_mgt_updated_;
    unsigned int age_;

    CIRCLEQ_ENTRY(buffer_entry) entries;

    u_int32_t addr(Mac802_11 *mac) {
	if (packet_) {
	    struct hdr_mac802_11 *dh = 
		(struct hdr_mac802_11 *)packet_->access(mac->off_mac_);
	    return ETHER_ADDR(dh->dh_da);
	}
	return addr_;
    }

    void dump(Mac802_11 *mac) {
	int uid = -1;
	if (packet_) {
	    struct hdr_cmn *ch = HDR_CMN(packet_);
	    uid = ch->uid_;
	}
#if 0
	fprintf(stdout, "%f: buffer entry to %x: sent ATIM=%d, received ACK=%d, sent DATA=%d, age=%u, uid=%d\n",
		Scheduler::instance().clock(),
		addr(mac), 
		sent_atim_, 
		recvd_ack_, 
		sent_data_, 
		age_,
		uid);
#endif
#ifdef DEBUG
	// let's check that this atim has a state that we expect!
	#define sa sent_atim_ 
	#define ra recvd_ack_
	#define sd sent_data_
	assert( (!sa && !ra && !sd) ||
		(sa && !ra && !sd) ||
		(!sa && ra && !sd) ||
		(sa && ra && !sd) ||
		
		(sa && ra && sd) ||
		(!sa && ra && sd) ||
		(sa && !ra && sd));
#endif
    }
};

/*
 * packet buffer entry states:
 * 
 * sent_atim     recvd_ack     sent_data     meaning
 * 0             0             0
 * 1             0             0
 * 0             1             0
 * 1             1             0
 * ----------------------------------------------------------------------------
 * 1             1             1
 * 0             1             1
 * 1             0             1
 *
 */

#endif /* __mac_80211_h__ */

