/* -*- c++ -*-
   
   sr_forwarder.cc
   source route forwarder
   $Id: sr_forwarder.cc,v 1.1.1.1 2001/11/15 19:43:17 benjie Exp $

   */


#include <object.h>
#include <packet.h>
#include <ip.h>
#include <cmu/mac.h>
#include <cmu/hdr_sr.h>
#include <cmu/dsr/dsr_proto.h>
#include <cmu/dsr/routecache.h>
#include <cmu/dsr/requesttable.h>
#include <cmu/dsr/sr_forwarder.h>

/*===========================================================================
  static classes for OTcl
---------------------------------------------------------------------------*/
static class SRForwarderClass : public TclClass {
public:
  SRForwarderClass() : TclClass("SRForwarder") {}
  TclObject* create(int, const char*const*) {
    return (new SRForwarder);
  }
} class_srforwarder;


/*===========================================================================
  Member functions
---------------------------------------------------------------------------*/
SRForwarder::SRForwarder()
{
  target_ = 0;
  route_agent_ = 0;
  tracetarget = 0;
  
  bind("off_sr_", &off_sr_);
  bind("off_ll_", &off_ll_);
  bind("off_mac_", &off_mac_);
  bind("off_ip_", &off_ip_);

}

int
SRForwarder::command(int argc, const char*const* argv)
{
  Tcl& tcl = Tcl::instance();

  if (argc == 2) 
    {
      if (strcmp(argv[1], "target") == 0) {
	if (target_ != 0)
	  tcl.result(target_->name());
	return (TCL_OK);
      }
      if (strcmp(argv[1], "routeagent") == 0) {
	if (route_agent_ != 0)
	  tcl.result(route_agent_->name());
	return (TCL_OK);
      }
    }
  else if(argc == 3) 
    {
      TclObject *obj;
      
      if( (obj = TclObject::lookup(argv[2])) == 0) {
	fprintf(stderr, "SRForwarder: %s lookup of %s failed\n", argv[1],
		argv[2]);
	return TCL_ERROR;
      }

      if (strcasecmp(argv[1], "tracetarget") == 0) 
	{
	  tracetarget = (Trace*) obj;
	  return TCL_OK;
	}
      else if (strcasecmp(argv[1], "routecache") == 0) 
	{
	  routecache_ = (RouteCache*) obj;
	  return TCL_OK;
	}
      else if (strcasecmp(argv[1], "target") == 0) 
	{
	  target_ = (NsObject*) obj;
	  return TCL_OK;
	}
      else if (strcasecmp(argv[1], "routeagent") == 0) 
	{
	  route_agent_ = (DSRProto*) obj;
	  return TCL_OK;
	}
    }
  
  return NsObject::command(argc, argv);
}

void
SRForwarder::handlePktWithoutSR(Packet *p)
{
  hdr_sr *srh =  (hdr_sr*)p->access(off_sr_);
  hdr_ip *iph =  (hdr_ip*)p->access(off_ip_);
  hdr_cmn *cmnh =  (hdr_cmn*)p->access(off_cmn_);
  Path path;

  /* check route cache for source route for this packet */
  if (routecache_->findRoute(ID(iph->dst()>>8,::IP),path))
    {
      srh->init();
      path.fillSR(srh);
      sprintf(tracetarget->buffer(),
	      "$hit %g originating 0x%x -> 0x%x %s", 
	      Scheduler::instance().clock(),
	      iph->src(), iph->dst(), srh->dump());
      tracetarget->dump();
      this->recv(p,0);		// send back to forwarder to send this out
      return;
    }
  
  /* if not found, give packet to routing protocol */
  sprintf(tracetarget->buffer(),
	  "dfu %g srforwarder: %s no route to 0x%x", 
	  Scheduler::instance().clock(),
	  routecache_->net_id.dump(), iph->dst());
  tracetarget->dump();

  route_agent_->noRouteForPacket(p);
}

void
SRForwarder::recv(Packet* p, Handler* callback)
{
  hdr_sr *srh =  (hdr_sr*)p->access(off_sr_);

  assert(tracetarget != 0);

  if (!srh->valid())
    {
      handlePktWithoutSR(p);
      return;
    }

  /* forward as per source route */
  hdr_ip *iph =  (hdr_ip*)p->access(off_ip_);
  hdr_cmn *cmnh =  (hdr_cmn*)p->access(off_cmn_);
  
  if (srh->route_request())
    { /* just broadcast */
      cmnh->next_hop() = MAC_BROADCAST;
      cmnh->addr_type() = AF_LINK;
      target_->recv(p, (Handler*)0);
      return;	  
    }
  
  /* it must be a normal data packet with a source route */
  if (tracetarget)
    {
      sprintf(tracetarget->buffer(),
	      "F %g 0x%x -> 0x%x via 0x%x %s", 
	      Scheduler::instance().clock(),
	      iph->src(), iph->dst(), srh->get_next_addr(), srh->dump());
      tracetarget->dump();
    }
  
  cmnh->next_hop() = srh->get_next_addr();
  cmnh->addr_type() = AF_INET;
  srh->cur_addr() = srh->cur_addr() + 1;
  target_->recv(p, (Handler*)0);
  return;
}
