#
# Copyright (c) 1995-1997 The Regents of the University of California.
# All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions
# are met:
# 1. Redistributions of source code must retain the above copyright
#    notice, this list of conditions and the following disclaimer.
# 2. Redistributions in binary form must reproduce the above copyright
#    notice, this list of conditions and the following disclaimer in the
#    documentation and/or other materials provided with the distribution.
# 3. All advertising materials mentioning features or use of this software
#    must display the following acknowledgement:
#	This product includes software developed by the Computer Systems
#	Engineering Group at Lawrence Berkeley Laboratory.
# 4. Neither the name of the University nor of the Laboratory may be used
#    to endorse or promote products derived from this software without
#    specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
# ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
# ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
# FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
# DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
# OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
# HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
# LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
# OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
# SUCH DAMAGE.
#
# @(#) $Header: /home/cr1/benjie/cvsroot/ns/our-ns/tcl/test/test-suite-red.tcl,v 1.1.1.1 2001/06/20 15:38:31 benjie Exp $
#
# This test suite reproduces most of the tests from the following note:
# Floyd, S., 
# Ns Simulator Tests for Random Early Detection (RED), October 1996.
# URL ftp://ftp.ee.lbl.gov/papers/redsims.ps.Z.
#
# To run all tests: test-all-red

set dir [pwd]
catch "cd tcl/test"
source misc.tcl
source topologies.tcl
catch "cd $dir"

set flowfile fairflow.tr; # file where flow data is written
set flowgraphfile fairflow.xgr; # file given to graph tool 

TestSuite instproc finish file {
	global quiet
	$self instvar ns_ tchan_ testName_
        exec ../../bin/getrc -s 2 -d 3 all.tr | \
          ../../bin/raw2xg -a -s 0.01 -m 90 -t $file > temp.rands
	if {$quiet == "false"} {
        	exec xgraph -bb -tk -nl -m -x time -y packets temp.rands &
	}
        ## now use default graphing tool to make a data file
        ## if so desired

	if { [info exists tchan_] && $quiet == "false" } {
		$self plotQueue $testName_
	}
	$ns_ halt
}

TestSuite instproc enable_tracequeue ns {
	$self instvar tchan_ node_
	set redq [[$ns link $node_(r1) $node_(r2)] queue]
	set tchan_ [open all.q w]
	$redq trace curq_
	$redq trace ave_
	$redq attach $tchan_
}

#
# Reconfigure the net2 topology for the RED experiments.
#
Topology/net2 instproc config ns {
    $self instvar node_
    # force identical behavior to ns-1.
    # the recommended value for linterm is now 10
    # and is placed in the default file (3/31/97)
    [[$ns link $node_(r1) $node_(r2)] queue] set linterm_ 50
    [[$ns link $node_(r2) $node_(r1)] queue] set linterm_ 50
	
}

TestSuite instproc plotQueue file {
	global quiet
	$self instvar tchan_
	#
	# Plot the queue size and average queue size, for RED gateways.
	#
	set awkCode {
		{
			if ($1 == "Q" && NF>2) {
				print $2, $3 >> "temp.q";
				set end $2
			}
			else if ($1 == "a" && NF>2)
				print $2, $3 >> "temp.a";
		}
	}
	set f [open temp.queue w]
	puts $f "TitleText: $file"
	puts $f "Device: Postscript"

	if { [info exists tchan_] } {
		close $tchan_
	}
	exec rm -f temp.q temp.a 
	exec touch temp.a temp.q

	exec awk $awkCode all.q

	puts $f \"queue
	exec cat temp.q >@ $f  
	puts $f \n\"ave_queue
	exec cat temp.a >@ $f
	###puts $f \n"thresh
	###puts $f 0 [[ns link $r1 $r2] get thresh]
	###puts $f $end [[ns link $r1 $r2] get thresh]
	close $f
	if {$quiet == "false"} {
		exec xgraph -bb -tk -x time -y queue temp.queue &
	}
}

TestSuite instproc tcpDumpAll { tcpSrc interval label } {
    global quiet
    $self instvar dump_inst_ ns_
    if ![info exists dump_inst_($tcpSrc)] {
	set dump_inst_($tcpSrc) 1
	set report $label/window=[$tcpSrc set window_]/packetSize=[$tcpSrc set packetSize_]
	if {$quiet == "false"} {
		puts $report
	}
	$ns_ at 0.0 "$self tcpDumpAll $tcpSrc $interval $label"
	return
    }
    $ns_ at [expr [$ns_ now] + $interval] "$self tcpDumpAll $tcpSrc $interval $label"
    set report time=[$ns_ now]/class=$label/ack=[$tcpSrc set ack_]/packets_resent=[$tcpSrc set nrexmitpack_]
    if {$quiet == "false"} {
    	puts $report
    }
}       


Class Test/red1 -superclass TestSuite
Test/red1 instproc init topo {
    $self instvar net_ defNet_ test_
    set net_	$topo
    set defNet_	net2
    set test_	red1
    $self next
}
Test/red1 instproc run {} {
    $self instvar ns_ node_ testName_

    set stoptime 10.0
    
    set tcp1 [$ns_ create-connection TCP/Reno $node_(s1) TCPSink $node_(s3) 0]
    $tcp1 set window_ 15

    set tcp2 [$ns_ create-connection TCP/Reno $node_(s2) TCPSink $node_(s3) 1]
    $tcp2 set window_ 15

    set ftp1 [$tcp1 attach-source FTP]
    set ftp2 [$tcp2 attach-source FTP]

    $self enable_tracequeue $ns_
    $ns_ at 0.0 "$ftp1 start"
    $ns_ at 3.0 "$ftp2 start"

    $self tcpDump $tcp1 5.0

    # trace only the bottleneck link
    $self traceQueues $node_(r1) [$self openTrace $stoptime $testName_]

    $ns_ run
}

Class Test/ecn -superclass TestSuite
Test/ecn instproc init topo {
    $self instvar net_ defNet_ test_
    Queue/RED set setbit_ true
    set net_	$topo
    set defNet_	net2
    set test_	ecn
    $self next
}
Test/ecn instproc run {} {
    $self instvar ns_ node_ testName_

    set stoptime 10.0
    set redq [[$ns_ link $node_(r1) $node_(r2)] queue]
    $redq set setbit_ true

    set tcp1 [$ns_ create-connection TCP/Reno $node_(s1) TCPSink $node_(s3) 0]
    $tcp1 set window_ 15
    $tcp1 set ecn_ 1

    set tcp2 [$ns_ create-connection TCP/Reno $node_(s2) TCPSink $node_(s3) 1]
    $tcp2 set window_ 15
    $tcp2 set ecn_ 1
        
    set ftp1 [$tcp1 attach-source FTP]
    set ftp2 [$tcp2 attach-source FTP]
        
    $self enable_tracequeue $ns_
    $ns_ at 0.0 "$ftp1 start"
    $ns_ at 3.0 "$ftp2 start"
        
    $self tcpDump $tcp1 5.0
        
    # trace only the bottleneck link
    $self traceQueues $node_(r1) [$self openTrace $stoptime $testName_]
        
    $ns_ run
}

# "Red2" changes some of the RED gateway parameters.
# This should give worse performance than "red1".
Class Test/red2 -superclass TestSuite
Test/red2 instproc init topo {
    $self instvar net_ defNet_ test_
    set net_	$topo
    set defNet_	net2
    set test_	red2
    $self next
}
Test/red2 instproc run {} {
    $self instvar ns_ node_ testName_

    set stoptime 10.0
    set redq [[$ns_ link $node_(r1) $node_(r2)] queue]
    $redq set thresh_ 5
    $redq set maxthresh_ 10
    $redq set q_weight_ 0.003
	
    set tcp1 [$ns_ create-connection TCP/Reno $node_(s1) TCPSink $node_(s3) 0]
    $tcp1 set window_ 15

    set tcp2 [$ns_ create-connection TCP/Reno $node_(s2) TCPSink $node_(s3) 1]
    $tcp2 set window_ 15

    set ftp1 [$tcp1 attach-source FTP]
    set ftp2 [$tcp2 attach-source FTP]

    $self enable_tracequeue $ns_
    $ns_ at 0.0 "$ftp1 start"
    $ns_ at 3.0 "$ftp2 start"

    $self tcpDump $tcp1 5.0
    
    # trace only the bottleneck link
    $self traceQueues $node_(r1) [$self openTrace $stoptime $testName_]

    $ns_ run
}

# The queue is measured in "packets".
Class Test/red_twoway -superclass TestSuite
Test/red_twoway instproc init topo {
    $self instvar net_ defNet_ test_
    set net_	$topo
    set defNet_	net2
    set test_	red_twoway
    $self next
}
Test/red_twoway instproc run {} {
    $self instvar ns_ node_ testName_

    set stoptime 10.0
	
    set tcp1 [$ns_ create-connection TCP/Reno $node_(s1) TCPSink $node_(s3) 0]
    $tcp1 set window_ 15
    set tcp2 [$ns_ create-connection TCP/Reno $node_(s2) TCPSink $node_(s4) 1]
    $tcp2 set window_ 15
    set ftp1 [$tcp1 attach-source FTP]
    set ftp2 [$tcp2 attach-source FTP]

    set tcp3 [$ns_ create-connection TCP/Reno $node_(s3) TCPSink $node_(s1) 2]
    $tcp3 set window_ 15
    set tcp4 [$ns_ create-connection TCP/Reno $node_(s4) TCPSink $node_(s2) 3]
    $tcp4 set window_ 15
    set ftp3 [$tcp3 attach-source FTP]
    set telnet1 [$tcp4 attach-source Telnet] ; $telnet1 set interval_ 0

    $self enable_tracequeue $ns_
    $ns_ at 0.0 "$ftp1 start"
    $ns_ at 2.0 "$ftp2 start"
    $ns_ at 3.5 "$ftp3 start"
    $ns_ at 1.0 "$telnet1 start"

    $self tcpDump $tcp1 5.0

    # trace only the bottleneck link
    $self traceQueues $node_(r1) [$self openTrace $stoptime $testName_]

    $ns_ run
}

# The queue is measured in "bytes".
Class Test/red_twowaybytes -superclass TestSuite
Test/red_twowaybytes instproc init topo {
    $self instvar net_ defNet_ test_
    set net_	$topo
    set defNet_	net2
    set test_	red_twowaybytes
    $self next
}
Test/red_twowaybytes instproc run {} {
    $self instvar ns_ node_ testName_

    set stoptime 10.0
    set redq [[$ns_ link $node_(r1) $node_(r2)] queue]
    $redq set bytes_ true
    $redq set queue-in-bytes_ true
		
    set tcp1 [$ns_ create-connection TCP/Reno $node_(s1) TCPSink $node_(s3) 0]
    $tcp1 set window_ 15
    set tcp2 [$ns_ create-connection TCP/Reno $node_(s2) TCPSink $node_(s4) 1]
    $tcp2 set window_ 15
    set ftp1 [$tcp1 attach-source FTP]
    set ftp2 [$tcp2 attach-source FTP]

    set tcp3 [$ns_ create-connection TCP/Reno $node_(s3) TCPSink $node_(s1) 2]
    $tcp3 set window_ 15
    set tcp4 [$ns_ create-connection TCP/Reno $node_(s4) TCPSink $node_(s2) 3]
    $tcp4 set window_ 15
    set ftp3 [$tcp3 attach-source FTP]
    set telnet1 [$tcp4 attach-source Telnet] ; $telnet1 set interval_ 0

    $self enable_tracequeue $ns_
    $ns_ at 0.0 "$ftp1 start"
    $ns_ at 2.0 "$ftp2 start"
    $ns_ at 3.5 "$ftp3 start"
    $ns_ at 1.0 "$telnet1 start"

    $self tcpDump $tcp1 5.0

    # trace only the bottleneck link
    $self traceQueues $node_(r1) [$self openTrace $stoptime $testName_]

    $ns_ run
}

#
#######################################################################

TestSuite instproc create_flowstats {} {

	global flowfile flowchan
	$self instvar ns_ node_ r1fm_

	set r1fm_ [$ns_ makeflowmon Fid]
	set flowchan [open $flowfile w]
	$r1fm_ attach $flowchan
	$ns_ attach-fmon [$ns_ link $node_(r1) $node_(r2)] $r1fm_ 1
#	$ns_ attach-fmon [$ns_ link $node_(r1) $node_(r2)] $r1fm_ 
}

#
# awk code used to produce:
#       x axis: # arrivals for this flow+category / # total arrivals [bytes]
#       y axis: # drops for this flow+category / # drops this category [pkts]
#	(verified compatible for ns2 - kfall, 10/30/97)
TestSuite instproc unforcedmakeawk { } {
        set awkCode {
            {
                if ($2 != prev) {
                        print " "; print "\"flow " $2;
			if ($13 > 0 && $14 > 0) {
			    print 100.0 * $9/$13, 100.0 * $10 / $14
			}
			prev = $2;
                } else if ($13 > 0 && $14 > 0) {
                        print 100.0 * $9 / $13, 100.0 * $10 / $14
		}
            }
        }
        return $awkCode
}

#
# awk code used to produce:
#       x axis: # arrivals for this flow+category / # total arrivals [bytes]
#       y axis: # drops for this flow+category / # drops this category [bytes]
#	(modified for compatibility with ns2 flowmon - kfall, 10/30/97)
TestSuite instproc forcedmakeawk { } {
        set awkCode {
            BEGIN { print "\"flow 0" }
            {
                if ($2 != prev) {
                        print " "; print "\"flow " $2;
			if ($13 > 0 && ($17 - $15) > 0) {
				print 100.0 * $9/$13, 100.0 * ($19 - $11) / ($17 - $15);
			}
			prev = $2;
                } else if ($13 > 0 && ($17 - $15) > 0) {
                        print 100.0 * $9 / $13, 100.0 * ($19 - $11) / ($17 - $15)
		}
            }
        }
        return $awkCode
}

#
# awk code used to produce:
#      x axis: # arrivals for this flow+category / # total arrivals [bytes]
#      y axis: # drops for this flow / # drops [pkts and bytes combined]
TestSuite instproc allmakeawk { } {
    set awkCode {
        BEGIN {prev=-1; tot_bytes=0; tot_packets=0; forced_total=0; unforced_total=0}
        {
            if ($5 != prev) {
                print " "; print "\"flow ",$5;
                prev = $5
            }
            tot_bytes = $19-$11;
            forced_total= $16-$14;
            tot_packets = $10;
            tot_arrivals = $9;
            unforced_total = $14;
            if (unforced_total + forced_total > 0) {
                if ($14 > 0) {
                    frac_packets = tot_packets/$14;
                }
                else { frac_packets = 0;}
                if ($17-$15 > 0) {
                    frac_bytes = tot_bytes/($17-$15);
                }
                else {frac_bytes = 0;} 
                if ($13 > 0) {
                    frac_arrivals = tot_arrivals/$13;
                }
                else {frac_arrivals = 0;}
                if (frac_packets + frac_bytes > 0) {
                    unforced_total_part = frac_packets * unforced_total / ( unforced_total + forced_total)
                    forced_total_part = frac_bytes * forced_total / ( unforced_total + forced_total)
                    print 100.0 * frac_arrivals, 100.0 * ( unforced_total_part +forced_total_part)
                }
            }
        }
    }
    return $awkCode
}

TestSuite instproc create_flow_graph { graphtitle graphfile } {
        global flowfile
	$self instvar awkprocedure_

puts "awkprocedure: $awkprocedure_"

        set tmpfile1 /tmp/fg1[pid]
        set tmpfile2 /tmp/fg2[pid]

        exec rm -f $graphfile
        set outdesc [open $graphfile w]
        #
        # this next part is xgraph specific
        #
        puts $outdesc "TitleText: $graphtitle"
        puts $outdesc "Device: Postscript"

        exec rm -f $tmpfile1 $tmpfile2
        puts "writing flow xgraph data to $graphfile..."

        exec sort -n +1 -o $flowfile $flowfile
        exec awk [$self $awkprocedure_] $flowfile >@ $outdesc
        close $outdesc
}

TestSuite instproc finish_flows testname {
	global flowgraphfile flowfile flowchan quiet
	$self instvar r1fm_
	$r1fm_ dump
	close $flowchan
	$self create_flow_graph $testname $flowgraphfile
	puts "running xgraph..."
	exec cp $flowgraphfile temp.rands
	if {$quiet == "false"} {
		exec xgraph -bb -tk -nl -m -lx 0,100 -ly 0,100 -x "% of data bytes" -y "% of discards" $flowgraphfile &
	}
	exit 0
}

TestSuite instproc new_tcp { startTime source dest window class verbose size } {
	$self instvar ns_
	set tcp [$ns_ create-connection TCP/Reno $source TCPSink $dest $class ]
	$tcp set window_ $window
	if {$size > 0}  {$tcp set packetSize_ $size }
	set ftp [$tcp attach-source FTP]
	$ns_ at $startTime "$ftp start"
	if {$verbose == "1"} {
	  $self tcpDumpAll $tcp 20.0 $class 
	}
}

TestSuite instproc new_cbr { startTime source dest pktSize interval class } {
	$self instvar ns_
    set cbr [$ns_ create-connection CBR $source LossMonitor $dest $class]
    if {$pktSize > 0} {
	$cbr set packetSize_ $pktSize
    }
    $cbr set interval_ $interval
    $ns_ at $startTime "$cbr start"
}

TestSuite instproc dumpflows interval {
    $self instvar dumpflows_inst_ ns_ r1fm_
    $self instvar awkprocedure_ dump_pthresh_
    global flowchan

    if ![info exists dumpflows_inst_] {
        set dumpflows_inst_ 1
        $ns_ at 0.0 "$self dumpflows $interval"
        return  
    }
    if { $awkprocedure_ == "unforcedmakeawk" } {
	set pcnt [$r1fm_ set epdrops_]
    } elseif { $awkprocedure_ == "forcedmakeawk" } {
	set pcnt [expr [$r1fm_ set pdrops_] - [$r1fm_ set epdrops_]]
    } elseif { $awkprocedure_ == "allmakeawk" } {
	set pcnt [$r1fm_ set pdrops_]
    } else {
	puts stderr "unknown handling of flow dumps!"
	exit 1
    }
    if { $pcnt >= $dump_pthresh_ } {
	    $r1fm_ dump
	    flush $flowchan
	    foreach f [$r1fm_ flows] {
		$f reset
	    }
	    $r1fm_ reset
	    set interval 2.0
    } else {
	    set interval 1.0
    }
    $ns_ at [expr [$ns_ now] + $interval] "$self dumpflows $interval"
}   

TestSuite instproc droptest { stoptime } {
	$self instvar ns_ node_ testName_ r1fm_ awkprocedure_

	set forwq [[$ns_ link $node_(r1) $node_(r2)] queue]
	set revq [[$ns_ link $node_(r2) $node_(r1)] queue]

	$forwq set mean_pktsize_ 1000
	$revq set mean_pktsize_ 1000
	$forwq set linterm_ 10
	$revq set linterm_ 10
	$forwq set limit_ 100
	$revq set limit_ 100

	$self create_flowstats 
	$self dumpflows 10.0

	$forwq set bytes_ true
	$forwq set queue-in-bytes_ true
	$forwq set wait_ false

        $self new_tcp 1.0 $node_(s1) $node_(s3) 100 1 1 1000
	$self new_tcp 1.2 $node_(s2) $node_(s4) 100 2 1 50
	$self new_cbr 1.4 $node_(s1) $node_(s4) 190 0.003 3

	$ns_ at $stoptime "$self finish_flows $testName_"

	$ns_ run
}


Class Test/flows_unforced -superclass TestSuite
Test/flows_unforced instproc init topo {
    $self instvar net_ defNet_ test_
    set net_    $topo   
    set defNet_ net2
    set test_   flows_unforced
    $self next 0; # zero here means don't product all.tr
}   

Test/flows_unforced instproc run {} {

	$self instvar ns_ node_ testName_ r1fm_ awkprocedure_
	$self instvar dump_pthresh_
 
        set stoptime 500.0
	set testName_ test_flows_unforced
	set awkprocedure_ unforcedmakeawk
	set dump_pthresh_ 100

	$self droptest $stoptime

}

Class Test/flows_forced -superclass TestSuite
Test/flows_forced instproc init topo {
    $self instvar net_ defNet_ test_
    set net_    $topo   
    set defNet_ net2
    set test_   flows_forced
    $self next 0; # zero here means don't product all.tr
}   

Test/flows_forced instproc run {} {

	$self instvar ns_ node_ testName_ r1fm_ awkprocedure_
	$self instvar dump_pthresh_
 
        set stoptime 500.0
	set testName_ test_flows_forced
	set awkprocedure_ forcedmakeawk
	set dump_pthresh_ 100

	$self droptest $stoptime
}

Class Test/flows_combined -superclass TestSuite
Test/flows_combined instproc init topo {
    $self instvar net_ defNet_ test_
    set net_    $topo   
    set defNet_ net2
    set test_   flows_combined
    $self next 0; # zero here means don't product all.tr
}   

Test/flows_combined instproc run {} {

	$self instvar ns_ node_ testName_ r1fm_ awkprocedure_
	$self instvar dump_pthresh_
 
        set stoptime 500.0
	set testName_ test_flows_combined
	set awkprocedure_ allmakeawk
	set dump_pthresh_ 100

	$self droptest $stoptime
}

TestSuite runTest
