#include "hdr_span.h"
#include "spanagent.hh"
#include "gfrt.hh"

#define StaticAssert(c) switch (c) case 0: case (c):
#define NOW (Scheduler::instance().clock())
#define TRACE_ANNOUNCE      1
#define ROTATE_COORDINATORS 1

double SpanBcast = 3;
double SpanExpire = 4.5 * SpanBcast;
double SpanAnnounceCheck = SpanBcast/3;
double SpanWithdrawCheck = SpanBcast/3;
double SpanCoordinatorWindow = 45/SpanWithdrawCheck;
double SpanCoordinatorWindowMin = 10/SpanWithdrawCheck;

static inline double
jitter (double max)
{ 
  return Random::uniform(max);
}

void 
SpanAgent::recv(Packet *p, Handler *callback)
{ 
  hdr_ip *iph = (hdr_ip *) p->access (_off_ip); 
  hdr_cmn *cmnh =  (hdr_cmn*)p->access(off_cmn_);
  hdr_span *spanh =  (hdr_span*)p->access(_off_span);

  if (!alive()) {
    Packet::free(p);
    return;
  }
  cmnh->retries_ = 0;

  if (_my_id == iph->src() && cmnh->num_forwards()==0) { // source
    spanh->type = SPAN_DATA;
    iph->ttl_ = SPAN_DEF_TTL;
    cmnh->size() += IP_HDR_LEN + spanh->size();
    double x, y, z;
    (God::instance()->nodelist[iph->dst()])->getLoc(&x, &y, &z);
    spanh->dst_x = (int)x;
    spanh->dst_y = (int)y;
    forward_pkt(p);
    return;
  }
  
  process_span_hdr(p);

  if (iph->dst() == _my_id && spanh->type == SPAN_DATA) { // sink
    _pkts_received++;
    _pkts_hops += ((SPAN_DEF_TTL-iph->ttl())+1);
    drop(p, SPAN_DROP_RECEIVED);
    return;
  }

  if (asleep() && _usespan && spanh->type == SPAN_DATA) {
    double now = NOW;
    _force_counter++;
    if (_force_counter > 5) {
      if (!is_coordinator() && _usespan) {
#if TRACE_ANNOUNCE > 0
        fprintf(stderr, "# %f: %d forced to announce\n", now, _my_id);
#endif
	announce_coordinator(true);
      }
    }
    if (_last_force == 0 || (now - _last_force) > 1) {
      _last_force = now;
      _force_counter = 0;
    }
  }

  switch(spanh->type) {
    case SPAN_DATA:
      forward_pkt(p);
      break;
    case SPAN_HELLO:
      process_span_hello(p);
      Packet::free(p);
      break;
    default:
      Packet::free(p);
      break;
  }
}

void 
SpanAgent::process_span_hdr(Packet *p)
{
  hdr_span *spanh =  (hdr_span*)p->access(_off_span);
  NeighborTable::Neighbor *n = _neighbors.search_neighbor(spanh->node_id);

  if (!spanh->is_coordinator && n && n->is_coordinator)
      _last_withdraw_seen = Scheduler::instance().clock();

  NeighborTable::Neighbor *new_n = new NeighborTable::Neighbor();
  new_n->id = spanh->node_id;
  new_n->x = spanh->x;
  new_n->y = spanh->y;
  new_n->nounce = spanh->nounce;
  new_n->is_coordinator = spanh->is_coordinator;
  new_n->coordinators = 0;

  if (_neighbors.update_neighbor(*new_n) < 0) {
    new_n->coordinators = 0;
    new_n->neighbors.clear();
    _neighbors.add_neighbor(new_n);
    n = new_n;
  } else
    delete new_n;

  if (spanh->is_coordinator)
    _neighbors.add_coordinator(_my_id, spanh->node_id, spanh->is_tentative);
  else
    _neighbors.remove_coordinator(_my_id, spanh->node_id);
}

void
SpanAgent::process_span_hello(Packet *p)
{
  hdr_span *spanh =  (hdr_span*)p->access(_off_span);
  span_hello *hello =  (span_hello*)p->access(_off_span_hello);

  NeighborTable::Neighbor *n = _neighbors.search_neighbor(spanh->node_id);
  if (!n) {
    n = new NeighborTable::Neighbor();
    n->id = spanh->node_id;
    n->x = spanh->x;
    n->y = spanh->y;
    n->nounce = spanh->nounce;
    n->is_coordinator = spanh->is_coordinator;
    n->coordinators = 0;
    _neighbors.add_neighbor(n);
  }

  n->clear_coordinators();
  for (unsigned i=0; i<hello->ncoordinators; i++)
    n->add_coordinator(hello->coordinators[i], false);
   
  n->neighbors.clear();
  for (unsigned i=0; i<hello->nneighbors; i++) {
    n->neighbors.push_back(hello->neighbors[i]);
    if (spanh->is_coordinator)
      _neighbors.add_coordinator
	(hello->neighbors[i], spanh->node_id, spanh->is_tentative);
  }
}

void
SpanAgent::add_span_hdr(Packet *p)
{
  hdr_span *spanh =  (hdr_span*)p->access(_off_span);
  spanh->is_coordinator = _is_coordinator;
  spanh->is_tentative = _is_tentative;
  double x, y, z;
  (God::instance()->nodelist[_my_id])->getLoc(&x, &y, &z);
  spanh->node_id = _my_id;
  spanh->x = (int)x;
  spanh->y = (int)y;
  spanh->nounce = (int)NOW;
}

void
SpanAgent::broadcast_hello()
{
  NeighborTable::Neighbor *self = _neighbors.search_neighbor(_my_id);
  _neighbors.expire_old_entries(_my_id, SpanExpire);

  Packet *p = allocpkt();
  hdr_cmn *cmnh =  (hdr_cmn*)p->access(off_cmn_);
  hdr_span *spanh =  (hdr_span*)p->access(_off_span);
  span_hello *hello =  (span_hello*)p->access(_off_span_hello);
  spanh->type = SPAN_HELLO;

  unsigned i = 0;
  LLIterator *c_i = self->coord_iter();
  NeighborTable::Coordinator *c = (NeighborTable::Coordinator*) c_i->peek();
  while (c && i < MAX_SPAN_COORDINATORS) {
    if (!c->tentative) {
      hello->coordinators[i] = c->id;
      i++;
    }
    c = (NeighborTable::Coordinator*)c_i->next();
  }
  delete c_i;
  hello->ncoordinators = i;

  i = 0;
  LLIterator *n_i = _neighbors.neighbors();
  NeighborTable::Neighbor *n = (NeighborTable::Neighbor*) n_i->peek();
  while(n && i < MAX_SPAN_NEIGHBORS) {
    hello->neighbors[i] = n->id;
    i++;
    n = (NeighborTable::Neighbor*)n_i->next();
  }
  delete n_i;
  hello->nneighbors = i;

  cmnh->next_hop_ = MAC_BROADCAST;
  cmnh->size() = spanh->size() + hello->size();
  hdr_ip *iph =  (hdr_ip*)p->access(off_ip_);
  iph->dst() = IP_BROADCAST;
  iph->src() = _my_id;

  add_span_hdr(p);
  process_span_hello(p); // this keeps self->neighbors[] in sync
  send_packet(p, false);
  
  double x, y, z;
  (God::instance()->nodelist[_my_id])->getLoc(&x, &y, &z);
  self->x = (int)x;
  self->y = (int)y;
  self->nounce = (int)NOW;

  _last_bcast = NOW;
}

  
void
SpanAgent::sleep()
{
  assert(!_is_coordinator);
  _asleep = true;
  if (_usepsm)
    _mobile_node->PowerSave(0, true);
}

void
SpanAgent::wakeup()
{
  _asleep = false;
  if (_usepsm)
    _mobile_node->PowerSave(0, false);
}

void
SpanAgent::update_location()
{ 
  double x, y, z; 
  (God::instance()->nodelist[_my_id])->getLoc(&x, &y, &z); 
  NeighborTable::Neighbor *self = _neighbors.search_neighbor(_my_id);
  self->x = (int)x;
  self->y = (int)y;
  self->nounce = (int)Scheduler::instance().clock();
}

void
SpanEventHandler::handle(Event *e)
{
  double now = Scheduler::instance().clock();
  if (_agent->active()) {
    bool should_broadcast = false;
    double since_withdraw = now - _last_withdraw;
    double since_broadcast = now - _last_broadcast;
   
    _agent->update_location();
   
    if (since_broadcast >= SpanBcast)
      should_broadcast = true; 
 
    if (_agent->id() > _agent->_srcsink) {
      // after withdraw, stay up for a couple of seconds to forward packets
      if (!_agent->is_coordinator() && (since_withdraw >= 2)) {
        if (!_agent->asleep()) 
	  _agent->sleep();
      }

      if (!_agent->is_coordinator() && _last_announce_check < now &&
	  (now - _last_announce_check >= SpanAnnounceCheck)) {
        double delay = _agent->check_announce_handler();
        _last_announce_check = now;
        _last_withdraw_check = now + delay + jitter(2);
      }
    
      else if (_agent->is_coordinator() && _last_withdraw_check < now &&
	       (now - _last_withdraw_check >= SpanWithdrawCheck)) {
        should_broadcast |= _agent->check_withdraw_handler();
        _last_withdraw_check = now;
        if (!_agent->is_coordinator()) { // withdrawn
	  _last_withdraw = now;
	  int n = _agent->_neighbors.nneighbors();
	  _last_announce_check = now + n; // give others a chance to announce
        }
      }

    }

    else if (_agent->asleep()) { // src/sink always awake in experiments
      _agent->wakeup();
      fprintf(stderr, "# %d src/sink\n", _agent->id());
    }

    if (should_broadcast) {
      _agent->broadcast_hello();
      _last_broadcast = now;
    }
    Scheduler::instance().schedule(this,e,0.1);

  } else {
#if TRACE_ANNOUNCE > 0
    fprintf(stderr,"# %f: %d out of energy\n", now, _agent->id());
#endif
  }
}

double
SpanAgent::check_announce_handler()
{
  if (is_coordinator() || (_usespan && !active()))
    return 0;

  if (!_announce_pending) {
    int c;
    if (_usespan && (c = check_announce()) > 0) {
      int n = _neighbors.nneighbors();
      double fe = 1-_energy_model->energy() / _energy_model->initialenergy();
      double U = (n == 1) ? 1 : (1-(((double)c)/(double)(n*(n-1)*0.5)));
      if (U < 0) U = 0;
      if (U > 1) U = 1;

      double R = jitter(1);
      int slot = (int)(n*(fe+U+R));
      double delay = slot * 0.4; // 400 ms propagation delay

      // compensate so everyone starts competing at the same time
      double now = Scheduler::instance().clock();
      if ((now-_last_withdraw_seen) < SpanAnnounceCheck) {
        double jj = _last_withdraw_seen+SpanAnnounceCheck-now;
        delay += jj;
      }

#if TRACE_ANNOUNCE > 1
      fprintf(stderr,"# %d: d=%3.3f (s %d n %d c %d fe %3.3f e %5.1f U %f)\n", 
              id(), delay, slot, n, c, fe, _energy_model->energy(), U);
#endif
      _announce_pending = true; 
      Event *ev = new Event; 
      Scheduler::instance().schedule 
        (&_announce_coordinator_handler, ev, delay); 
      return delay;
    } 
    else if (!_usespan) {
      _announce_pending = true; 
      Event *ev = new Event; 
      Scheduler::instance().schedule (&_announce_coordinator_handler, ev, 0); 
      return 0;
    }
  }
  return 0;
}

int
SpanAgent::check_announce()
{
  if (_my_id <= _srcsink)
    return 0;

  int nn = 0;
  LLIterator *ni = _neighbors.neighbors();
  NeighborTable::Neighbor *n = (NeighborTable::Neighbor*)ni->peek();
  while(n) {
    LLIterator *ni2 = _neighbors.neighbors();
    NeighborTable::Neighbor *n2 = (NeighborTable::Neighbor*)ni2->peek();
    while (n2) {
      if (n2->id > n->id) {
        int j = 0;
	if ((j = _neighbors.share_coordinator(n2->id, n->id, -1)) < 0) {
#if TRACE_ANNOUNCE > 2
	  fprintf(stderr,"## %d: %d %d share %d (%f)\n",
	          _my_id, n2->id, n->id, j, Scheduler::instance().clock());
#endif
	  nn++; 
	}
      }
      n2 = (NeighborTable::Neighbor*)ni2->next();
    }
    delete ni2;
    n = (NeighborTable::Neighbor*)ni->next();
  }
  delete ni;
  return nn;
}

// returns true if a broadcast msg should be sent afterward
bool
SpanAgent::check_withdraw_handler()
{
  double fe = _energy_model->energy() / _energy_model->initialenergy();
  unsigned window = (unsigned)(SpanCoordinatorWindow*fe);
  if (window < SpanCoordinatorWindowMin)
    window = (unsigned)SpanCoordinatorWindowMin;

  if (!_usespan)
    return false;

  _withdraw_meter++;

  if (!active() || check_withdraw(false)) {
    _withdraw_meter = 0;
    _is_tentative = false;
    withdraw_coordinator();
    return true;
  }

#if ROTATE_COORDINATORS
  else if (_withdraw_meter >= window) {
    if (check_withdraw(true)) {
      if (!_is_tentative) {
        _is_tentative = true;
#if TRACE_ANNOUNCE > 1
        fprintf(stderr, "# %f: %d coordinator is tentative\n", NOW, _my_id);
#endif
	return true;
      }
      unsigned n = _neighbors.nneighbors();
      n = (unsigned)(n/SpanWithdrawCheck)+1;
      if (_withdraw_meter >= window+n) {
	// nobody else elected
        _withdraw_meter = 0;
	_is_tentative = false;
#if TRACE_ANNOUNCE > 1
        fprintf(stderr, "# %f: %d coordinator not tentative\n", NOW, _my_id);
#endif
	return true;
      }
    }
    else {
#if TRACE_ANNOUNCE > 1
      fprintf(stderr, "# %f: %d coordinator must stay on\n", NOW, _my_id);
#endif
      _withdraw_meter = 0;
      if (_is_tentative) {
        _is_tentative = false;
	return true;
      }
    }
  }
#endif
  return false;
}

bool
SpanAgent::check_withdraw(bool force_withdraw)
{
  bool withdraw = true;
  LLIterator *ni = _neighbors.neighbors();
  NeighborTable::Neighbor *n = (NeighborTable::Neighbor*)ni->peek();
  while (n && withdraw) {
    if (n->id == _my_id) {
      n = (NeighborTable::Neighbor*)ni->next();
      continue;
    }
    LLIterator *ni2 = _neighbors.neighbors();
    NeighborTable::Neighbor *n2 = (NeighborTable::Neighbor*)ni2->peek();
    while(n2 && withdraw) {
      int joint;
      if ((joint = _neighbors.share_coordinator(n->id, n2->id, _my_id))<0) {
	if (force_withdraw) {
          if ((joint = _neighbors.share_neighbor(n->id, n2->id, _my_id))<0) {
#if TRACE_ANNOUNCE > 2
	    fprintf(stderr,"%d: %d+%d not joint, force\n",_my_id,n->id,n2->id);
#endif
	    withdraw = false;
	  }
#if TRACE_ANNOUNCE > 2
	  else fprintf(stderr,"%d: %d+%d by n%d\n",_my_id,n->id,n2->id,joint);
#endif
	}
	else {
#if TRACE_ANNOUNCE > 2
          fprintf(stderr, "%d: %d+%d not joint\n", _my_id, n->id, n2->id);
#endif
	  withdraw = false;
	}
      }
#if TRACE_ANNOUNCE > 3
      else fprintf(stderr, "%d: %d+%d by c%d\n", _my_id, n->id, n2->id, joint);
#endif
      n2 = (NeighborTable::Neighbor*)ni2->next();
    }
    delete ni2;
    n = (NeighborTable::Neighbor*)ni->next();
  }
  delete ni;
  return withdraw;
}

void
SpanAnnounceCoordinatorHandler::handle(Event *e)
{
  _agent->announce_coordinator();
  delete e;
}

void
SpanAgent::announce_coordinator(bool force)
{
  _announce_pending = false;
  if (force || !_usespan || check_announce() > 0) {
    if (_asleep && _usespan) 
      wakeup();
#if TRACE_ANNOUNCE
    fprintf(stderr,"# %f: %d announce coordinator (%f/%f)\n", NOW, _my_id,
	    _energy_model->energy(),_energy_model->initialenergy());
#endif
    _is_coordinator = true;
    _neighbors.add_coordinator(_my_id, _my_id, false);
    NeighborTable::Neighbor *self = _neighbors.search_neighbor(_my_id);
    self->is_coordinator = true;
    self->nounce = (int)NOW;
    broadcast_hello();
  } 
}

void
SpanAgent::withdraw_coordinator()
{
#if TRACE_ANNOUNCE
  fprintf(stderr,"# %f: %d withdraw coordinator (%f/%f)\n",
	  NOW,_my_id,_energy_model->energy(),_energy_model->initialenergy());
#endif
  _is_coordinator = false;
  NeighborTable::Neighbor *self = _neighbors.search_neighbor(_my_id);
  self->is_coordinator = false;
  self->nounce = (int)NOW;
  _neighbors.remove_coordinator(_my_id, _my_id);
}

void
SpanAgent::forward_pkt(Packet *p)
{
  int err = _gfrt->forward_pkt(_my_id, (hdr_ip*)p->access(_off_ip),
                               (hdr_cmn*)p->access(off_cmn_),
			       (hdr_span*)p->access(_off_span));
  if (err == GF_TTL)
    drop(p, SPAN_GF_TTL);
  else if (err == GF_HOLE)
    drop(p, SPAN_GF_HOLE);
  else if (err == GF_FWD)
    send_packet(p, false);
  else
    send_packet(p, true);
}

void
SpanAgent::send_packet(Packet *p, bool to_coordinator)
{
  hdr_cmn *cmnh =  (hdr_cmn*)p->access(off_cmn_);
  hdr_span *spanh =  (hdr_span*)p->access(_off_span);
  hdr_ip *iph =  (hdr_ip*)p->access(off_ip_);
  add_span_hdr(p);
  if (_usespan) {
    if (cmnh->next_hop_ == iph->dst() && cmnh->next_hop_ != (int)MAC_BROADCAST)
      spanh->to_coordinator = true;
    else
      spanh->to_coordinator = to_coordinator;
  }
  else
    spanh->to_coordinator = false;
  cmnh->addr_type_ = AF_INET;
  cmnh->xmit_failure_ = tx_failure_cb;
  cmnh->xmit_failure_data_ = (void*) this;
#if USE_ARP 
  Scheduler::instance().schedule(_ll, p, 0);
#else
  _mac->hdr_dst(p->access(_off_mac), cmnh->next_hop_);
  _mac->hdr_src(p->access(_off_mac), _my_id);
  Scheduler::instance().schedule(_ifq, p, 0);
#endif
}

void
SpanAgent::tx_failure_cb(Packet *p, void *thunk)
{
  SpanAgent *agent = (SpanAgent*) thunk;
  hdr_cmn *cmnh =  (hdr_cmn*)p->access(agent->off_cmn_);
  cmnh->retries_++;

  if (cmnh->xmit_reason_ != XMIT_REASON_BUF && 
      cmnh->next_hop_ != (int)MAC_BROADCAST && agent->alive()) {
#if 0
    fprintf(stderr, "# %d remove %d because tx failed, reason: %d\n",
	    agent->id(), cmnh->next_hop_, cmnh->xmit_reason_);
#endif
    agent->_neighbors.remove_coordinator(agent->id(), cmnh->next_hop_);
    agent->_neighbors.remove_neighbor(cmnh->next_hop_);
    Packet *pp = 0;
    while((pp = agent->_mac->reclaim_packet(cmnh->next_hop_))) {
      agent->forward_pkt(pp);
      pp = 0;
    }
    int i=0;
    Packet* vec[64]; // cheat: we know IFQ max len is 50...
    while((pp = agent->_real_ifq->prq_get_nexthop(cmnh->next_hop_))) {
      vec[i] = pp; i++;
      if (i == 64) break;
      pp = 0;
    }
    for (int j=0; j<i; j++)
      agent->forward_pkt(vec[j]);
  }

  // don't retry broadcasts or drop because of stale packets in buffer
  if (!agent->alive() ||
      cmnh->next_hop_ == (int)MAC_BROADCAST || 
      cmnh->retries_ >= 3 ||
      cmnh->xmit_reason_ == XMIT_REASON_BUF) {
    if (cmnh->xmit_reason_ == XMIT_REASON_BUF)
      agent->drop(p, SPAN_DROP_BUF);
    else 
      agent->drop(p, SPAN_DROP_TX_FAILED);
    return;
  }

  agent->forward_pkt(p);
}

void
SpanAgent::dump_table()
{
  if (_my_id == 1) 
    fprintf(stderr,"StartOfDump\n");
  LLIterator *ni = _neighbors.neighbors();
  NeighborTable::Neighbor *n = (NeighborTable::Neighbor*)ni->peek();
  double x, y, z;
  (God::instance()->nodelist[_my_id])->getLoc(&x, &y, &z);
  while (n) {
    fprintf(stderr, "%f %f %d %c%d %d %d %d %f %f\n",
	    x, y, _my_id, n->is_coordinator ? 'c' : 'n', n->id, n->id,
	    1, n->nounce, _energy_model->energy(),
	    _energy_model->initialenergy());
    n = (NeighborTable::Neighbor*)ni->next();
  }
  delete ni;
}

static class SpanAgentClass : public TclClass
{
public:
  SpanAgentClass() : TclClass("Agent/SpanAgent") { }
  TclObject* create(int, const char *const *) { return new SpanAgent(); }
} class_SpanAgent;


SpanAgent::SpanAgent()
  : Agent(PT_SPAN),
    _event_handler(this), 
    _announce_coordinator_handler(this),
    _usespan(true),
    _usepsm(true),
    _srcsink(0)
{
  _pkts_sent = 0;
  _pkts_received = 0;
  _pkts_hops = 0;

  bind("off_span_hello_", &_off_span_hello);
  bind("off_span_", &_off_span);
  bind("off_ll_", &_off_ll);
  bind("off_mac_", &_off_mac);
  bind("off_ip_", &_off_ip);

  _gfrt = new GFRouter(_neighbors);
}

int
SpanAgent::command(int argc, const char*const* argv)
{
  TclObject *obj;  

  if (argc == 2) { 
    if (strcasecmp(argv[1], "start-span") == 0) { 
      start(); 
      return TCL_OK; 
    }
    if (strcasecmp(argv[1], "dump-tables") == 0) { 
      dump_table(); 
      return TCL_OK; 
    } 
  }
  else if(argc == 3) { 
    if (strcasecmp(argv[1], "ip-addr") == 0) { 
      _my_id = atoi(argv[2]); 
      return TCL_OK; 
    } 
    else if (strcasecmp(argv[1], "srcsink") == 0) {
      _srcsink = atoi(argv[2]); 
      return TCL_OK; 
    } 
    else if (strcasecmp(argv[1], "usepsm") == 0) { 
      int usepsm = atoi(argv[2]);
      if (usepsm > 0) 
	_usepsm = true;
      else 
	_usepsm = false;
      return TCL_OK; 
    }
    else if (strcasecmp(argv[1], "usespan") == 0) { 
      int usespan = atoi(argv[2]);
      if (usespan > 0) 
	_usespan = true;
      else 
	_usespan = false;
      return TCL_OK; 
    }
    else if (strcasecmp(argv[1], "addnetif") == 0) {
      _netif = (SharedMedia *) TclObject::lookup(argv[2]);;
      if (_netif == 0) {
        fprintf(stderr, "SpanAgent: %s lookup of %s failed\n", 
	        argv[1], argv[2]);
        return TCL_ERROR;
      }
      assert(_netif);
      return TCL_OK;
    }
    else if (strcasecmp(argv[1], "mobile-node") == 0) {
      _mobile_node = (MobileNode *) TclObject::lookup(argv[2]);;
      if (_mobile_node == 0) {
        fprintf(stderr, "SpanAgent: %s lookup of %s failed\n", 
	        argv[1], argv[2]);
        return TCL_ERROR;
      }
      assert(_mobile_node);
      return TCL_OK;
    }
  }
  else if (argc == 4) {
    if (strcasecmp(argv[1], "add-mac") == 0) {
      _real_ifq = _ifq = (PriQueue *) TclObject::lookup(argv[2]);
      if (_ifq == 0) {
        fprintf(stderr, "SpanAgent: %s lookup of %s failed\n", 
	        argv[1], argv[2]);
        return TCL_ERROR;
      } 
      _mac = (Mac802_11 *) TclObject::lookup(argv[3]);
      if (_mac == 0) {
        fprintf(stderr, "SpanAgent: %s lookup of %s failed\n", 
	        argv[1], argv[3]);
        return TCL_ERROR;
      }
      return TCL_OK;
    }

    if (strcasecmp(argv[1], "add-ll") == 0) { 
      if( (obj = TclObject::lookup(argv[2])) == 0) { 
	fprintf(stderr, "SpanAgent: %s lookup of %s failed\n", 
	        argv[1], argv[2]); 
	return TCL_ERROR; 
      }
      _ll = (NsObject*) obj;
      if((obj = TclObject::lookup(argv[3])) == 0) { 
	fprintf(stderr, "SpanAgent: %s lookup of %s failed\n", 
	        argv[1], argv[3]); 
	return TCL_ERROR; 
      } 
      _ifq = (PriQueue *) obj; 
      return TCL_OK; 
    } 
  }
  else if (argc == 5) {
    if (strcasecmp(argv[1], "add-mac") == 0) {
      _ifq = (PriQueue *) TclObject::lookup(argv[2]);
      if (_ifq == 0) {
        fprintf(stderr, "SpanAgent: %s lookup of %s failed\n", 
	        argv[1], argv[2]);
        return TCL_ERROR;
      } 
      _real_ifq = (PriQueue *) TclObject::lookup(argv[3]);
      if (_real_ifq == 0) {
        fprintf(stderr, "SpanAgent: %s lookup of %s failed\n", 
	        argv[1], argv[3]);
        return TCL_ERROR;
      } 
      _mac = (Mac802_11 *) TclObject::lookup(argv[4]);
      if (_mac == 0) {
        fprintf(stderr, "SpanAgent: %s lookup of %s failed\n", 
	        argv[1], argv[4]);
        return TCL_ERROR;
      }
      return TCL_OK;
    }
  }

  return Agent::command(argc, argv);
}

void
SpanAgent::start()
{
  _is_coordinator = false;
  _announce_pending = false;
  _withdraw_meter = 0;
  _last_withdraw = 0;
  _last_withdraw_seen = 0;
  _is_tentative = false;
  _energy_model = _mobile_node->energy_model();
  assert(_energy_model);
  _last_force = 0;
  _force_counter = 0;

  NeighborTable::Neighbor *self = new NeighborTable::Neighbor;
  self->id = _my_id;
  self->nounce = 0;
  self->x = 0;
  self->y = 0;
  self->is_coordinator = false;
  self->coordinators = 0;
  _neighbors.add_neighbor(self);
  
  double j = jitter(10);
  Scheduler::instance().schedule 
    (&_event_handler, &_event_handler_event, j);
  sleep();
}

