/* SharedMedia.cc 
 * $Id: sharedmedia.cc,v 1.2 2001/09/19 15:42:33 benjie Exp $ 
 */

#include <math.h>
#include <packet.h>

#define BENJIE_DEBUG

#include <cmu/node.h>
#include <cmu/net-if.h>
#include <cmu/propagation.h>
#include <cmu/modulation.h>
#include <cmu/omni-antenna.h>
#include <cmu/sharedmedia.h>

#define min(a,b) (((a)<(b))?(a):(b))
#define max(a,b) (((a)<(b))?(b):(a))

void Idle_Timer::expire(Event *)
{
    a_->UpdateIdleEnergy(Scheduler::instance().clock());
    if (a_->if_on())
	a_->idle_timer_.resched(1.0);
}

/* ======================================================================
   SharedMedia Interface
   ====================================================================== */
static class SharedMediaClass:public TclClass { public:
    SharedMediaClass():TclClass("NetIf/SharedMedia") {
    } TclObject *create(int, const char *const *) {
	return (new SharedMedia);
}}
class_SharedMedia;


SharedMedia::SharedMedia(void)
  : NetIf(), idle_timer_(this), state_(pwr_state_idle)
{
    Rb = 2 * 1e6;		// 2 Mb
    Pt = pow(10, 2.45) * 1e-3;	// 24.5 dbm, ~ 281.8mw
    lambda = SPEED_OF_LIGHT / (914 * 1e6); // 914 Mhz
    L = 1.0;
    freq = -1.0;

    /*
     *  It sounds like 10db should be the capture threshold.
     *
     *  If a node is presently receiving a packet a a power level
     *  Pa, and a packet at power level Pb arrives, the following
     *  comparion must be made to determine whether or not capture
     *  occurs:
     *
     *    10 * log(Pa) - 10 * log(Pb) > 10db
     *
     *  OR equivalently
     *
     *    Pa/Pb > 10.
     *
     */
    CPThresh = 10.0;
    CSThresh = 1.559e-11;
    RXThresh = 3.652e-10;

    bind("CPThresh_", &CPThresh);
    bind("CSThresh_", &CSThresh);
    bind("RXThresh_", &RXThresh);
    bind("Rb_", &Rb);
    bind("Pt_", &Pt);
    bind("freq_", &freq);
    bind("L_", &L);

    if (freq != -1.0) {		// freq was set by tcl code
	lambda = SPEED_OF_LIGHT / freq;
    }
    // these are measurements of the wavelan card
    Pt_consume_ = 1.4;		// 1.4 W drained power for transmission
    Pr_consume_ = 1.0;		// 1.0 W drained power for reception
    P_off_ = 0.043;		// negligible drained power for off
    P_idle_ = 0.830;		// 830 mW idle power consumption
    P_sleep_ = 0.130;		// 130 mW drained power for sleep

    channel_idle_time_ = update_energy_time_ =
	last_send_time_ = Scheduler::instance().clock();
    // channel_idle_time_ = update_energy_time_ = last_send_time_ = 0.0;  
    idle_timer_.resched(1.0);

    idle_time = 0;
    sleep_time = 0;
}

int SharedMedia::command(int argc, const char *const *argv)
{
    TclObject *obj;

    if (argc == 3) {
	if ((obj = TclObject::lookup(argv[2])) == 0) {
	    fprintf(stderr, "SharedMedia: %s lookup of %s failed\n",
		    argv[1], argv[2]);
	    return TCL_ERROR;
	}
	if (strcasecmp(argv[1], "antenna") == 0) {
	    ant = (Antenna *) obj;
	    return TCL_OK;
	}
    }
    return NetIf::command(argc, argv);
}

void SharedMedia::xmitPacket(Packet * p)
{
    EnergyModel *em = node()->energy_model();

    if (!if_listen()) {
	Packet::free(p);
	return;
    }

    /*
     * Decrease node's energy
     */
    if (em) {
	if (em->energy() > 0) {
	    double txtime = (8. * HDR_CMN(p)->size()) / Rb;
	    double start_time = max(channel_idle_time_,
				    Scheduler::instance().clock());
	    double end_time = max(channel_idle_time_,
				  Scheduler::instance().clock() + txtime);
	    double actual_txtime = end_time - start_time;

	    if (start_time > update_energy_time_) {
		UpdateIdleEnergy(start_time);
		if (!if_listen()) {
		    Packet::free(p);
		    return;
		}
	    }

	    /* It turns out that MAC sends packet even
	     * though, it's receiving some packets.

	     if (txtime-actual_txtime > 0.000001) { 
	     fprintf(stderr,"Something may be wrong at MAC\n"); 
	     fprintf(stderr,"act_tx = %lf, tx = %lf\n", actual_txtime, txtime); 
	     }
	     */

	    // Sanity check
	    double temp =
		max(Scheduler::instance().clock(), last_send_time_);
	    double begin_adjust_time = min(channel_idle_time_, temp);
	    double finish_adjust_time = min(channel_idle_time_,
					    Scheduler::instance().clock() +
					    txtime);
	    double gap_adjust_time =
		finish_adjust_time - begin_adjust_time;

	    if (gap_adjust_time < 0.0)
		fprintf(stderr, "What the heck ! negative gap time.\n");

	    if ((gap_adjust_time > 0.0) && (status_ == CH_STATUS_RECV))
		em->DecrTxEnergy(gap_adjust_time,
				 Pt_consume_ - Pr_consume_);
	    em->DecrTxEnergy(actual_txtime, Pt_consume_);

	    if (end_time > channel_idle_time_)
		status_ = CH_STATUS_SEND;

	    last_send_time_ = Scheduler::instance().clock() + txtime;
	    channel_idle_time_ = end_time;
	    update_energy_time_ = end_time;

	    if (em->energy() <= 0) {
		em->setenergy(0);
		node()->log_energy(0);
		// turn interface OFF
		state_ = pwr_state_off;
	    }
	} else {
	    Packet::free(p);
	    return;
	}
    }

    /* 
     * Stamp the packet with the interface arguments 
     */
    p->txinfo.stamp(node_, ant->copy(), Pt, lambda);

    // Send the packet
    channel_->recv(p, this);
}

int SharedMedia::recvPacket(Packet * p, double *RxPr)
{
    PacketStamp s;
    double Pr = 0;
    int pkt_recvd = 0;
    EnergyModel *em = node()->energy_model();

    // if the interface is in sleeping mode, drop the packet simply
    if (!if_listen() || (em && em->energy() <= 0)) {
	pkt_recvd = 0;
	goto DONE;
    }

    if (propagation_) {
	s.stamp(node_, ant, 0, lambda);
	Pr = propagation_->Pr(&p->txinfo, &s, this);
	if (Pr < CSThresh) {
	    pkt_recvd = 0;
	    goto DONE;
	}
	if (Pr < RXThresh) {
	    /*
	     * We can detect, but not successfully receive
	     * this packet.
	     */
	    hdr_cmn *hdr = HDR_CMN(p);
	    hdr->error() = 1;
#if DEBUG > 3
	    printf("SM %f.9 _%d_ drop pkt from %d low POWER %e/%e\n",
		   Scheduler::instance().clock(), node_->index(),
		   p->txinfo.getNode()->index(), Pr, RXThresh);
#endif
	}
    }

    if (modulation) {
	hdr_cmn *hdr = HDR_CMN(p);
	hdr->error() = modulation->BitError(Pr);
    }

    /*
     * The MAC layer must be notified of the packet reception
     * now - ie; when the first bit has been detected - so that
     * it can properly do Collision Avoidance / Detection.
     */
    pkt_recvd = 1;

  DONE:
    p->txinfo.getAntenna()->release();
    *RxPr = Pr;

    /* WILD HACK: The following two variables are a wild hack.
       They will go away in the next release...
       They're used by the mac-802_11 object to determine
       capture.  This will be moved into the net-if family of 
       objects in the future. */
    p->txinfo.RxPr = Pr;
    p->txinfo.CPThresh = CPThresh;

    /*
     * Decrease energy if packet successfully received
     */
    if (pkt_recvd && em) {
	double rcvtime = (8. * HDR_CMN(p)->size_) / Rb;

	/* node()->add_rcvtime(rcvtime);    
	 * em->DecrRcvEnergy(rcvtime,Pr_consume_);
	 */

	double start_time = max(channel_idle_time_,
				Scheduler::instance().clock());
	double end_time = max(channel_idle_time_,
			      Scheduler::instance().clock() + rcvtime);
	double actual_rcvtime = end_time - start_time;

	if (start_time > update_energy_time_) {
	    UpdateIdleEnergy(start_time);
	    if (!if_listen()) {
		pkt_recvd = 0;
		goto END;
	    }
	}

	em->DecrRcvEnergy(actual_rcvtime, Pr_consume_);

	if (end_time > channel_idle_time_)
	    status_ = CH_STATUS_RECV;

	channel_idle_time_ = end_time;
	update_energy_time_ = end_time;

	/*
	   hdr_diff *dfh = HDR_DIFF(p);
	   printf("Node %d receives (%d, %d, %d) energy %lf.\n",
	   node()->address(), dfh->sender_id.addr_, 
	   dfh->sender_id.port_, dfh->pk_num, node()->energy());
	 */

	if (em->energy() <= 0) {
	    em->setenergy(0);
	    node()->log_energy(0);
	    // turn interface off
	    state_ = pwr_state_off;
	}
    }
  END:
    if (!if_listen())
	pkt_recvd = 0;
    return pkt_recvd;
}

void SharedMedia::dump(void) const
{
    fprintf(stdout,
	    "--------------------------------------------------\n");
    NetIf::dump();
    fprintf(stdout, "\tPt: %f, Gt: %f, Gr: %f, lambda: %f, L: %f\n",
	    Pt, ant->getTxGain(0, 0, 0, lambda),
	    ant->getRxGain(0, 0, 0, lambda), lambda, L);
    fprintf(stdout,
	    "--------------------------------------------------\n");
}

void SharedMedia::UpdateIdleEnergy(double now)
{
    EnergyModel *em = node()->energy_model();
    if (em == NULL)
	return;
    if (now > update_energy_time_) {
	double P = 0;
	switch (state_) {
	case pwr_state_sleep:
	    sleep_time += now - update_energy_time_;
	    P = P_sleep_;
	    break;
	case pwr_state_off:
	    P = P_off_;
	    break;
	case pwr_state_idle:
	    idle_time += now - update_energy_time_;
	    P = P_idle_;
	    break;
	default:
	    assert(0);
	}
	em->DecrIdleEnergy(now - update_energy_time_, P);
	update_energy_time_ = now;
    }
    if (em->energy() <= 0) {
	em->setenergy(0);
	node()->log_energy(0);
	state_ = pwr_state_off;
    }
}

void SharedMedia::PowerSave(bool on)
{
    char state_char;
    UpdateIdleEnergy(Scheduler::instance().clock());
    if (on) {
	power_mode_ = pwr_mode_powersave;
	state_char = 'v';
    } else {
	power_mode_ = pwr_mode_awake;
	state_char = '^';
    }

    // log a power event
    Scheduler & s = Scheduler::instance();
    sprintf(node()->log_target()->buffer(),
	    "%c %.9f _%d_ %f", state_char, s.clock(), node_->index(),
	    node()->energy_model()->energy());
    node()->log_target()->dump();

    if (power_mode_ == pwr_mode_awake && state_ != pwr_state_idle)
      set_state(pwr_state_idle);
}

void SharedMedia::set_state(power_t state)
{
    UpdateIdleEnergy(Scheduler::instance().clock());
    state_ = state;
#ifdef BENJIE_DEBUG
    char state_char;
    if (state_ == pwr_state_idle)
      state_char = '-';
    else if (state == pwr_state_sleep)
      state_char = '~';
    else
      state_char = 'x';
    Scheduler & s = Scheduler::instance();
    sprintf(node()->log_target()->buffer(),
	    "%c %.9f _%d_ %f", state_char, s.clock(), node_->index(),
	    node()->energy_model()->energy());
    node()->log_target()->dump();
#endif
}

