#include <assert.h>
#include <fcntl.h>
#include <math.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/uio.h>
#include <unistd.h>
#include <err.h>

#include "quadtree.h"

#define SE 0
#define SW 1
#define NE 2
#define NW 3

#define _SE_ 6
#define _SW_ 5
#define _NE_ 10
#define _NW_ 9


Quadtree::Quadtree(double nwX, double nwY, double seX,double seY, int l, Quadtree *par, int ct) 
{

    assert(level>=0);
    level = l;

    nw_x = nwX;
    nw_y = nwY;

    se_x = seX;
    se_y = seY;

    LIST_INIT(&head);

    children = NULL;
    parent = par;
    child_type = ct;
}

Quadtree::~Quadtree()
{
    assert(children == NULL);
    assert(head.lh_first == NULL);
    if (children) {
	for (int i=0;i<4;i++) {
	    delete children[i];
	}
	delete children;
    }
}

/* this implementation is adaptive, i split only when there
   are more than 1 node in a grid 
void
Quadtree::insertNode(double x, double y, void *obj, qtree_node *&qtree_node_ptr, Quadtree *&qtree_ptr)
{

    double mid_x = (nw_x+se_x)/2;	
    double mid_y = (nw_y+se_y)/2;	


    if ((level>=1) && (head.lh_first!=NULL)) {
	//split into four children if my level is not zero and
	// i have more than 1 node in my territory
	children = new Quadtree*[4];
	
	children[NE] = new Quadtree(mid_x,nw_y,se_x,mid_y,level-1,this, NE);
	children[NW] = new Quadtree(nw_x,nw_y,mid_x,mid_y,level-1,this, NW);
	children[SW] = new Quadtree(nw_x,mid_y,mid_x,se_y,level-1,this, SW);
	children[SE] = new Quadtree(mid_x,mid_y,se_x,se_y,level-1,this, SE);

	// NOTE: i have to reinsert the node into one of my children
	qtree_node *p = head.lh_first;
	assert(p && (p->le_next == NULL));

	//create some dummy variables
	qtree_node_ptr *ptr1;
	qtree_ptr *ptr2;

	//insert the node to its appropriate children
	if ((p->x > mid_x) && (p->y > mid_y)) {
	    //insert it into NE child
	    children[NE]->insertNode(p->x,p->y,obj,ptr1,ptr2);
	}else if ((p->x <= mid_x) && (p->y > mid_y)) {
	    children[NW]->insertNode(p->x,p->y,obj,ptr1,ptr2);
	}else if ((p->x <= mid_x) && (p->y <= mid_y)) {
	    children[SW]->insertNode(p->x,p->y,obj,ptr1,ptr2);
	}else if ((p->x > mid_x) && (p->y <= mid_y)) {
	    children[SE]->insertNode(p->x,p->y,obj,ptr1,ptr2);
	}

	//delete the point from my nodelist
	head.lh_first = NULL;
    }

    //if i have children, i'll insert the node to one of my children
    //this is a recursive function
    if (children) {

	if ((x > mid_x) && (y > mid_y)) {
	    //insert it into NE child
	    children[NE]->insertNode(x,y,obj,qtree_node_ptr,qtree_ptr);
	}else if ((x <= mid_x) && (y > mid_y)) {
	    children[NW]->insertNode(x,y,obj,qtree_node_ptr,qtree_ptr);
	}else if ((x <= mid_x) && (y <= mid_y)) {
	    children[SW]->insertNode(x,y,obj,qtree_node_ptr,qtree_ptr);
	}else if ((x > mid_x) && (y <= mid_y)) {
	    children[SE]->insertNode(x,y,obj,qtree_node_ptry,qtree_ptr);
	}
    }else if ((level == 0) || (head.lh_first == NULL)) {
	//this is already the lowest level Quadtree, i.e. it's a leaf
	// or there's currently no node in it
	//store the node into the linked list
	qtree_node_ptr = new qtree_node;
	qtree_node_ptr->x = x;
	qtree_node_ptr->y = y;
	qtree_node_ptr->node = obj;
	LIST_INSERT_HEAD(&head,qtree_node_ptr,node_link);
	qtree_ptr = this;
    }
}
*/

/* this implementation is not adaptive, i always insert the
   node into the lowest level, i.e. level = 0 splitting
   grids along the way */
void
Quadtree::insertNode(double x, double y, void *obj, qtree_node *&qtree_node_ptr, Quadtree *&qtree_ptr)
{

    double mid_x = (nw_x+se_x)/2;	
    double mid_y = (nw_y+se_y)/2;	


    if (level == 0) {
	qtree_node_ptr = new qtree_node;
	qtree_node_ptr->x = x;
	qtree_node_ptr->y = y;
	qtree_node_ptr->node = obj;
	LIST_INSERT_HEAD(&head,qtree_node_ptr,node_link);
	qtree_ptr = this;
    }else {
	if (children ==NULL) {
	    //split into four children if my level is not zero and
	    // i have more than 1 node in my territory
	    children = new Quadtree*[4];

	    children[NE] = new Quadtree(mid_x,nw_y,se_x,mid_y,level-1,this, _NE_);
	    children[NW] = new Quadtree(nw_x,nw_y,mid_x,mid_y,level-1,this, _NW_);
	    children[SW] = new Quadtree(nw_x,mid_y,mid_x,se_y,level-1,this, _SW_);
	    children[SE] = new Quadtree(mid_x,mid_y,se_x,se_y,level-1,this, _SE_);
	}

	if ((x > mid_x) && (y > mid_y)) {
	    //insert it into NE child
	    children[SE]->insertNode(x,y,obj,qtree_node_ptr,qtree_ptr);
	}else if ((x <= mid_x) && (y > mid_y)) {
	    children[SW]->insertNode(x,y,obj,qtree_node_ptr,qtree_ptr);
	}else if ((x <= mid_x) && (y <= mid_y)) {
	    children[NW]->insertNode(x,y,obj,qtree_node_ptr,qtree_ptr);
	}else if ((x > mid_x) && (y <= mid_y)) {
	    children[NE]->insertNode(x,y,obj,qtree_node_ptr,qtree_ptr);
	}
    }
}


void
Quadtree::deleteNode(qtree_node *p)
{

    //since I do the re-insertion when splitting into 
    //four children, i have to be careful
    
    if (head.lh_first == NULL) {
	//take a look at x, y values before descending into
	//one of my children
	double mid_x = (nw_x+se_x)/2;	
        double mid_y = (nw_y+se_y)/2;	

	if ((p->x > mid_x) && (p->y > mid_y)) {
	    //insert it into NE child
	    children[SE]->deleteNode(p);
	}else if ((p->x <= mid_x) && (p->y > mid_y)) {
	    children[SW]->deleteNode(p);
	}else if ((p->x <= mid_x) && (p->y <= mid_y)) {
	    children[NW]->deleteNode(p);
	}else if ((p->x > mid_x) && (p->y <= mid_y)) {
	    children[NE]->deleteNode(p);
	}
    }else {
#ifdef GOD_DEBUG
	//check if the p is indeed in my nodelist
	qtree_node *ptr = head.lh_first;
	for(;ptr;ptr->node_link.le_next) {
	    if (ptr == p) goto NEXT;
	}
	abort();
#endif
NEXT:
	LIST_REMOVE(p,node_link);

	//although it probably won't happen, but as a
	/* general rule, the quadtree might shrink
	   however,there are also other nasty things such as 
	   keeping quadtree balanced, so let's forget it
	   for now

	if (head.lh_first == NULL) {
	    for (int i=0;i<4;i++) {
		if (((parent->children[i])->head).lh_first != NULL) {
		    return;
		}
	    }

	    //shrink the quadtree
	}
	*/
	delete p;
    }
}

Quadtree *
Quadtree::getNeighbor(int which_nb)
{

    if (parent == NULL) return NULL;

    Quadtree *tree;
    int c;
    int ns,ew,child_ns,child_ew;
  
    /* the numbering of N,S,E,W is like 
	             N: 1 0 0 0
                     S: 0 1 0 0
                     E: 0 0 1 0
                     W: 0 0 0 1
		    SE: 0 1 1 0
		    SW: 0 1 0 1
		    NE: 1 0 1 0
		    NW: 1 0 0 1
    */

    // split which_nb into NS and EW part
    ns = which_nb >> 2;
    ew = which_nb & 3;
    
    //split child_type into NS and EW part
    child_ns = child_type>> 2;
    child_ew = child_type & 3;

    if ((ns>=1) && (ew >=1)){
	if (((child_ns & ns) == 0) && ((child_ew & ew) == 0)) {
	    c = (ns&2) | (ew&1);
	    return parent->children[c];
	}else if ((child_ns & ns) == 0) {
	    tree = parent->getNeighbor(ew);
	    if ((!tree) || (!tree->level)) {
	        return tree;
	    }else {
		ew = 3 - ew;
		c = (ns&2) | (ew&1);
		return tree->children[c];
	    }
	}else if ((child_ew & ew) == 0) {
	    tree = parent->getNeighbor(ns<<2);
	    if ((!tree) || (!tree->level)) {
	        return tree;
	    }else {
		ns = 3 - ns;
		c = (ns&2) | (ew&1);
		return tree->children[c];
	    }
	}else {
	   tree = parent->getNeighbor(which_nb);
	   if ((!tree) || (!tree->level)) {
	        return tree;
	   }else {
	       ns = 3 - ns;
	       ew = 3 - ew;
	       c = (ns&2) | (ew&1);
	       return tree->children[c];
	   }
	}
    }else if (ns >= 1) {
	// which_nb is either N or S
	if ((child_ns & ns) == 0) {  //child_type is SW or SE
	    c = (ns&2)|(child_ew&1);
	    return parent->children[c];
	}else {
	    tree = parent->getNeighbor((ns<<2));
	    if ((!tree) || (!tree->level)) {
	        return tree;
	    }else {
		ns = 3 - ns; //turn N to S, S to N
		c = (ns & 2)|(child_ew & 1);
		return tree->children[c];
	    }
	}
    }else if (ew >= 1) {
	// which_nb is either E or W
	if ((child_ew & ew) == 0) {  //child_type is SW or SE
	    c = (child_ns&2)|(ew&1);
	    return parent->children[c];
	}else {
	    tree = parent->getNeighbor(ew);
	    if ((!tree) || (!tree->level)) {
	        return tree;
	    }else {
		ew = 3 - ew; //turn E to W, E to W
		c = (child_ns&2)|(ew&1);
		return tree->children[c];
	    }
	}
    }

}
