/* 
   net-if.cc
   $Id: net-if.cc,v 1.1.1.1 2001/06/20 15:37:50 benjie Exp $
   */

#include <math.h>

#include <object.h>
#include <packet.h>

#include <cmu/debug.h>
#include <cmu/channel.h>
#include <cmu/node.h>
#include <cmu/propagation.h>
#include <cmu/net-if.h>


/* ======================================================================
   NS Initialization Commands
   ====================================================================== */
static int InterfaceIndex = 0;

NetIf::NetIf()
{
	index = InterfaceIndex++;
	propagation_ = 0;
	node_ = 0;
	modulation = 0;

	Rb = 0.0;
	recvtarget_ = 0;
	channel_ = 0;
}

int
NetIf::command(int argc, const char*const* argv)
{
	if (argc == 2) {
		Tcl& tcl = Tcl::instance();

		if(strncasecmp(argv[1], "id", 2) == 0) {
			tcl.resultf("%d", index);
			return TCL_OK;
		}
		else if(strncasecmp(argv[1], "node", 4) == 0) {
			tcl.resultf("%s", node_->name());
			return TCL_OK;
		}
		else if(strcmp(argv[1], "recvtarget") == 0) {
			tcl.resultf("%s", recvtarget_->name());
			return TCL_OK;
		}
	}

	else if(argc == 3) {

		TclObject *obj;

		if( (obj = TclObject::lookup(argv[2])) == 0) {
			fprintf(stderr, "%s lookup failed\n", argv[1]);
			return TCL_ERROR;
		}

		if (strncasecmp(argv[1], "propagation", 11) == 0) {
                        assert(propagation_ == 0);
			propagation_ = (Propagation*) obj;
			return TCL_OK;
		}
		else if (strcmp(argv[1], "recvtarget") == 0) {
			recvtarget_ = (NsObject*) obj;
			return TCL_OK;
		}
		else if (strncasecmp(argv[1], "channel", 7) == 0) {
                        assert(channel_ == 0);
			channel_ = (Channel*) obj;
			// LIST_INSERT_HEAD() is done by Channel
			return TCL_OK;
		}
		else if (strncasecmp(argv[1], "node", 4) == 0) {
                        assert(node_ == 0);
			node_ = (MobileNode*) obj;
			// LIST_INSERT_HEAD() is done by Node
			return TCL_OK;
		}
	}
	return NsObject::command(argc, argv);
}


/* ======================================================================
   Other NetIf Functions
   ====================================================================== */
void
NetIf::recv(Packet* p, Handler* h)
{
  double RxPr;			// power of pkt being received

  /*
   * Sanity Check
   */
  assert(initialized());

#ifdef DEBUG
  // TEST CODE
  dump();
	fprintf(stdout, "\tPacket %d received in net intf for node: %d\n",
		HDR_CMN(p)->uid(), node()->index());
#endif

  /*
   * Handle outgoing packets
   */
  if(h) {
    /*
     * The MAC schedules its own EOT event so we just
     * ignore the handler here.  It's only purpose
     * it distinguishing between incoming and outgoing
     * packets.
     */
    xmitPacket(p);
    return;
  } else {
    if (recvPacket(p, &RxPr) == 0) {
      /*
       * XXX - This packet, even though not detected,
       * contributes to the Noise floor and hence
       * may affect the reception of other packets.
       */
      Packet::free(p);
      return;
    } else {
      recvtarget_->recv(p, (Handler*) 0);
    }
  }
}

/* NOTE: this might not be the best way to structure the relation
between the actual interfaces subclassed from net-if and net-if.  It's
fine for now, but if we were to decide to have the interfaces
themselves properly handle multiple incoming packets (they currently
require assistance from the mac layer to do this), then it's not as
generic as I'd like.  The way it is now, each interface will have to
have it's own logic to keep track of the packets that are arriving.
Seems like this is general service that net-if could provide.

Ok.  A fair amount of restructuring is going to have to happen here
when/if net-if keep track of the noise floor at their location.  I'm
gonna punt on it for now.

Actually, this may be all wrong.  Perhaps we should keep a separate 
noise floor per antenna, which would mean the particular interface types
would have to track noise floor themselves, since only they know what
kind of antenna diversity they have.  -dam 8/7/98 */


void
NetIf::dump(void) const
{
	fprintf(stdout, "\tINDEX: %d, propagation: %x, modulation: %x\n",
		index, (u_int32_t) propagation_, (u_int32_t) modulation);
	fprintf(stdout, "\trecvtarget: %x, channel: %x, node: %x Rb: %f\n",
		(u_int32_t) recvtarget_, (u_int32_t) channel_, (u_int32_t) node_, Rb);

}
