/* 
   mac-wavelan1.cc
   $Id: mac-wavelan1.cc,v 1.1.1.1 2001/06/20 15:37:50 benjie Exp $
   */
#include <delay.h>
#include <connector.h>
#include <packet.h>
#include <random.h>

//#define DEBUG
#include <cmu/debug.h>
#include <cmu/arp.h>
#include <cmu/ll.h> 
#include <cmu/mac.h>
#include <cmu/mac-wavelan1.h>


/* ======================================================================
   TCL hooks for the WaveLAN1 class.
   ====================================================================== */
static class MacWaveLAN1Class : public TclClass {
public:
	MacWaveLAN1Class() : TclClass("Mac/WaveLAN1") {}
	TclObject* create(int, const char*const*) {
		return (new Mac_WaveLAN1);
	}
} class_mac_wavelan1;


/* ======================================================================
   Defer Handler Functions
	- used to resume the upper layer
   ====================================================================== */
void
MacWL_HandlerDefer::handle(Event *)
{
	busy_ = 0;
	callback->handle((Event*) 0);
}


void
MacWL_HandlerDefer::schedule(Handler *h, double t)
{
	Scheduler& s = Scheduler::instance();
	assert(h && busy_ == 0);
	s.schedule(this, &intr, t);
	busy_ = 1;
	callback = h;
}


/* ======================================================================
   Send Handler Functions
	- called when a transmission completes without any collisions
   ====================================================================== */
void
MacWL_HandlerSend::handle(Event*)
{
	busy_ = 0;
	/*
	 *  Transmission completed successfully
	 */

#ifdef DEBUG
  	fprintf(stderr, "mac send complete Pkt %d node: %d\n",
		HDR_CMN(mac->pktTx)->uid(), mac->address());
#endif

	assert(mac->pktTx);

      	Packet::free(mac->pktTx);
	mac->pktTx = 0;
	mac->pktTxcnt = 0;

	mac->resume();
}

void
MacWL_HandlerSend::schedule(double t)
{
	Scheduler& s = Scheduler::instance();
	assert(busy_ == 0);
	s.schedule(this, &intr, t);
	busy_ = 1;
}


/* ======================================================================
   Backoff Handler Functions
   ====================================================================== */
void
MacWL_HandlerBack::handle(Event* e)
{
	busy_ = 0;
	mac->send((Packet*) e);
}

void
MacWL_HandlerBack::schedule(Packet *p, double t) {
	Scheduler& s = Scheduler::instance();
	assert(p && busy_ == 0);
	s.schedule(this, p, t);
	busy_ = 1;
}


/* ======================================================================
   Receive Handler Functions
   ====================================================================== */
void
MacWL_HandlerRecv::handle(Event* e)
{
	busy_ = 0;

	/*
	 * Reception Successful
	 */
	mac->pktRx = 0;
	mac->recv_complete((Packet*) e);
}

void
MacWL_HandlerRecv::schedule(Packet *p, double t) {
	Scheduler& s = Scheduler::instance();
	assert(p && busy_ == 0);
	s.schedule(this, p, t);
	busy_ = 1;
	p_ = p;
	endTime_ = s.clock() + t;
}

double
MacWL_HandlerRecv::expire(void) {
        Scheduler& s = Scheduler::instance();
	assert(busy_ == 1);
	return (endTime_ - s.clock());
}


/* ======================================================================
   Packet Headers Routines
   ====================================================================== */
inline u_int32_t
Mac_WaveLAN1::hdr_dst(void* hdr, u_int32_t dst)
{
	struct hdr_mac802_3 *mh = (struct hdr_mac802_3*) hdr;

	if(dst) {
		STORE4BYTE(&dst, mh->mh_da);
	}

	return ETHER_ADDR(mh->mh_da);
}

inline u_int32_t
Mac_WaveLAN1::hdr_src(void* hdr, u_int32_t src)
{
	struct hdr_mac802_3 *mh = (struct hdr_mac802_3*) hdr;

	if(src) {
		STORE4BYTE(&src, mh->mh_sa);
	}

	return ETHER_ADDR(mh->mh_sa);
}

inline int 
Mac_WaveLAN1::hdr_type(void* hdr, u_int16_t type)
{
	struct hdr_mac802_3 *mh = (struct hdr_mac802_3*) hdr;

	if(type)
		mh->mh_type = type;

	return  mh->mh_type;
}


/* ======================================================================
   Mac Class Functions
   ====================================================================== */

Mac_WaveLAN1::Mac_WaveLAN1() : Mac(), mhBack(this),mhDefer(this), mhRecv(this), mhSend(this)
{
        pktTxcnt   = 0;
	backoff_   = 0;
	collision_ = 0;
}


int
Mac_WaveLAN1::command(int argc, const char*const* argv)
{
        if (argc == 3) {
		if(strcmp(argv[1], "nodes") == 0) {
		  //		   	if(cache) return TCL_ERROR;
		  // 			cache_node_count = atoi(argv[2]);
		  //		        cache = new Host[cache_node_count + 1];
		  // 			assert(cache);
		  // 			bzero(cache, sizeof(Host) * (cache_node_count+1 ));
			return TCL_OK;
       		
		}
        }
	return Mac::command(argc, argv);
}




void 
Mac_WaveLAN1::discard(Packet *p, const char*)
{
	hdr_mac802_3* mh = (hdr_mac802_3*)p->access(off_mac_);

#ifdef DEBUG
	printf("mac %d discarding pkt %d\n",address(),HDR_CMN(p)->uid());
#endif

	if(ETHER_ADDR(mh->mh_da) == index)
		drop(p);
	else
		Packet::free(p);
}


void
Mac_WaveLAN1::collision(Packet *p)
{


#ifdef DEBUG
  	fprintf(stderr, "mac collision Pkt %d node: %d state %d backoff %d\n",
		  HDR_CMN(p)->uid(), address(), state, backoff_);
#endif

	switch(state) {

	case MAC_SEND:
		assert(pktTx);

		/*
		 *  Collisions occur at the receiver, so receiving a
		 *  packet, while transmitting may not be a problem.
		 *
		 *  If the packet is for us, then drop() the packet -
		 *  ie; log an event recording the collision.
		 *  Otherwise, just free the packet and forget about
		 *  it.
		 */
		discard(p);
		break;

	case MAC_RECV:
		assert(pktRx);
		collision_++;

		/*
		 * decide if if the new packet or the packet currently
		 * being received will terminate last
		 */

		if(netif_->txtime(p) > mhRecv.expire() ) {
		  mhRecv.cancel();
		  mhRecv.schedule(p, netif_->txtime(p) );

		/*
		 * drop the packet that is presently being received.
		 * Again, if the packet was not for us, then its not
		 * really worth logging.
		 */
		  discard(pktRx);
		  pktRx = p;
		}
		else {
		/*
		 *  If the packet is for us, then drop() the packet -
		 *  ie; log an event recording the collision.
		 *  Otherwise, just free the packet and forget about
		 *  it.
		 */
		 discard(p);
		}

       		break;
	default:
	  assert("SHOULD NEVER HAPPEN" == 0);
	}
}


void
Mac_WaveLAN1::send(Packet *p, Handler *h)
{
	hdr_cmn *hdr = HDR_CMN(p);
	hdr->size() += WAVELAN_HDR_LEN;

	upcall_ = h;
	send(p);
}

/*
 * This function gets called when packets are passed down from the
 * link layer or up from the network interface.
 */
void
Mac_WaveLAN1::recv(Packet *p, Handler *h)
{
	/*
	 * Sanity Check
	 */
	assert(initialized());

#ifdef DEBUG
	fprintf(stderr, "mac recv Pkt %d node: %d state %d h %p backoff_ %d\n",
		HDR_CMN(p)->uid(), address(), state, h, backoff_);
#endif 

	/*
	 *  Handle outgoing packets.
	 */
	if(h) {
		send(p, h);
		return;
	}

	/*
	 *  Handle incoming packets.
	 *
	 *  We just received the 1st bit of a packet on the network
	 *  interface.
	 *
	 */
	switch(state) {
	
	case MAC_IDLE:
		state = MAC_RECV;
		pktRx = p;

		/* need to set the incoming flag to 1 so that
		 * the link layer realizes this packet is incoming
		 * (see ns-src/packet.h)
		 */
		p->incoming = 1;
		/*
		 * Schedule the reception of this packet, in
		 * txtime seconds.
		 */
		mhRecv.schedule(p, netif_->txtime(p));
		break;
	case MAC_SEND:
	  /*
	   * collisions are handled on the receiving end, so when
	   * sending just keep sending until the packet is done
	   */
	  Packet::free(p);
	  break;
	default:
		/*
		 *  If the power of the incoming packet is smaller than the
		 *  power of the packet currently being received by at least
                 *  the capture threshold, then we ignore the new packet.
		 */
		if(p->txinfo.RxPr > 0){
		  if(pktRx->txinfo.RxPr / p->txinfo.RxPr >= p->txinfo.CPThresh) {
#ifdef DEBUG
		    fprintf(stderr, "mac node: %d lost Pkt: %d due to capture effects\n", address(), HDR_CMN(p)->uid());
#endif
		    Packet::free(p);
		  }
		  else {  
		    collision(p);
		  }
		} // end switch 
	}
}


void
Mac_WaveLAN1::send(Packet *p)
{
	double txtime = netif_->txtime(p);

	/*
	 *  Sanity Check
	 */
	if(pktTx) {
		drop(pktTx);
		drop(p);
		fprintf(stderr, "index: %d\n", index);
		fprintf(stderr, "Backoff Timer: %d\n", mhBack.busy());
		fprintf(stderr, "Defer Timer: %d\n", mhDefer.busy());
		fprintf(stderr, "Recv Timer: %d\n", mhRecv.busy());
		fprintf(stderr, "Send Timer: %d\n", mhSend.busy());
		exit(1);
	}
	assert(upcall_ && pktTx == 0);

	pktTx = p;

	/*
	 *  Perform carrier sense.  If the medium is not IDLE, then
	 *  backoff again.  Otherwise, go for it!
	 */

#ifdef DEBUG
	fprintf(stderr, "mac carrier sense Pkt %d node: %d state %d backoff_ %d\n",
		  HDR_CMN(p)->uid(), address(), state, backoff_);
#endif

	if(state != MAC_IDLE) {
	  //	  fprintf(stderr, "mac backoff Pkt %d node: %d state %d\n",
	  //  HDR_CMN(p)->uid(), address(), state);
       	  backoff_ = 1;
	  
	  return;
	}

	state = MAC_SEND;

	/*
	 * pass the packet on the "interface" which will in turn
	 * place the packet on the channel.
	 *
	 * NOTE: a handler is passed along so that the Network Interface
	 *	 can distinguish between incoming and outgoing packets.
	 */

#ifdef DEBUG
	fprintf(stderr, "mac send now Pkt %d node: %d state %d txtime %f backoff_ %d\n",
		  HDR_CMN(p)->uid(), address(), state, txtime, backoff_);
#endif

	sendtarget_->recv(p->copy(), this);

	/*
	 * This MAC will remain in the MAC_SEND state until packet
	 * transmission completes.  Schedule an event that will
	 * reset the MAC to the IDLE state.
	 */
	mhSend.schedule(txtime);
}


void
Mac_WaveLAN1::recv_complete(Packet *p)
{

	hdr_cmn *hdr = HDR_CMN(p);
	hdr_mac802_3* mh = (hdr_mac802_3*)p->access(off_mac_);
	
	if(collision_ >= 1){
	  discard(p);
	  collision_ = 0;
	  pktRx = 0;
	  goto done;
	}

	/*
	 * Address Filtering
	 */
	if(ETHER_ADDR(mh->mh_da) != (u_int32_t) index &&
	   ETHER_ADDR(mh->mh_da) != MAC_BROADCAST) {
		/*
		 *  We don't want to log this event, so we just free
		 *  the packet instead of calling the drop routine.
		 */

#ifdef DEBUG
	  fprintf(stderr, "mac recv complete Pkt %d node: %d for filtering out   backoff_ %d\n",HDR_CMN(p)->uid(), address(), backoff_);
#endif

		Packet::free(p);
		goto done;
	}

	/*
	 * Now, check to see if this packet was received with enough
	 * bit errors that the current level of FEC still could not
	 * fix all of the problems - ie; after FEC, the checksum still
	 * failed.
	 */
	if( hdr->error() ) {

#ifdef DEBUG
	  fprintf(stderr, "mac recv complete Pkt %d node: %d for discard   backoff_ %d\n", HDR_CMN(p)->uid(), address(), backoff_);
#endif

		discard(p);
		goto done;
	}

	/*
	 * Adjust the MAC packet size - ie; strip off the mac header
	 */

#ifdef DEBUG
	fprintf(stderr, "mac recv complete Pkt %d node: %d to LL with upcall %p   backoff_ %d\n", HDR_CMN(p)->uid(), address(), upcall_, backoff_);
#endif

	hdr->size() -= WAVELAN_HDR_LEN;

	/*
	 *  Pass the packet up to the link-layer.
	 *  XXX - we could schedule an event to account
	 *  for this processing delay.
	 */
	recvtarget_->recv(p, (Handler*) 0);

	done:
	/*
	 * check for backoff and reset the MAC to the IDLE state
	 * Since the backoff timer should start after the media
	 * becomes free, the backoff() call needs to be here...
	 */
	if (backoff_ == 1) {
	  backoff();
	}
	resume();
}


void
Mac_WaveLAN1::backoff() {
        
  // if mhBack.busy() == 1 then the backoff timer has already been
  // set for this particuar packet's Nth transmission attempt, so 
  // don't try and set it again
          if(mhBack.busy() == 0){
	  int k, r;

	  k = min( (WAVELAN_BMIN + pktTxcnt), WAVELAN_BLIMIT);
	  pktTxcnt++;
	  r = Random::integer(1 << k);

	  // a slot time delay of 1 means xmit immiediately, so 
	  // subtract 1 from r to adjust
	  r -= 1;
	  if(r < 0){
	    r = 0;      // just to make sure...
	  }
	  assert(pktTx);
	
	  if(pktTxcnt < WAVELAN_ALIMIT) {
	    mhBack.schedule(pktTx, WAVELAN_IFS + r * WAVELAN_ANTENNA_SLOT);

#ifdef DEBUG
	    fprintf(stderr, "mac backoff r = %d Pkt %d node: %d state %d timer: %f backoff_ %d\n",
		    r, HDR_CMN(pktTx)->uid(), address(), state, WAVELAN_IFS + r*WAVELAN_ANTENNA_SLOT, backoff_);
#endif

	    pktTx = 0;
	    // return;
	  }
	  else {
 
#ifdef DEBUG
	    fprintf(stderr, "mac backoff Pkt %d node: %d state %d DROP backoff_ %d\n",
		    HDR_CMN(pktTx)->uid(), address(), state, backoff_);
#endif

	    drop(pktTx);
	    pktTx = 0;
	    pktTxcnt = 0;
	  }

	  }
	// change backoff_ flag so that the backoff() function is not 
	// called again until this backoff attempt has passed
	  backoff_ = 0; 
}

void
Mac_WaveLAN1::resume()
{

#ifdef DEBUG
  	fprintf(stderr, "mac resume node: %d state %d upcall %p backoff_ %d\n",
		 address(), state, upcall_, backoff_);
#endif

	state = MAC_IDLE;

	/*
	 *  Sanity Check
	 */
	assert(pktRx == 0);
	assert(pktTx == 0);
	// assert(mhDefer.busy() == 0);
	assert(mhRecv.busy() == 0);
	assert(mhSend.busy() == 0);

	/*
	 *  If we're not backing off right now, go ahead and unblock
	 *  the upper layer so that we can get more packets to send.
	 */
	if(upcall_ && mhBack.busy() == 0) {

		mhDefer.schedule(upcall_, WAVELAN_IFS + WAVELAN_XDELAY * WAVELAN_ANTENNA_SLOT);
		upcall_ = 0;
	}
}


