#include <delay.h>
#include <connector.h>
#include <packet.h>
#include <random.h>
#include <stdlib.h>
#include <math.h>
#include <time.h>

// #define DEBUG 1
// #define COLLISION_DEBUG 1
// #define BENJIE_DEBUG 1
// #define TXF_DEBUG 1

#include <cmu/debug.h>
#include <cmu/arp.h>
#include <cmu/ll.h>
#include <cmu/mac.h>
#include <cmu/mac-timers.h>
#include <cmu/mac-802_11-mlme.h>
#include <cmu/cmu-trace.h>
#include <cmu/tsftimer.h>
#include <cmu/node.h>
#include <cmu/energy-model.h>
#include <cmu/sharedmedia.h>
#if USE_SPAN
#include <cmu/span/hdr_span.h>
#endif

/* ======================================================================
   Macros
   ====================================================================== */
#define max(x,y) ((x > y) ? (x) : (y))

#ifdef DEBUG
#define CHECK_BACKOFF_TIMER()						\
{									\
	if(is_idle() && mhBackoff.paused()) {				\
                fprintf(stderr, "%f: %d resumed backoff timer\n",       \
		       Scheduler::instance().clock(), index);           \
		mhBackoff.resume(difs);					\
        }                                                               \
	if(! is_idle() && mhBackoff.busy() && ! mhBackoff.paused()) {	\
                fprintf(stderr, "%f: %d paused backoff timer\n",        \
                       Scheduler::instance().clock(), index);           \
		mhBackoff.pause();					\
        }                                                               \
}
#else
#define CHECK_BACKOFF_TIMER()						\
{									\
	if(is_idle() && mhBackoff.paused()) {				\
		mhBackoff.resume(difs);					\
        }                                                               \
	if(! is_idle() && mhBackoff.busy() && ! mhBackoff.paused()) {	\
		mhBackoff.pause();					\
        }                                                               \
}
#endif

#define SET_RX_STATE(x)			\
{					\
	rx_state = (x);			\
					\
	CHECK_BACKOFF_TIMER();		\
}

#define SET_TX_STATE(x)				\
{						\
	tx_state = (x);				\
						\
	CHECK_BACKOFF_TIMER();			\
}

/* ======================================================================
   Global Variables
   ====================================================================== */
extern char *pt_names[];

static PHY_MIB PMIB = {
  DSSS_CWMin, DSSS_CWMax, DSSS_SlotTime, DSSS_CCATime,
  DSSS_RxTxTurnaroundTime, DSSS_SIFSTime, DSSS_PreambleLength,
  DSSS_PLCPHeaderLength
};

static MAC_MIB MMIB = {
  // XXXXXXXXXXXX: minor todo turn on the rts threshold when not
  // debugging
  0 /* MAC_RTSThreshold */ , MAC_ShortRetryLimit,
  MAC_LongRetryLimit, MAC_FragmentationThreshold,
  MAC_MaxTransmitMSDULifetime, MAC_MaxReceiveLifetime,
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, MAC_BeaconPeriod,
  MAC_ATIMWindow, MAC_TBTTPreWake
};

// Packet Formats

int
 hdr_beacon::offset_;

static class BeaconHeaderClass:public PacketHeaderClass { public:
  BeaconHeaderClass():PacketHeaderClass("PacketHeader/Beacon",
					sizeof(struct hdr_beacon)) {
    offset(&hdr_beacon::offset_);
}} class_beaconhdr;

/* ======================================================================
   TCL Hooks for the simulator
   ====================================================================== */
static class Mac802_11Class:public TclClass { public:
  Mac802_11Class():TclClass("Mac/802_11") {
    srand((int) time(0));
    srandom((int) time(0));
  } TclObject *create(int, const char *const *) {
    return (new Mac802_11(&PMIB, &MMIB));
}} class_mac802_11;


/* ======================================================================
   Mac Class Functions
   ====================================================================== */
Mac802_11::Mac802_11(PHY_MIB * p, MAC_MIB * m)
  : Mac(), initialized_(0), mhIF(this), mhNav(this), 
    mhRecv(this), mhSend(this), mhDefer(this, p->SlotTime), 
    mhBackoff(this, p->SlotTime), mhTBTT(this), 
    mhBeacon(this), mhAdvEnd(this), mhEndATIM(this), tsfTimer(index)
{
  bind("off_beacon_", &off_beacon_);
#if USE_SPAN
  bind("off_span_", &off_span_);
#endif
  macmib = m;
  phymib = p;

  nav = 0.0;

  tx_state = rx_state = MAC_IDLE;
  tx_active = 0;

  pktRTS = 0;
  pktCTRL = 0;
  pktBEACON = 0;
  pktATIM = 0;

  cw = phymib->CWMin;
  ssrc = slrc = atimrc = rtsrc = 0;

  sta_seqno = 1;
  cache = 0;
  cache_node_count = 0;

  initial_wake_count_ = 10;
  got_beacon_ = false;
  psm_mode_ = false;

  CIRCLEQ_INIT(&buf_head_);
  upcall_ = 0;

  /* jinyang */
  total_retransmit = 0;
  total_wasted_time = 0;
  idle_start = 0.0;
  total_true_idle_time = 0.0;
}


Mac802_11::~Mac802_11()
{
}

void
 Mac802_11::init()
{
  Mac::init();

  sifs = phymib->SIFSTime;
  pifs = sifs + phymib->SlotTime;
  difs = sifs + 2 * phymib->SlotTime;
  eifs = sifs + difs + DATA_Time(ETHER_ACK_LEN +
				 phymib->PreambleLength / 8 +
				 phymib->PLCPHeaderLength / 8);
  tx_sifs = sifs - phymib->RxTxTurnaroundTime;
  tx_pifs = tx_sifs + phymib->SlotTime;
  tx_difs = tx_sifs + 2 * phymib->SlotTime;
  initialized_ = 1;
}
	
void Mac802_11::Terminate()
{
  struct buffer_entry *this_entry;
  CIRCLEQ_FOREACH(this_entry, &buf_head_, entries) {
    CIRCLEQ_REMOVE(&buf_head_, this_entry, entries);
    Packet *p = this_entry->packet_;
    drop(p, DROP_END_OF_SIMULATION);
    delete this_entry;
  }
}

int Mac802_11::command(int argc, const char *const *argv)
{
  if (argc == 2) {
    if (strcasecmp(argv[1], "reset") == 0) {
      Terminate();
      return TCL_OK;
    }
    if (strcasecmp(argv[1], "initialize") == 0) {
      init();
      return TCL_OK;
    }
    if (strcasecmp(argv[1], "start") == 0) {
      // start generating beacons
      psm_mode_ = true;
      mhTBTT.start(2 * macmib->BeaconPeriod -
		   fmod(tsfTimer.getTSF(),
			macmib->BeaconPeriod * 1e6) / 1e6, "initial tbtt");
      return TCL_OK;
    }
  } else if (argc == 3) {

    if (strcmp(argv[1], "log-target") == 0) {
      logtarget_ = (NsObject *) TclObject::lookup(argv[2]);
      if (logtarget_ == 0)
	return TCL_ERROR;
      return TCL_OK;
    }

    if (strcmp(argv[1], "nodes") == 0) {
      if (cache)
	return TCL_ERROR;
      cache_node_count = atoi(argv[2]);
      cache = new Host[cache_node_count + 1];
      assert(cache);
      bzero(cache, sizeof(Host) * (cache_node_count + 1));
      return TCL_OK;
    }

    if (strcmp(argv[1], "setnode") == 0) {
      node_ = (MobileNode *) TclObject::lookup(argv[2]);
      if (!node_)
	return TCL_ERROR;
      return TCL_OK;
    }
  }
  return Mac::command(argc, argv);
}


/* ======================================================================
   Debugging Routines
   ====================================================================== */
double Mac802_11::bftime()
{
  return mhBackoff.total_bftime;
}

void Mac802_11::trace_pkt(Packet * p)
{
  struct hdr_cmn *ch = HDR_CMN(p);
  struct hdr_mac802_11 *dh = HDR_MAC11(p);

  fprintf(stderr, "[%d<-%d] %d %s %d\n",
	  ETHER_ADDR(dh->dh_da), ETHER_ADDR(dh->dh_sa),
	  ch->uid(), pt_names[ch->ptype()], ch->size());
}

void Mac802_11::dump(char *fname)
{
  fprintf(stderr,
	  "\n%s --- (INDEX: %d, time: %2.9f)\n",
	  fname, index, Scheduler::instance().clock());

  fprintf(stderr,
	  "\ttx_state: %x, rx_state: %x, nav: %2.9f, idle: %d\n",
	  tx_state, rx_state, nav, is_idle());

  fprintf(stderr,
	  "\tpktTx: %x, pktRx: %x, pktRTS: %x, pktCTRL: %x, upcall: %x\n",
	  (int) pktTx, (int) pktRx, (int) pktRTS,
	  (int) pktCTRL, (int) upcall_);

  fprintf(stderr,
	  "\tDefer: %d, Backoff: %d (%d), Recv: %d, Timer: %d Nav: %d\n",
	  mhDefer.busy(), mhBackoff.busy(), mhBackoff.paused(),
	  mhRecv.busy(), mhSend.busy(), mhNav.busy());
  fprintf(stderr, "\tBackoff Expire: %f\n", mhBackoff.expire());
}


/* ======================================================================
   Packet Headers Routines
   ====================================================================== */
inline MacAddr Mac802_11::hdr_dst(void *hdr, MacAddr dst)
{
  struct hdr_mac802_11 *dh = (struct hdr_mac802_11 *) hdr;

  if (dst)
    STORE4BYTE(&dst, (dh->dh_da));

  return ETHER_ADDR(dh->dh_da);
}

inline MacAddr Mac802_11::hdr_src(void *hdr, MacAddr src)
{
  struct hdr_mac802_11 *dh = (struct hdr_mac802_11 *) hdr;

  if (src)
    STORE4BYTE(&src, dh->dh_sa);

  return ETHER_ADDR(dh->dh_sa);
}

inline int Mac802_11::hdr_type(void *hdr, u_int16_t type)
{
  struct hdr_mac802_11 *dh = (struct hdr_mac802_11 *) hdr;

  if (type)
    SET_ETHER_TYPE(dh->dh_body, type);

  return GET_ETHER_TYPE(dh->dh_body);
}


/* ======================================================================
   Misc Routines
   ====================================================================== */

Packet * Mac802_11::reclaim_packet(int dst)
{ 
  struct buffer_entry *this_entry;
  CIRCLEQ_FOREACH(this_entry, &buf_head_, entries) { 
    if (this_entry->packet_ && this_entry->addr(this) == (unsigned)dst) {
      CIRCLEQ_REMOVE(&buf_head_, this_entry, entries); 
      Packet *p = this_entry->packet_;
      delete this_entry;
      return p;
    }
  }
  return 0;
}

inline void Mac802_11::TRANSMIT(Packet * p, double t)
{
  struct hdr_mac802_11 *dh = HDR_MAC11(p);
#ifdef DEBUG
  fprintf(stderr, "%f: %d TRANSMIT\n",
	  Scheduler::instance().clock(), index);
#endif

  struct hdr_beacon *bh;
  tx_active = 1;

  /*                                                              
   * If I'm transmitting without doing CS, such as when           
   * sending an ACK, any incoming packet will be "missed"         
   * and hence, must be discarded.                                
   */
  if (rx_state != MAC_IDLE) {
    if ((dh->dh_fc.fc_type != MAC_Type_Control) &&
	(dh->dh_fc.fc_type != MAC_Type_Management)) assert(0);
    if ((dh->dh_fc.fc_subtype != MAC_Subtype_ACK) &&
	(dh->dh_fc.fc_type == MAC_Type_Control)) assert(0);

    assert(pktRx);
    struct hdr_cmn *ch = HDR_CMN(pktRx);
#ifdef DEBUG
    fprintf(stderr, "%f: %d TRANSMIT forced packet discard\n",
	    Scheduler::instance().clock(), index);
#endif
    ch->error() = 1;		/* force packet discard */
  }

  /*                                                              
   * pass the packet on the "interface" which will in turn        
   * place the packet on the channel.                             
   *                                                              
   * NOTE: a handler is passed along so that the Network         
   *       Interface can distinguish between incoming and         
   *       outgoing packets.                                      
   */
  if ((dh->dh_fc.fc_type == MAC_Type_Management) &&
      (dh->dh_fc.fc_subtype == MAC_Subtype_Beacon)) {
    // if we are dealing with a beacon we need to fill in the tsf
    // just before the first bit enters the PHY (c.f. 11.1.2)
    bh = HDR_BEACON(p);
    bh->bh_bb.bb_ts = tsfTimer.getTSF();
  }
  if (((SharedMedia *) netif_)->state() == pwr_state_sleep) {
#ifdef BENJIE_DEBUG
    fprintf(stderr, "%f: _%d_ try to send, turn on radio\n",
	    Scheduler::instance().clock(), index);
#endif
    ((SharedMedia *) netif_)->set_state(pwr_state_idle);
  }
  sendtarget_->recv(p->copy(), this);
  assert(!mhSend.busy());
  assert(!mhIF.busy());
  mhSend.start(t, "sendtimer");
  mhIF.start(TX_Time(p), "xmittimer");
}

inline int Mac802_11::is_idle()
{
  if (rx_state != MAC_IDLE)
    return 0;

  if (tx_state != MAC_IDLE)
    return 0;

  if (nav > Scheduler::instance().clock())
    return 0;

  return 1;
}


void Mac802_11::discard(Packet * p, const char *why)
{
  hdr_mac802_11 *mh = HDR_MAC11(p);
  hdr_cmn *ch = HDR_CMN(p);

  /* if the rcvd pkt contains errors, a real MAC layer couldn't
     necessarily read any data from it, so we just toss it now */
  if (ch->error() != 0) {
    Packet::free(p);
    return;
  }

  switch (mh->dh_fc.fc_type) {

  case MAC_Type_Management:
    if (ETHER_ADDR(mh->dh_da) == index ||
	ETHER_ADDR(mh->dh_sa) == index ||
	ETHER_ADDR(mh->dh_da) == MAC_BROADCAST) {
      drop(p, why);
      return;
    }
    break;

  case MAC_Type_Control:
    switch (mh->dh_fc.fc_subtype) {

    case MAC_Subtype_RTS:
      if (ETHER_ADDR(mh->dh_sa) == index) {
	drop(p, why);
	return;
      }

      /* fall through - if necessary */

    case MAC_Subtype_CTS:
    case MAC_Subtype_ACK:
      if (ETHER_ADDR(mh->dh_da) == index) {
	drop(p, why);
	return;
      }
      break;

    default:
      fprintf(stderr, "invalid MAC Control subtype\n");
      exit(1);
    }
    break;

  case MAC_Type_Data:
    switch (mh->dh_fc.fc_subtype) {

    case MAC_Subtype_Data:
      if (ETHER_ADDR(mh->dh_da) == index ||
	  ETHER_ADDR(mh->dh_sa) == index ||
	  ETHER_ADDR(mh->dh_da) == MAC_BROADCAST) {
	drop(p, why);
	return;
      }
      break;

    default:
      fprintf(stderr, "invalid MAC Data subtype\n");
      exit(1);
    }
    break;

  default:
    fprintf(stderr, "invalid MAC type (%x): ", mh->dh_fc.fc_type);
    trace_pkt(p);
    exit(1);
  }
  Packet::free(p);
}


void Mac802_11::capture(Packet * p)
{
  /*
   * Update the NAV so that this does not screw
   * up carrier sense.
   */
  set_nav(usec(eifs + TX_Time(p)));

  Packet::free(p);
}


void Mac802_11::collision(Packet * p)
{
#ifdef COLLISION_DEBUG
  hdr_mac802_11 *mh_rx = HDR_MAC11(pktRx);
  hdr_mac802_11 *mh_p = HDR_MAC11(p);
  u_int32_t dst_rx = ETHER_ADDR(mh_rx->dh_da);
  u_int32_t dst_p = ETHER_ADDR(mh_p->dh_da);

  fprintf(stderr, "%d: collision, pktRx has power %e, p has %e, thresh %e\n",
          index, pktRx->txinfo.RxPr, p->txinfo.RxPr, p->txinfo.CPThresh);
#endif

  switch (rx_state) {

  case MAC_RECV:
    SET_RX_STATE(MAC_COLL);

    /* fall through */

  case MAC_COLL:
    assert(pktRx);
    assert(mhRecv.busy());

    /*
     *  Since a collision has occurred, figure out
     *  which packet that caused the collision will
     *  "last" the longest.  Make this packet,
     *  pktRx and reset the Recv Timer if necessary.
     */
    if (TX_Time(p) > mhRecv.expire()) {
      mhRecv.stop();
#ifdef COLLISION_DEBUG
      if (dst_rx == index || dst_p == index) {
	fprintf(stderr, "%f: %d collision, dropping pktRx: ",
		Scheduler::instance().clock(), index);
	trace_pkt(pktRx);
	fprintf(stderr, "%f: %d new pktRx is: ",
		Scheduler::instance().clock(), index);
	trace_pkt(p);
      }
#endif
      discard(pktRx, DROP_MAC_COLLISION);
      pktRx = p;
      assert(!mhRecv.busy());
      mhRecv.start(TX_Time(pktRx), "recvtimer");
    } else {
#ifdef COLLISION_DEBUG
      if (dst_rx == index || dst_p == index) {
	fprintf(stderr, "%f: %d collision, dropping p: ",
		Scheduler::instance().clock(), index);
	trace_pkt(p);
	fprintf(stderr, "%f: %d collision, pktRx is: ",
		Scheduler::instance().clock(), index);
	trace_pkt(pktRx);
      }
#endif
      discard(p, DROP_MAC_COLLISION);
    }
    break;

  default:
    assert(0);
  }
}

void Mac802_11::upcall()
{
  if (upcall_) {
#ifdef DEBUG
    fprintf(stderr, "%f: %d upcall\n",
	    Scheduler::instance().clock(), index);
#endif
    Handler *up = upcall_;
    upcall_ = 0;
    up->handle((Event *) 0);
  }
}

void Mac802_11::goto_sleep()
{ 
  struct buffer_entry *this_entry; 
  int n = 0;
  CIRCLEQ_FOREACH(this_entry, &buf_head_, entries) {
    n++;
  }
  if (n < 5) 
    ((SharedMedia *) netif_)->set_state(pwr_state_sleep);
  else {
#if BENJIE_DEBUG
    fprintf(stderr, "%f: _%d_ radio off denied, too many packets to send...\n",
	    Scheduler::instance().clock(), index);
#endif
  }
}

void Mac802_11::tx_resume()
{
  assert(!mhSend.busy());

  if (mhEndATIM.busy() && got_beacon_) {	// inside ATIM window
    bool active = false;

    if (pktCTRL) { 
      if (mhDefer.busy()==0)
	  mhDefer.start(sifs, "sifs defer for cts/ack");
      active = true;
    }
    if (!active) {
      atim_scan();
      if (pktBEACON || pktATIM) {
	if (mhBackoff.busy() == 0) {
	  if (mhDefer.busy() == 0)
	    mhDefer.start(difs);
	}
      } else
	upcall();
    }

  } else if (!mhEndATIM.busy()) {	// outside ATIM window 
    bool active = false;

    if (pktCTRL) { 
      if (mhDefer.busy()==0) 
	mhDefer.start(sifs, "sifs defer for cts/ack");
      active = true;
    }

    else if (pktRTS) {
      if (mhBackoff.busy() == 0)
        if (mhDefer.busy()==0)
	  mhDefer.start(difs);
      active = true;
    }

    else if (pktTx) {
      if (mhBackoff.busy() == 0) {
        if (mhDefer.busy()==0)
  	  mhDefer.start(difs);
      }
      active = true;
    }

    if (!active) {
      tx_scan();

      if (pktRTS || pktTx) {
	if (mhBackoff.busy() == 0) {
	  if (is_idle()) {
	    /* If we are already deferring, there is no need to 
	     * reset the Defer timer. */
	    if (mhDefer.busy() == 0)
	      mhDefer.start(difs);
	  }
	  /* If the medium is NOT IDLE, then we start 
	   * the backoff timer. */
	  else
	    mhBackoff.start(cw, 0, is_idle());
	}
      } else {
	has_packet_tx_ = 0;

	// no data to send nor receive, so can go to sleep if needs to
	if (!initial_wake_count_ &&
	    ((!recvd_bcast_atim_ && !recvd_ucast_atim_)
	     || !mhAdvEnd.busy()) && ((SharedMedia *) netif_)->PowerSave()) {
#ifdef BENJIE_DEBUG
	  fprintf(stderr, "%f: _%d_ done tx, nothing to rx, turn radio off?\n",
		  Scheduler::instance().clock(), index);
#endif
	  goto_sleep();
	} else {
	  upcall();
#if BENJIE_DEBUG > 1
	  fprintf(stderr,
		  "%f: _%d_ done tx, stay on, b %d, u %d, w %d\n",
		  Scheduler::instance().clock(), index,
		  recvd_bcast_atim_, recvd_ucast_atim_,
		  initial_wake_count_);
#endif
	}
      }
    }
  } else {
    // inside atim window, no beacon heard, but has a 
    // control packet, need to send it.
    if (pktCTRL) { 
      if (mhDefer.busy()==0) 
	mhDefer.start(sifs, "sifs defer for cts/ack");
    }
  }
  SET_TX_STATE(MAC_IDLE);
}


void Mac802_11::rx_resume()
{
  assert(pktRx == 0);
  assert(mhRecv.busy() == 0);
  SET_RX_STATE(MAC_IDLE);

#if ADVWINDOW
  if (((SharedMedia *) netif_)->PowerSave() &&
      ((SharedMedia *) netif_)->state() == pwr_state_idle &&
      !mhAdvEnd.busy() && !pktTx && !pktRTS && rx_state == MAC_IDLE &&
      tx_state == MAC_IDLE && !initial_wake_count_) {
#ifdef BENJIE_DEBUG
    fprintf(stderr, "%f: _%d_ end of adv win, turn off radio?\n",
	    Scheduler::instance().clock(), index);
#endif
    goto_sleep();
  }
#endif
}

bool Mac802_11::tx_scan()
{
  if (pktTx)
    return false;
  bool remove = false;
  struct buffer_entry *this_entry = 0;

  // try to find packet with an ACK, or broadcast packets
  if (!remove && mhAdvEnd.busy()) {
    CIRCLEQ_FOREACH(this_entry, &buf_head_, entries) {
      assert(this_entry);
      u_int32_t dst = ETHER_ADDR(HDR_MAC11(this_entry->packet_)->dh_da);

      // if dst is not in pwr mgt mode, or if unicast pkt w/o ATIM ACK, or
      // if un-advertised broadcast traffic, skip
      if (!this_entry->packet_ || 
#if FASTROUTE
	  !this_entry->pwr_mgt_ ||
#endif
	  (!this_entry->recvd_ack_ && dst != MAC_BROADCAST) ||
	  (dst == MAC_BROADCAST && !this_entry->sent_atim_))
	continue;

      this_entry->sent_data_ = true;
      sendDATA(this_entry->packet_);
      sendRTS(dst);

      CIRCLEQ_REMOVE(&buf_head_, this_entry, entries);
#if BENJIE_DEBUG > 1
      fprintf(stderr, "%f: %d tx_scan sending advertised packet\n",
	      Scheduler::instance().clock(), index);
#endif
      remove = true;
      break;
    }
  }
#if FASTROUTE
  // try to find an un-advertised packet
  if (!remove) {
    CIRCLEQ_FOREACH(this_entry, &buf_head_, entries) {
      assert(this_entry);
      u_int32_t dst = ETHER_ADDR(HDR_MAC11(this_entry->packet_)->dh_da);

      // if unicast dst in pwr mgt mode but did not receive ack, or if
      // un-advertised broadcast traffic, skip
      if (!this_entry->packet_ ||
	  (this_entry->pwr_mgt_ && !this_entry->recvd_ack_
	   && dst != MAC_BROADCAST) ||
	  (dst == MAC_BROADCAST && !this_entry->sent_atim_)) continue;

      // if dst is in power saving mode and advertised traffic window is
      // over, skip packet until next round.
      if (!mhAdvEnd.busy() && this_entry->pwr_mgt_)
	continue;

      this_entry->sent_data_ = true;
      sendDATA(this_entry->packet_);
      sendRTS(dst);

      CIRCLEQ_REMOVE(&buf_head_, this_entry, entries);
#if BENJIE_DEBUG > 1
      fprintf(stderr, "%f: %d tx_scan sending un-advertised packet\n",
	      Scheduler::instance().clock(), index);
#endif
      remove = true;
      break;
    }
  }
#endif
  if (remove) {
    delete this_entry;
    return true;
  }
  return false;

}

void Mac802_11::atim_scan()
{
  // scan for an unsent ATIM in the buffer and send it.
  struct buffer_entry *this_entry;
  if (!pktATIM) {
    CIRCLEQ_FOREACH(this_entry, &buf_head_, entries) {
      if (!this_entry->sent_atim_ && !this_entry->recvd_ack_
#if LIMITATIM
	  && this_entry->pwr_mgt_
#endif
         ) 
      {
	pktATIM = this_entry->packet_->copy();
	entryATIM = this_entry;

	// mac 802.11 header
	struct hdr_mac802_11 *dh = HDR_MAC11(pktATIM);
	dh->dh_fc.fc_type = MAC_Type_Management;
	dh->dh_fc.fc_subtype = MAC_Subtype_ATIM;
	dh->dh_fc.fc_pwr_mgt =
	    ((SharedMedia *) netif_)->PowerSave()? 1 : 0;
        
	if (ETHER_ADDR(dh->dh_da) != MAC_BROADCAST) 
	  dh->dh_duration = DATA_DURATION(); 
	else 
	  dh->dh_duration = 0;

	// common header
	struct hdr_cmn *ch = HDR_CMN(pktATIM);
	ch->uid() = 0;
	ch->ptype() = PT_MAC;
	ch->iface() = -2;
	ch->error() = 0;
	ch->size() = ETHER_HDR_LEN11;
	break;
      }
    }
  }
}

/* ======================================================================
   Timer Handler Routines
   ====================================================================== */
void Mac802_11::backoffHandler()
{
  if (mhEndATIM.busy()) {
    if (pktCTRL) {
      assert(mhSend.busy() || mhDefer.busy()); 
      return;
    }
    if (!got_beacon_) {
      if (check_pktBEACON() == 0)
	return;
      return;
    }
    // we are inside the atim window 
    atim_scan();
    if (check_pktATIM() == 0) {
      entryATIM->sent_atim_ = true;
      return;
    }
  } else {
    // outside the atim window 
    if (pktCTRL) {
      assert(mhSend.busy() || mhDefer.busy()); 
      return;
    }

    if (check_pktRTS() == 0)
      return;
    if (check_pktTx() == 0)
      return;

    tx_scan();

    if (check_pktRTS() == 0)
      return;
    if (check_pktTx() == 0)
      return;
    else
      has_packet_tx_ = 0;
  }
}

void Mac802_11::deferHandler()
{
  if (mhEndATIM.busy()) {	// we are inside the atim window 
    if (check_pktCTRL() == 0)
      return;

    if (got_beacon_) {
      atim_scan();
      if (check_pktATIM() == 0) {
	entryATIM->sent_atim_ = true;
	return;
      }
    }

  } else {			// outside the ATIM window

    if (check_pktCTRL() == 0)
      return;

    if (mhBackoff.busy())
      mhBackoff.stop();

    if (check_pktRTS() == 0)
      return;

    else if (check_pktTx() == 0)
      return;

    tx_resume();
  }
}

void Mac802_11::navHandler()
{
  if (is_idle() && mhBackoff.paused())
    mhBackoff.resume(difs);
}

void Mac802_11::recvHandler()
{
  recv_timer();
}

void Mac802_11::sendHandler()
{
  send_timer();
}


void Mac802_11::txHandler()
{
  tx_active = 0;
}

void Mac802_11::tbttHandler()
{
  // schedule end of advertised traffic window
  if (mhAdvEnd.busy())
    mhAdvEnd.stop();
#if ADVWINDOW
  mhAdvEnd.start(MAC_AdvWindow);
#else
  mhAdvEnd.start(MAC_BeaconPeriod);
#endif

  // schedule the end of the ATIM window
  if (mhEndATIM.busy())
    mhEndATIM.stop();
  mhEndATIM.start(macmib->ATIMWindow);

#if BENJIE_DEBUG
  if (((SharedMedia *) netif_)->state() != pwr_state_sleep) {
    fprintf(stderr, "xx %f: _%d_ atim win, was %c\n",
	    Scheduler::instance().clock(), index,
	    ((SharedMedia *) netif_)->state() ==
	    pwr_state_sleep ? '~' : '^');
  }
#endif
  // immediately wake up if not already awake
  if (((SharedMedia *) netif_)->state() == pwr_state_sleep) {
#if BENJIE_DEBUG
    fprintf(stderr, "xx %f: _%d_ atim win, turning radio on\n",
	    Scheduler::instance().clock(), index);
#endif
    ((SharedMedia *) netif_)->set_state(pwr_state_idle);
  }
  // our next TBTT is exactly one aBeaconPeriod away
  double tbtt = macmib->BeaconPeriod -
      fmod(tsfTimer.getTSF(), macmib->BeaconPeriod * 1e6) / 1e6;
  if (tbtt < macmib->BeaconPeriod / 3)
    tbtt += macmib->BeaconPeriod;
  assert(!mhTBTT.busy());
  mhTBTT.start(tbtt, "tbtt (tbttHandler)");
  assert(!mhBeacon.busy());
  mhBeacon.start(0.0001, "beacontimer (tbttHandler)");

  got_beacon_ = false;
  first_atim_ = true;
  recvd_atim_ = false;
  has_packet_tx_ = 0;
  recvd_bcast_atim_ = 0;
  recvd_ucast_atim_ = 0;

  // bool found = false;
  struct buffer_entry *this_entry;

  // expire "old" atims
  CIRCLEQ_HEAD(buffer_head, buffer_entry) expired;
  CIRCLEQ_INIT(&expired);

  CIRCLEQ_FOREACH(this_entry, &buf_head_, entries) {
    if ((this_entry->age_ >= 2) || this_entry->sent_data_) {
      CIRCLEQ_REMOVE(&buf_head_, this_entry, entries);
      CIRCLEQ_INSERT_TAIL(&expired, this_entry, entries);
    }
    else {
      assert(this_entry);
#if BENJIE_DEBUG > 1
      if (this_entry->sent_atim_) {
        unsigned dst = this_entry->addr(this);
        fprintf(stderr, "%f: %d recycle entry: pwr %d, ack %d, to %d\n",
		Scheduler::instance().clock(), index,
		this_entry->pwr_mgt_, this_entry->recvd_ack_, dst);
      }
#endif
      // retry ATIM transmission at the beginning of the ATIM window, but
      // don't reset pwr_mgt_, so we don't send extra ATIMs.
      this_entry->sent_atim_ = this_entry->recvd_ack_ = false;
      this_entry->pwr_mgt_updated_ = false;
      this_entry->age_++;
    }
  }

  CIRCLEQ_FOREACH(this_entry, &expired, entries) { 
    if (!this_entry->sent_data_) { 
      /* tell the callback the send operation failed 
       * before discarding the packet */ 
      hdr_cmn *ch = HDR_CMN(this_entry->packet_); 
      if (ch->xmit_failure_) { 
	ch->xmit_reason_ = XMIT_REASON_BUF; 
	ch->xmit_failure_ 
	  (this_entry->packet_->copy(), ch->xmit_failure_data_); 
      } 
      discard(this_entry->packet_, DROP_MAC_BUFFER_FULL); 
    } 
    delete this_entry;
  }
}

void Mac802_11::beaconHandler()
{
  sendBEACON();
  if (tx_state == MAC_IDLE && !mhSend.busy()) {
    if (!mhBackoff.busy() && !mhDefer.busy()) 
      mhBackoff.start(DSSS_CWMin * 2, 0.0, is_idle(), "beacon");
  }
}

void Mac802_11::advEndHandler()
{
#if ADVWINDOW
  if (((SharedMedia *) netif_)->PowerSave() &&
      ((SharedMedia *) netif_)->state() == pwr_state_idle &&
      !pktTx && !pktRTS && rx_state == MAC_IDLE &&
      tx_state == MAC_IDLE && !initial_wake_count_) {
#ifdef BENJIE_DEBUG
    fprintf(stderr, "%f: _%d_ end of adv win, turn off radio?\n",
	    Scheduler::instance().clock(), index);
#endif
    goto_sleep();
  }
#endif
}

void Mac802_11::atimEndHandler()
{
  double atimPendingCorrection = 0.0;

  assert(tx_state != MAC_CTS);

  if (tx_state == MAC_ATIM) {
    // if we are currently transmitting an ATIM when the ATIM
    // window ends, add a correction to the backoff timer.
    atimPendingCorrection = max(mhSend.expire(), mhIF.expire()) + 0.000001;
  }
  // end of the ATIM window -- search for acked buffered packets in
  // the buffer and send data after a backoff...

  if (tx_scan() || pktTx) {
    // we have data to send
    if (!mhBackoff.busy()) {
      mhBackoff.start(cw, atimPendingCorrection, 1,
		      "first data backoff, atim end handler");
    }
#if BENJIE_DEBUG > 1
    fprintf(stderr, "%f: _%d_ has pkt to send, tx %d, b %d, u %d\n",
	    Scheduler::instance().clock(), index,
	    has_packet_tx_, recvd_bcast_atim_, recvd_ucast_atim_);
#endif
  } 
  else if ((!recvd_atim_ && !initial_wake_count_ && !has_packet_tx_) && 
           ((SharedMedia *) netif_)->PowerSave()) {
#ifdef BENJIE_DEBUG
    fprintf(stderr, "%f: _%d_ no atim, turn off radio?\n",
	    Scheduler::instance().clock(), index);
#endif
    goto_sleep();
  } else if (((SharedMedia *) netif_)->PowerSave()) {
#if BENJIE_DEBUG > 1
    fprintf(stderr, "%f: _%d_ end of atim, stay on, tx %d, b %d, u %d\n",
	    Scheduler::instance().clock(), index,
	    has_packet_tx_, recvd_bcast_atim_, recvd_ucast_atim_);
#endif
  }

  if (initial_wake_count_)
    initial_wake_count_--;
}



/* ======================================================================
   The "real" Timer Handler Routines
   ====================================================================== */
void Mac802_11::send_timer()
{
  switch (tx_state) {
    /*
     * Sent a RTS, but did not receive a CTS.
     */
  case MAC_RTS:
    RetransmitRTS();
    break;

    /*
     * Sent a CTS, but did not receive a DATA packet.
     */
  case MAC_CTS:
    assert(pktCTRL);
    Packet::free(pktCTRL);
    pktCTRL = 0;

    break;

    /*
     * Sent DATA, but did not receive an ACK packet.
     */
  case MAC_SEND:
    RetransmitDATA();

    break;

    /*
     * Sent an ACK, and now ready to resume transmission.
     */
  case MAC_ACK:
    assert(pktCTRL);
    Packet::free(pktCTRL);
    pktCTRL = 0;
    break;

    /*
     * Sent a Beacon, and now ready to resume transmission.
     */
  case MAC_BEACON:
    if (pktBEACON) {
      Packet::free(pktBEACON);
      pktBEACON = 0;
    }
    break;

    /* 
     * Sent an ATIM, but did not receive an ACK packet.  Could
     * have been a multicast ATIM.  RetransmitATIM checks this
     * case.
     */
  case MAC_ATIM:
    RetransmitATIM();
    break;

  case MAC_IDLE:
    break;

  default:
    assert(0);

  }

  tx_resume();
}


/* ======================================================================
   Outgoing Packet Routines
   ====================================================================== */
int Mac802_11::check_pktCTRL()
{
  struct hdr_mac802_11 *mh;
  double timeout;

  if (pktCTRL == 0)
    return -1;
  
  if (tx_state == MAC_CTS || tx_state == MAC_ACK) {
    return -1;
  }

  mh = HDR_MAC11(pktCTRL);

  switch (mh->dh_fc.fc_subtype) {

    /*
     *  If the medium is not IDLE, don't send the CTS.
     */
  case MAC_Subtype_CTS:
    if (!is_idle()) {
      discard(pktCTRL, DROP_MAC_BUSY);
      pktCTRL = 0;
      return 0;
    }

    SET_TX_STATE(MAC_CTS);
    timeout = (mh->dh_duration * 1e-6) + CTS_Time;	// XXX

    break;

    /*
     * IEEE 802.11 specs, section 9.2.8
     * Acknowledments are sent after an SIFS, without regard to
     * the busy/idle state of the medium.
     */
  case MAC_Subtype_ACK:
    SET_TX_STATE(MAC_ACK);
    timeout = ACK_Time;

    break;

  default:
    fprintf(stderr, "Invalid MAC Control subtype\n");
    exit(1);
  }

  TRANSMIT(pktCTRL, timeout);

  return 0;
}


int Mac802_11::check_pktRTS()
{
  struct hdr_mac802_11 *mh;
  double timeout;

  assert(mhBackoff.busy() == 0);

  if (pktRTS == 0)
    return -1;

  mh = HDR_MAC11(pktRTS);

  switch (mh->dh_fc.fc_subtype) {

  case MAC_Subtype_RTS:
    if (!is_idle()) {
      inc_cw();
#ifdef DEBUG
      fprintf(stderr, "%f: %d check_pktRTS backoff start\n",
	      Scheduler::instance().clock(), index);
#endif
      mhBackoff.start(cw, 0.0, is_idle(), "rts");
      return 0;
    }

    SET_TX_STATE(MAC_RTS);
#ifdef DEBUG
    fprintf(stderr, "%f: %d SET_TX_STATE MAC_RTS\n",
	    Scheduler::instance().clock(), index);
#endif
    timeout = CTSTimeout;

    break;

  default:
    fprintf(stderr, "Invalid MAC Control subtype\n");
    exit(1);
  }

  TRANSMIT(pktRTS, timeout);

  return 0;
}

int Mac802_11::check_pktTx()
{
  struct hdr_mac802_11 *mh;
  double timeout;

  assert(mhBackoff.busy() == 0);

  if (pktTx == 0)
    return -1;

  mh = HDR_MAC11(pktTx);
  int len = HDR_CMN(pktTx)->size();

  switch (mh->dh_fc.fc_subtype) {

  case MAC_Subtype_Data:

    if (!is_idle()) {
      // sets pktRTS
      sendRTS(ETHER_ADDR(mh->dh_da));

      inc_cw();
#ifdef DEBUG
      fprintf(stderr, "%f: %d check_pktTx data backoff start\n",
	      Scheduler::instance().clock(), index);
#endif
      mhBackoff.start(cw, 0.0, is_idle(), "data backoff");
      return 0;
    }
    SET_TX_STATE(MAC_SEND);
#ifdef DEBUG
    fprintf(stderr, "%f: %d SET_TX_STATE MAC_SEND\n",
	    Scheduler::instance().clock(), index);
#endif
    if (ETHER_ADDR(mh->dh_da) != MAC_BROADCAST)
      timeout = ACKTimeout(len);
    else
      timeout = TX_Time(pktTx);

    break;

  default:
    fprintf(stderr, "Invalid MAC Control subtype\n");
    exit(1);
  }

  TRANSMIT(pktTx, timeout);

  return 0;
}

int Mac802_11::check_pktBEACON()
{
  struct hdr_mac802_11 *mh;

  if (!pktBEACON)
    return -1;
  if (tx_state != MAC_IDLE)
    return -1;
  mh = HDR_MAC11(pktBEACON);
  switch (mh->dh_fc.fc_subtype) {
  case MAC_Subtype_Beacon:
    SET_TX_STATE(MAC_BEACON);
#ifdef DEBUG
    fprintf(stderr, "%f: %d SET_TX_STATE MAC_BEACON; got beacon\n",
	    Scheduler::instance().clock(), index);
#endif
    break;

  default:
    fprintf(stderr, "Invalid MAC Control subtype\n");
    exit(1);
  }

  got_beacon_ = true;
  TRANSMIT(pktBEACON, TX_Time(pktBEACON));
  return 0;
}


int Mac802_11::check_pktATIM()
{
  struct hdr_mac802_11 *atim_mh;
  struct hdr_cmn *ch;

  assert(!mhBackoff.busy());

  if (tx_state != MAC_IDLE)
    return -1;

  if (pktATIM == 0)
    return -1;

  assert(entryATIM);
  atim_mh = HDR_MAC11(pktATIM);
  ch = HDR_CMN(pktATIM);

  if (!is_idle()) {
    if (!mhBackoff.busy()) {
      inc_cw();
#ifdef DEBUG
      fprintf(stderr, "%f: %d check_pktATIM ATIM backoff start\n",
	      Scheduler::instance().clock(), index);
#endif
      mhBackoff.start(cw, 0.0, is_idle(), "atim backoff sending");
    }
    return -1;
  } else {
    SET_TX_STATE(MAC_ATIM);
#ifdef DEBUG
    fprintf(stderr, "%f: %d SET_TX_STATE MAC_ATIM\n",
	    Scheduler::instance().clock(), index);
    fprintf(stderr, "%f: da=%d\n", Scheduler::instance().clock(),
	    ETHER_ADDR(atim_mh->dh_da));
#endif
    // create an ATIM packet based on pktATIM
    atim_mh->dh_fc.fc_type = MAC_Type_Management;
    atim_mh->dh_fc.fc_subtype = MAC_Subtype_ATIM;

    double timeout;
    if (ETHER_ADDR(atim_mh->dh_da) != MAC_BROADCAST)
      timeout = ACKTimeout(ch->size());
    else
      timeout = TX_Time(pktATIM);

    // store the address of the STA we last transmitted an ATIM
    // to
    STORE4BYTE(atim_mh->dh_da, last_atim_da_);

    if (ch->next_hop_ == (int) MAC_BROADCAST)
      has_packet_tx_++;
#if BENJIE_DEBUG > 1
    fprintf(stderr, "%f: _%d_ send ATIM %d, %d\n",
	    Scheduler::instance().clock(),
	    index, ch->next_hop_, has_packet_tx_);
#endif
    // transmit the ATIM
    TRANSMIT(pktATIM, timeout);
  }
  return 0;
}

/*
 * Low-level transmit functions that actually place the packet onto
 * the channel.
 */
void Mac802_11::sendRTS(int dst)
{
  Packet *p = Packet::alloc();
  hdr_cmn *ch = HDR_CMN(p);
  struct rts_frame *rf = (struct rts_frame *) p->access(off_mac_);

  assert(pktTx);
  assert(pktRTS == 0);

  /*
   *  If the size of the packet is larger than the
   *  RTSThreshold, then perform the RTS/CTS exchange.
   *
   *  XXX: also skip if destination is a broadcast
   */

  if ((u_int32_t) HDR_CMN(pktTx)->size() < macmib->RTSThreshold ||
      (u_int32_t) dst == MAC_BROADCAST) {
    Packet::free(p);
    return;
  }

  ch->uid() = 0;
  ch->ptype() = PT_MAC;
  ch->size() = ETHER_RTS_LEN;
  ch->iface() = -2;
  ch->error() = 0;

  bzero(rf, MAC_HDR_LEN);

  rf->rf_fc.fc_protocol_version = MAC_ProtocolVersion;
  rf->rf_fc.fc_type = MAC_Type_Control;
  rf->rf_fc.fc_subtype = MAC_Subtype_RTS;
  rf->rf_fc.fc_to_ds = 0;
  rf->rf_fc.fc_from_ds = 0;
  rf->rf_fc.fc_more_frag = 0;
  rf->rf_fc.fc_retry = 0;
  rf->rf_fc.fc_pwr_mgt = ((SharedMedia *) netif_)->PowerSave()? 1 : 0;
  rf->rf_fc.fc_more_data = 0;
  rf->rf_fc.fc_wep = 0;
  rf->rf_fc.fc_order = 0;

  rf->rf_duration = RTS_DURATION(pktTx);
  STORE4BYTE(&dst, rf->rf_ra);
  STORE4BYTE(&index, rf->rf_ta);
  // rf->rf_fcs;

  pktRTS = p;
}

void Mac802_11::sendCTS(int dst, double rts_duration)
{
  Packet *p = Packet::alloc();
  hdr_cmn *ch = HDR_CMN(p);
  struct cts_frame *cf = (struct cts_frame *) p->access(off_mac_);

  assert(pktCTRL == 0);

  ch->uid() = 0;
  ch->ptype() = PT_MAC;
  ch->size() = ETHER_CTS_LEN;
  ch->iface() = -2;
  ch->error() = 0;

  bzero(cf, MAC_HDR_LEN);

  cf->cf_fc.fc_protocol_version = MAC_ProtocolVersion;
  cf->cf_fc.fc_type = MAC_Type_Control;
  cf->cf_fc.fc_subtype = MAC_Subtype_CTS;
  cf->cf_fc.fc_to_ds = 0;
  cf->cf_fc.fc_from_ds = 0;
  cf->cf_fc.fc_more_frag = 0;
  cf->cf_fc.fc_retry = 0;
  cf->cf_fc.fc_pwr_mgt = ((SharedMedia *) netif_)->PowerSave()? 1 : 0;
  cf->cf_fc.fc_more_data = 0;
  cf->cf_fc.fc_wep = 0;
  cf->cf_fc.fc_order = 0;

  cf->cf_duration = CTS_DURATION(rts_duration);
  STORE4BYTE(&dst, cf->cf_ra);
  // cf->cf_fcs;

  pktCTRL = p;
}

void Mac802_11::sendACK(int dst)
{
  Packet *p = Packet::alloc();
  hdr_cmn *ch = HDR_CMN(p);
  struct ack_frame *af = (struct ack_frame *) p->access(off_mac_);

  assert(pktCTRL == 0);

  ch->uid() = 0;
  ch->ptype() = PT_MAC;
  ch->size() = ETHER_ACK_LEN;
  ch->iface() = -2;
  ch->error() = 0;

  bzero(af, MAC_HDR_LEN);

  af->af_fc.fc_protocol_version = MAC_ProtocolVersion;
  af->af_fc.fc_type = MAC_Type_Control;
  af->af_fc.fc_subtype = MAC_Subtype_ACK;
  af->af_fc.fc_to_ds = 0;
  af->af_fc.fc_from_ds = 0;
  af->af_fc.fc_more_frag = 0;
  af->af_fc.fc_retry = 0;
  af->af_fc.fc_pwr_mgt = ((SharedMedia *) netif_)->PowerSave()? 1 : 0;
  af->af_fc.fc_more_data = 0;
  af->af_fc.fc_wep = 0;
  af->af_fc.fc_order = 0;

  af->af_duration = ACK_DURATION();
  STORE4BYTE(&dst, af->af_ra);
  // af->af_fcs;

  pktCTRL = p;
}

void Mac802_11::sendDATA(Packet * p)
{
  hdr_cmn *ch = HDR_CMN(p);
  struct hdr_mac802_11 *dh = HDR_MAC11(p);

  if (pktTx)
    assert(0);

  /*
   * Update the MAC header
   */
  ch->size() += ETHER_HDR_LEN11;

  dh->dh_fc.fc_protocol_version = MAC_ProtocolVersion;
  dh->dh_fc.fc_type = MAC_Type_Data;
  dh->dh_fc.fc_subtype = MAC_Subtype_Data;
  dh->dh_fc.fc_to_ds = 0;
  dh->dh_fc.fc_from_ds = 0;
  dh->dh_fc.fc_more_frag = 0;
  dh->dh_fc.fc_retry = 0;
  dh->dh_fc.fc_pwr_mgt = ((SharedMedia *) netif_)->PowerSave()? 1 : 0;
  dh->dh_fc.fc_more_data = 0;
  dh->dh_fc.fc_wep = 0;
  dh->dh_fc.fc_order = 0;

  if (ETHER_ADDR(dh->dh_da) != MAC_BROADCAST)
    dh->dh_duration = DATA_DURATION();
  else
    dh->dh_duration = 0;

  pktTx = p;
}

void Mac802_11::sendBEACON()
{
  Packet *p = Packet::alloc();
  hdr_cmn *ch = HDR_CMN(p);
  struct hdr_mac802_11 *dh = HDR_MAC11(p);
  struct hdr_beacon *bh = HDR_BEACON(p);

  // 802.11 mac header
  bzero(dh, MAC_HDR_LEN);
  dh->dh_fc.fc_protocol_version = MAC_ProtocolVersion;
  dh->dh_fc.fc_type = MAC_Type_Management;
  dh->dh_fc.fc_subtype = MAC_Subtype_Beacon;
  dh->dh_fc.fc_to_ds = 0;
  dh->dh_fc.fc_from_ds = 0;
  dh->dh_fc.fc_more_frag = 0;
  dh->dh_fc.fc_retry = 0;
  dh->dh_fc.fc_pwr_mgt = ((SharedMedia *) netif_)->PowerSave()? 1 : 0;
  dh->dh_fc.fc_more_data = 0;
  dh->dh_fc.fc_wep = 0;
  dh->dh_fc.fc_order = 0;
  dh->dh_duration = 0;
  u_int32_t dst = MAC_BROADCAST;
  STORE4BYTE(&index, &dh->dh_sa);
  STORE4BYTE(&dst, &dh->dh_da);

  // beacon header
  bzero(bh, sizeof(struct hdr_beacon));
  // bh->bh_bb.bb_ts = tsfTimer.getTSF();
  // note: tsf is filled in later
  bh->bh_bb.bb_bi = (u_int16_t) (macmib->BeaconPeriod * 1e3);	// Kus
  // XXX settings for STAs in an IBSS only
  bh->bh_bb.bb_ci.ci_ess = (u_char) 0x0;
  bh->bh_bb.bb_ci.ci_ibss = (u_char) 0x1;
  bh->bh_bb.bb_ci.ci_cfpollable = (u_char) 0x0;
  bh->bh_bb.bb_ci.ci_cfpollreq = (u_char) 0x0;
  // XXX assume no WEP for now
  bh->bh_bb.bb_ci.ci_privacy = (u_char) 0x0;
  bh->bh_bb.bb_ssid.ssid_eid = (u_int8_t) 0x0;
  bh->bh_bb.bb_ssid.ssid_length = (u_int8_t) 0x1;
  // we set the SSID to zero so all stations in the simulation
  // are in the same universe.
  bh->bh_bb.bb_sr.sr_eid = (u_int8_t) 0x0;
  bh->bh_bb.bb_sr.sr_length = (u_int8_t) 0x0;
  bh->bh_bb.bb_sr.sr_rates[0] = (u_char) 0x12;
  // XXX bh->bh_bb.bb_fps.fps_fps_eid
  bh->bh_bb.bb_fps.fps_length = (u_int8_t) 1;
  bh->bh_bb.bb_fps.fps_dwelltime = (u_int16_t) 390;
  bh->bh_bb.bb_fps.fps_hopset = (u_int8_t) 1;
  // bh->bh_bb.bb_fps.fps_hoppat
  // bh->bh_bb.bb_fps.fps_hopind
  // XXX don't do indices for now
  bh->bh_bb.bb_dps = 0;
  // note: cf_param_set unused for PCF only hence unused
  bh->bh_bb.bb_ips.ips_eid = 0;
  bh->bh_bb.bb_ips.ips_length = (u_int8_t) 2;
  bh->bh_bb.bb_ips.ips_atim_wind = (u_int16_t) (macmib->ATIMWindow * 1e3);
  //XXX p. 135 is unclear
  bh->bh_bb.bb_tim.tim_eid = (u_int8_t) 0x0;
  // note: TIM unused for DCF STAs
  // bh->_bh_fcs: FCS unimplemented (but takes up space) XXXXX

  // common header
  ch->ptype() = PT_MAC;
  ch->iface() = -2;
  ch->error() = 0;
  ch->size() = ETHER_BEACON_LEN(bh);
  pktBEACON = p;
}

/* ======================================================================
   Retransmission Routines
   ====================================================================== */
void Mac802_11::RetransmitRTS()
{
  assert(pktTx);
  assert(pktRTS);

  assert(mhBackoff.busy() == 0);

  macmib->RTSFailureCount++;

  ssrc += 1;			// STA Short Retry Count
#if RTS_RETRY_COUNTER 
  rtsrc += 1;
#endif

  if (ssrc >= macmib->ShortRetryLimit || rtsrc >= MAC_RTSRetryLimit) {

    discard(pktRTS, DROP_MAC_RETRY_COUNT_EXCEEDED);
    pktRTS = 0;

    /* tell the callback the send operation failed 
       before discarding the packet */
    hdr_cmn *ch = HDR_CMN(pktTx);
    if (ch->xmit_failure_) {
      ch->size() -= ETHER_HDR_LEN11;
      ch->xmit_reason_ = XMIT_REASON_RTS;
      ch->xmit_failure_(pktTx->copy(), ch->xmit_failure_data_);
#ifdef TXF_DEBUG
      if (ch->next_hop_ != MAC_BROADCAST) {
        fprintf(stderr, "%f: %d->%d RetransmitRTS TXF\n",
	        Scheduler::instance().clock(), index, ch->next_hop_);
      }
#endif
    }
    discard(pktTx, DROP_MAC_RETRY_COUNT_EXCEEDED);
    pktTx = 0;
    ssrc = 0;
    rtsrc = 0;
    rst_cw();
  }

  else {
    struct rts_frame *rf;

    rf = (struct rts_frame *) pktRTS->access(off_mac_);
    rf->rf_fc.fc_retry = 1;

    inc_cw();
#ifdef DEBUG
    fprintf(stderr, "%f: %d RetransmitRTS backoff\n",
	    Scheduler::instance().clock(), index);
#endif
    mhBackoff.start(cw, 0.0, is_idle(), "rexmit rts backoff");
  }
}


void Mac802_11::RetransmitDATA()
{
  struct hdr_cmn *ch;
  struct hdr_mac802_11 *mh;
  u_int32_t *rcount, *thresh;

  if (mhBackoff.busy()) {
    fprintf(stderr, "%f: %d: RetransmitDATA assertion failed\n",
	    Scheduler::instance().clock(), index);
    assert(0);
  }

  assert(pktTx);
  assert(pktRTS == 0);

  ch = HDR_CMN(pktTx);
  mh = HDR_MAC11(pktTx);

  /*
   *  Broadcast packets don't get ACKed and therefore
   *  are never retransmitted.
   */
  if (ETHER_ADDR(mh->dh_da) == MAC_BROADCAST) {
    Packet::free(pktTx);
    pktTx = 0;

    /*
     * Backoff at end of TX.
     */
    rst_cw();
    mhBackoff.start(cw, 0.0, is_idle(), "rexmit data backoff");

    return;
  }

  macmib->ACKFailureCount++;

  if ((u_int32_t) ch->size() <= macmib->RTSThreshold) {
    rcount = &ssrc;
    thresh = &macmib->ShortRetryLimit;
  } else {
    rcount = &slrc;
    thresh = &macmib->LongRetryLimit;
  }

  (*rcount)++;

  if (*rcount > *thresh) {

    macmib->FailedCount++;

    /* tell the callback the send operation failed 
       before discarding the packet */
    hdr_cmn *ch = HDR_CMN(pktTx);
    if (ch->xmit_failure_) {
      ch->size() -= ETHER_HDR_LEN11;
      ch->xmit_reason_ = XMIT_REASON_ACK;
      ch->xmit_failure_(pktTx->copy(), ch->xmit_failure_data_);
    }

    discard(pktTx, DROP_MAC_RETRY_COUNT_EXCEEDED);
    pktTx = 0;

    *rcount = 0;
    rst_cw();
  } else {
    /* jinyang */
    total_retransmit++;

    struct hdr_mac802_11 *dh;

    dh = HDR_MAC11(pktTx);
    dh->dh_fc.fc_retry = 1;

    sendRTS(ETHER_ADDR(mh->dh_da));
    inc_cw();
    mhBackoff.start(cw, 0.0, is_idle(), "rexmit data backoff - 2");
  }
}

void Mac802_11::RetransmitATIM()
{
  struct hdr_mac802_11 *mh = HDR_MAC11(pktATIM);;
#ifdef DEBUG
  fprintf(stderr, "%f: %d RetransmitATIM, to %d\n",
	  Scheduler::instance().clock(), index, ETHER_ADDR(last_atim_da_));
#endif
  if (mhBackoff.busy())
    mhBackoff.stop();
  if (!pktATIM)
    assert(0);

  if ((!mhEndATIM.busy()) || (ETHER_ADDR(mh->dh_da) == MAC_BROADCAST)) {
    Packet::free(pktATIM);
    pktATIM = 0;
    rst_cw();
    mhBackoff.start(cw, 0.0, is_idle(), "rexmit atim backoff");
#ifdef DEBUG
    fprintf(stderr, "%f: %d end of ATIM or broadcast, bye-bye\n", 
	    Scheduler::instance().clock(), index);
#endif
    return;
  }

  // look to see if this is still necessary 
  struct buffer_entry *this_entry;
  CIRCLEQ_FOREACH(this_entry, &buf_head_, entries) {
    if ((this_entry->addr(this) == ETHER_ADDR(mh->dh_da)) &&
	(this_entry->recvd_ack_ || !this_entry->pwr_mgt_)) {
      Packet::free(pktATIM);
      pktATIM = 0;
      rst_cw();
      mhBackoff.start(cw, 0.0, is_idle(), "rexmit atim backoff");
#ifdef DEBUG
      fprintf(stderr, "%f: %d ATIM no longer necessary, %d, %d\n", 
	      Scheduler::instance().clock(), index,
	      this_entry->recvd_ack_, this_entry->pwr_mgt_);
#endif
      return;
    }
  }

  macmib->ACKFailureCount++;
  atimrc++;
  if (atimrc > MAC_ATIMRetryLimit) {
#ifdef DEBUG
    fprintf(stderr, "%f: %d ATIM retries exceeded...\n", 
	    Scheduler::instance().clock(), index);
#endif
    macmib->FailedCount++;
    discard(pktATIM, DROP_MAC_ATIM_RETRY_COUNT);
    pktATIM = 0;
    atimrc = 0;
    rst_cw();
    
    CIRCLEQ_HEAD(buffer_head, buffer_entry) expired;
    CIRCLEQ_INIT(&expired);

    CIRCLEQ_FOREACH(this_entry, &buf_head_, entries) {
      if (this_entry->addr(this) == ETHER_ADDR(last_atim_da_)) { 
	CIRCLEQ_REMOVE(&buf_head_, this_entry, entries); 
        CIRCLEQ_INSERT_TAIL(&expired, this_entry, entries);
      }
    }

    CIRCLEQ_FOREACH(this_entry, &expired, entries) { 
      if (!this_entry->sent_data_) { 
        hdr_cmn *ch = HDR_CMN(this_entry->packet_); 
        if (ch->xmit_failure_) { 
	  ch->xmit_reason_ = XMIT_REASON_ATIM;
	  ch->xmit_failure_
	    (this_entry->packet_->copy(), ch->xmit_failure_data_); 
        } 
        discard(this_entry->packet_, DROP_MAC_RETRY_COUNT_EXCEEDED);
      }
      delete this_entry;
    }
  } else {
    // retry count is at most threshold
    mh->dh_fc.fc_retry = 1;
    inc_cw();
    mhBackoff.start(cw, 0.0, is_idle(), "rexmit data backoff - 2");
#ifdef DEBUG
    fprintf(stderr, "%f: %d RetransmitATIM, backoff %d\n", 
	    Scheduler::instance().clock(), index, cw);
#endif
  }
}

/* ======================================================================
   Incoming Packet Routines
   ====================================================================== */
void Mac802_11::send(Packet * p, Handler * h)
{
  struct hdr_mac802_11 *dh = HDR_MAC11(p);
#if USE_SPAN
  hdr_span *spanh =  (hdr_span*)p->access(off_span_);
#endif
  struct buffer_entry *this_entry = 0;
#if FASTROUTE
  bool fastroute = spanh->to_coordinator;
#else
  bool fastroute = false;
#endif

  if (((SharedMedia*)netif_)->state() == pwr_state_off) {
    Packet::free(p);
    return;
  }

  upcall_ = h;

  if (idle_start > 0.01) {
    total_true_idle_time += (Scheduler::instance().clock() - idle_start);
    idle_start = 0.0;
  }

  /* Assign the data packet a sequence number. */
  dh->dh_scontrol = sta_seqno++;

  /* correct to_coordinator field in Span header if we
   * have more up to date pwr mgt status */
  CIRCLEQ_FOREACH(this_entry, &buf_head_, entries) { 
    if (this_entry->addr(this) == ETHER_ADDR(dh->dh_da)) {
      if (this_entry->pwr_mgt_updated_) {
	fastroute = !this_entry->pwr_mgt_;
	break;
      }
    }
  }

  if (!psm_mode_) {
    assert(pktTx == 0);
    assert(((SharedMedia *) netif_)->state() == pwr_state_idle ||
	   ((SharedMedia *) netif_)->state() == pwr_state_off);
    assert(!((SharedMedia *) netif_)->PowerSave());
  }
  
  if (((SharedMedia *) netif_)->state() != pwr_state_idle) {
    struct buffer_entry *new_entry =
	new buffer_entry(p, false, false, false, 0);
    new_entry->pwr_mgt_ = !fastroute;
    CIRCLEQ_INSERT_TAIL(&buf_head_, new_entry, entries);
    return;
  }


  if (mhEndATIM.busy() && got_beacon_) { // inside ATIM window

    struct buffer_entry *new_entry =
	new buffer_entry(p, false, false, false, 0);
    bool found = false;

    // check all ATIM entries to same next hop
    CIRCLEQ_FOREACH(this_entry, &buf_head_, entries) {
      if ((this_entry->addr(this) == new_entry->addr(this)) &&
	  (this_entry->sent_atim_ || !this_entry->pwr_mgt_)) {
#ifdef DEBUG
	fprintf(stderr, "%f: %d send, found a matching atim\n",
		Scheduler::instance().clock(), index);
#endif
	new_entry->sent_atim_ = this_entry->sent_atim_;
	new_entry->recvd_ack_ = this_entry->recvd_ack_;
        new_entry->pwr_mgt_ = !fastroute;
	found = true;
	break;
      }
    }
    CIRCLEQ_INSERT_TAIL(&buf_head_, new_entry, entries);

    // send an atim only if not gratuitous
    if (!found && !pktATIM) {
      atim_scan();

      if (pktATIM) {
        if (first_atim_) {
	  if (!mhBackoff.busy() && !mhDefer.busy())
	    mhBackoff.start(cw, 0.0, 1, "first atim backoff");
	  first_atim_ = false;
        } else {
	  if (mhBackoff.busy() == 0) {
	    if (is_idle()) {
	      if (mhDefer.busy() == 0)
	        mhDefer.start(difs, "difs defer");
	    } else {
	      mhBackoff.start(cw, 0.0, is_idle(), "backoff (send)");
	    }
	  }
        }
      }
    } 
    else 
      upcall();
  }

  else {			/* !mhEndATIM.busy() || !got_beacon_ */
    // we are outside the ATIM window, or inside the atim
    // window and don't have a beacon yet: search for the
    // station in acked or bdcast pkts.
#if BENJIE_DEBUG > 1
    fprintf(stderr,
	    "%f: %d sending outside atim window or not got beacon.\n",
	    Scheduler::instance().clock(), index);
#endif

    bool found = false;
    CIRCLEQ_FOREACH(this_entry, &buf_head_, entries) {
      if ((this_entry->addr(this) == ETHER_ADDR(dh->dh_da)) &&
	  (this_entry->recvd_ack_ ||
	   (ETHER_ADDR(dh->dh_da) == MAC_BROADCAST &&
	    this_entry->sent_atim_) ||
	   (!this_entry->pwr_mgt_ && fastroute))) { 
	found = true;
	break;
      }
    }
    
    if ((!found && !fastroute && psm_mode_) || pktTx || mhEndATIM.busy()) {
      // no atim was found, or we are inside the atim window waiting for
      // a beacon, or there is a packet pending transmission, or if fast
      // route is not on...
#if BENJIE_DEBUG > 1
      fprintf(stderr, "%f: %d buffering the pkt (found %d, fr %d): ",
	      Scheduler::instance().clock(), index, found, fastroute);
      trace_pkt(p);
#endif
      // buffer the new packet and do nothing for now
      struct buffer_entry *new_entry =
	  new buffer_entry(p, false, false, false, 0);
      new_entry->pwr_mgt_ = !fastroute;
      CIRCLEQ_INSERT_TAIL(&buf_head_, new_entry, entries);

    } else {

      // an atim was found and we are outside the atim window; send data
      // with the dcf, OR we are doing fast route

#if BENJIE_DEBUG > 1
      fprintf(stderr, "%f: %d sending with the dcf\n",
	      Scheduler::instance().clock(), index);
#endif
      sendDATA(p);
      sendRTS(ETHER_ADDR(dh->dh_da));

      if (fastroute && !found) {
#if BENJIE_DEBUG > 2
	fprintf(stderr, "%f: %d optimistic send to coordinator %d\n",
		Scheduler::instance().clock(),
		index, ETHER_ADDR(dh->dh_da));
#endif
      }

      /*
       *  If the medium is IDLE, we must wait for a DIFS
       *  Space before transmitting.
       */
      if (mhBackoff.busy() == 0) {
	if (is_idle()) {
	  /* If we are already deferring, there is no
	   * need to reset the Defer timer. */
	  if (mhDefer.busy() == 0)
	    mhDefer.start(difs, "difs defer");
	}

	/* If the medium is NOT IDLE, then we start
	 * the backoff timer. */
	else {
	  mhBackoff.start(cw, 0.0, is_idle(), "backoff (send)");
	}
      }

    }
  }
}


void Mac802_11::recv(Packet * p, Handler * h)
{
  struct hdr_cmn *hdr = HDR_CMN(p);

  /*
   * Sanity Check
   */
  assert(initialized());

  /*
   *  Handle outgoing packets.
   */
  if (h) {
    send(p, h);
    return;
  }
  assert(((SharedMedia *) netif_)->state() == pwr_state_idle);

  /*
   *  Handle incoming packets.
   *
   *  We just received the 1st bit of a packet on the network
   *  interface.
   *
   */

  /*
   *  If the interface is currently in transmit mode, then
   *  it probably won't even see this packet.  However, the
   *  "air" around me is BUSY so I need to let the packet
   *  proceed.  Just set the error flag in the common header
   *  to that the packet gets thrown away.
   */
  if (tx_active && hdr->error() == 0) {
    hdr->error() = 1;
  }

  first_bit_time = Scheduler::instance().clock();

  if (rx_state == MAC_IDLE) {
    SET_RX_STATE(MAC_RECV);
    pktRx = p;

    /*
     * Schedule the reception of this packet, in
     * txtime seconds.
     */
    assert(!mhRecv.busy());
    mhRecv.start(TX_Time(p), "incoming recv");
  } else {
    /*
     *  If the power of the incoming packet is smaller than the
     *  power of the packet currently being received by at least
     *  the capture threshold, then we ignore the new packet.
     */
    if (pktRx->txinfo.RxPr / p->txinfo.RxPr >= p->txinfo.CPThresh) {
      capture(p);
    } else {
      collision(p);
    }
  }
}


void Mac802_11::recv_timer()
{
  hdr_cmn *ch = HDR_CMN(pktRx);
  hdr_mac802_11 *mh = HDR_MAC11(pktRx);
  u_int32_t dst = ETHER_ADDR(mh->dh_da);
  u_int8_t type = mh->dh_fc.fc_type;
  u_int8_t subtype = mh->dh_fc.fc_subtype;

  assert(pktRx);
  assert(rx_state == MAC_RECV || rx_state == MAC_COLL);

  /*
   *  If the interface is in TRANSMIT mode when this packet
   *  "arrives", then I would never have seen it and should
   *  do a silent discard without adjusting the NAV.
   */
  if (tx_active) {
    Packet::free(pktRx);
    goto done;
  }

  /*
   * Handle collisions.
   */
  if (rx_state == MAC_COLL) {
#ifdef COLLISION_DEBUG
    if (dst == index) {
      fprintf(stderr, "%f: %d collision, after recv timer: ",
	      Scheduler::instance().clock(), index);
      trace_pkt(pktRx);
    }
#endif
    discard(pktRx, DROP_MAC_COLLISION);
    set_nav(usec(eifs));
    goto done;
  }

  /*
   * Check to see if this packet was received with enough
   * bit errors that the current level of FEC still could not
   * fix all of the problems - ie; after FEC, the checksum still
   * failed.
   */
  if (ch->error()) {
    Packet::free(pktRx);
    set_nav(usec(eifs));
    goto done;
  }

  /*
   * IEEE 802.11 specs, section 9.2.5.6
   *      - update the NAV (Network Allocation Vector)
   */
  if (dst != index) {
    set_nav(mh->dh_duration);
  }

  /* tap out */
  if (tap && type == MAC_Type_Data && MAC_Subtype_Data == subtype)
    tap->tap(pktRx);

  /* update our power management table */
  buffer_entry *this_entry;
  switch (type) {
  case MAC_Type_Management:
    switch (subtype) {
    case MAC_Subtype_Beacon:
    case MAC_Subtype_ATIM:
      CIRCLEQ_FOREACH(this_entry, &buf_head_, entries) {
	if (this_entry->addr(this) == ETHER_ADDR(mh->dh_sa)) {
	  this_entry->pwr_mgt_ = (mh->dh_fc.fc_pwr_mgt == 1);
	  this_entry->pwr_mgt_updated_ = true;
	}
      }
      break;
    default:
      break;
    }
  case MAC_Type_Control:
    switch (subtype) {
    case MAC_Subtype_RTS:
      {
	struct rts_frame *rf =
	    (struct rts_frame *) pktRx->access(off_mac_);
	CIRCLEQ_FOREACH(this_entry, &buf_head_, entries) {
	  if (this_entry->addr(this) == ETHER_ADDR(rf->rf_ta)) {
	    this_entry->pwr_mgt_ = (rf->rf_fc.fc_pwr_mgt == 1);
	    this_entry->pwr_mgt_updated_ = true;
	  }
	}
      }
      break;
    case MAC_Subtype_CTS:
      if (pktRTS && dst == index) {
	struct cts_frame *cf =
	    (struct cts_frame *) pktRx->access(off_mac_);
	struct rts_frame *rf =
	    (struct rts_frame *) pktRTS->access(off_mac_);
	CIRCLEQ_FOREACH(this_entry, &buf_head_, entries) {
	  // check against dst addr of RTS packet
	  if (this_entry->addr(this) == ETHER_ADDR(rf->rf_ra)) {
	    this_entry->pwr_mgt_ = (cf->cf_fc.fc_pwr_mgt == 1);
	    this_entry->pwr_mgt_updated_ = true;
	  }
	}
      }
      break;
    default:
      break;
    }
  case MAC_Type_Data:
    switch (subtype) {
    case MAC_Subtype_Data:
      CIRCLEQ_FOREACH(this_entry, &buf_head_, entries) {
	if (this_entry->addr(this) == ETHER_ADDR(mh->dh_sa)) {
	  this_entry->pwr_mgt_ = (mh->dh_fc.fc_pwr_mgt == 1);
	  this_entry->pwr_mgt_updated_ = true;
	}
      }
      break;
    default:
      break;
    }
  }

  /*
   * Address Filtering
   */
  if ((dst != index) && (dst != MAC_BROADCAST)) {
    /*
     *  We don't want to log this event, so we just free
     *  the packet instead of calling the drop routine.
     */
    discard(pktRx, "---");
    goto done;
  }

  switch (type) {

  case MAC_Type_Management:
    switch (subtype) {
    case MAC_Subtype_Beacon:
      recvBEACON(pktRx);
      break;

    case MAC_Subtype_ATIM:
      recvATIM(pktRx);
      break;

    default:
      discard(pktRx, DROP_MAC_PACKET_ERROR);
      goto done;
    }
    break;

  case MAC_Type_Control:

    switch (subtype) {

    case MAC_Subtype_RTS:
      recvRTS(pktRx);
      break;

    case MAC_Subtype_CTS:
      recvCTS(pktRx);
      break;

    case MAC_Subtype_ACK:
      recvACK(pktRx);
      break;

    default:
      fprintf(stderr, "Invalid MAC Control Subtype %x\n", subtype);
      exit(1);
    }
    break;

  case MAC_Type_Data:

    switch (subtype) {

    case MAC_Subtype_Data:
      recvDATA(pktRx);
      break;

    default:
      fprintf(stderr, "Invalid MAC Data Subtype %x\n", subtype);
      exit(1);
    }
    break;

  default:
    fprintf(stderr, "Invalid MAC Type %x\n", subtype);
    exit(1);
  }

done:

  pktRx = 0;
  rx_resume();
}


void Mac802_11::recvRTS(Packet * p)
{
  struct rts_frame *rf = (struct rts_frame *) p->access(off_mac_);

  if (tx_state != MAC_IDLE || mhEndATIM.busy()) {
    discard(p, DROP_MAC_BUSY);
    return;
  }

  /*
   *  If I'm responding to someone else, discard this RTS.
   */
  if (pktCTRL) {
    discard(p, DROP_MAC_BUSY);
    return;
  }

  sendCTS(ETHER_ADDR(rf->rf_ta), rf->rf_duration);

  /*
   *  Stop deferring - will be reset in tx_resume().
   */
  if (mhDefer.busy())
    mhDefer.stop();

  tx_resume();

  mac_log(p);
}


void Mac802_11::recvCTS(Packet * p)
{
  if (tx_state != MAC_RTS) {
    discard(p, DROP_MAC_INVALID_STATE);
    if (pktTx) {
      // buffer the packet and do nothing for now
      hdr_cmn *ch = HDR_CMN(pktTx);
      ch->size() -= ETHER_HDR_LEN11;
      struct buffer_entry *new_entry = 
	new buffer_entry(pktTx, false, false, false, 0);
      CIRCLEQ_INSERT_TAIL(&buf_head_, new_entry, entries);
      pktTx = 0;
    }
    return;
  }

  assert(pktRTS);
  assert(mhBackoff.busy() == 0);
  Packet::free(pktRTS);
  pktRTS = 0;

  assert(pktTx);

  mhSend.stop();

  /*
   * The successful reception of this CTS packet implies
   * that our RTS was successful.  Hence, we can reset
   * the Short Retry Count and the CW.
   */
  ssrc = 0;
  rtsrc = 0;
  rst_cw();

  tx_resume();

  mac_log(p);
}

void Mac802_11::recvDATA(Packet * p)
{
  struct hdr_mac802_11 *dh = HDR_MAC11(p);
  u_int32_t dst, src, size;

  {
    struct hdr_cmn *ch = HDR_CMN(p);

    dst = ETHER_ADDR(dh->dh_da);
    src = ETHER_ADDR(dh->dh_sa);
    size = ch->size();

    /*
     * Adjust the MAC packet size - ie; strip
     * off the mac header
     */
    ch->size() -= ETHER_HDR_LEN11;
    ch->num_forwards() += 1;
  }
  
  /*
   *  If we sent a CTS, clean up...
   */
  if (dst != MAC_BROADCAST) {
    if (size >= macmib->RTSThreshold) {
      if (tx_state == MAC_CTS) {
	assert(pktCTRL);
	Packet::free(pktCTRL);
	pktCTRL = 0;

	mhSend.stop();

	/*
	 * Our CTS got through.
	 */
	ssrc = 0;
	rst_cw();
      } else {
	discard(p, DROP_MAC_BUSY);
	return;
      }
      sendACK(src);
      tx_resume();
    }
    /*
     *  We did not send a CTS and there's no
     *  room to buffer an ACK.
     */
    else {
      if (pktCTRL) {
	discard(p, DROP_MAC_BUSY);
#ifdef DEBUG
	fprintf(stderr, "%f: %d discard busy 1\n",
		Scheduler::instance().clock(), index);
#endif
	return;
      }
      sendACK(src);
      if (mhSend.busy() == 0)
	tx_resume();
    }
  }

  /* ============================================================
     Make/update an entry in our sequence number cache.
     ============================================================ */
  if (dst != MAC_BROADCAST) {
    Host *h = &cache[src];

    if (h->seqno && h->seqno == dh->dh_scontrol) {
      discard(p, DROP_MAC_DUPLICATE);
      return;
    }
    h->seqno = dh->dh_scontrol;
  }

  /*
   *  Pass the packet up to the link-layer.
   */
  p->incoming = 1;
  recvtarget_->recv(p, (Handler *) 0);

#if CHEATBCAST
  if (dst == MAC_BROADCAST && recvd_bcast_atim_ > 0) {
    recvd_bcast_atim_--;
    if ((!mhAdvEnd.busy() ||
	 (!recvd_bcast_atim_ && !recvd_ucast_atim_ && !has_packet_tx_)) &&
	!initial_wake_count_ && ((SharedMedia *) netif_)->PowerSave()) {
      // no data to send nor receive, so can go to sleep
#ifdef BENJIE_DEBUG
      fprintf(stderr, "%f: _%d_ end of bcasts or adv, turn off radio?\n",
	      Scheduler::instance().clock(), index);
#endif
      goto_sleep();
    }
  }
#endif
}


void Mac802_11::recvACK(Packet * p)
{
  struct hdr_cmn *ch = HDR_CMN(p);
  buffer_entry *this_entry;
  struct ack_frame *af = (struct ack_frame *) p->access(off_mac_);

  if ((tx_state != MAC_SEND) && (tx_state != MAC_ATIM)) {
    // this could happen if we canceled a send when a beacon period
    // started...
    mac_log(p);
    return;
  }

  if (tx_state == MAC_ATIM) {
    atimrc = 0;
    // mark buffered packets to the sender as acked
    CIRCLEQ_FOREACH(this_entry, &buf_head_, entries) {
      if (this_entry->addr(this) == ETHER_ADDR(last_atim_da_)) {
	this_entry->recvd_ack_ = true;
	this_entry->pwr_mgt_ = (af->af_fc.fc_pwr_mgt == 1);
	this_entry->pwr_mgt_updated_ = true;
	has_packet_tx_++;
#if BENJIE_DEBUG > 1
	fprintf(stderr, "%f: %d got ack for ATIM %d, from %d\n",
		Scheduler::instance().clock(), index,
		has_packet_tx_, ETHER_ADDR(last_atim_da_));
#endif
      }
    }
  }
#ifdef DEBUG
  else
    fprintf(stderr, "%f: %d recvACK tx_state is MAC_SEND\n",
	    Scheduler::instance().clock(), index);
#endif

  /*
   * Inform the upper layer that we have successfully sent
   * a packet
   */

  if (pktTx && tx_state != MAC_ATIM) {
    hdr_cmn *Txch = HDR_CMN(pktTx);
    if (Txch->xmit_success_) {
      /*  Need to remove the MAC header so that re-cycled packets don't
       *  keep getting bigger.  */
      Txch->size() -= ETHER_HDR_LEN11;
      Txch->xmit_reason_ = 0;
      Txch->xmit_success_(pktTx->copy(), Txch->xmit_success_data_);
    }
  }

  if (tx_state == MAC_SEND) {
    Packet::free(pktTx);
    pktTx = 0;
  } else {
    assert(tx_state == MAC_ATIM);
    Packet::free(pktATIM);
    pktATIM = 0;
  }

  mhSend.stop();

  /*
   * The successful reception of this ACK packet implies that
   * our DATA or ATIM transmission was successful.  Hence, we
   * can reset the Short/Long Retry Count and the CW.  */
  if ((u_int32_t) ch->size() <= macmib->RTSThreshold)
    ssrc = 0;
  else
    slrc = 0;

  /*
   * Backoff before sending again.
   */
  rst_cw();
#ifdef DEBUG
  fprintf(stderr, "%f: %d recvACK: backoff before sending again\n",
	  Scheduler::instance().clock(), index);
#endif
  if (mhBackoff.busy())
    mhBackoff.stop();
  mhBackoff.start(cw, 0.0, is_idle(), "backoff ack recv");

  tx_resume();

  mac_log(p);
}

void Mac802_11::recvBEACON(Packet * p)
{
  u_int64_t tsf;

#ifdef DEBUG
  fprintf(stderr, "%f: %d got a beacon\n",
	  Scheduler::instance().clock(), index);
#endif
  got_beacon_ = true;
  if (pktBEACON) {
#ifdef DEBUG
    fprintf(stderr, "%f: %d set pktBEACON = 0\n",
	    Scheduler::instance().clock(), index);
#endif
    Packet::free(pktBEACON);
    pktBEACON = 0;
  }

  // extract headers
  struct hdr_mac802_11 *dh = HDR_MAC11(p);
  struct hdr_beacon *bh = HDR_BEACON(p);

  // verify that we really have a beacon
  assert((dh->dh_fc.fc_type == MAC_Type_Management) &&
	 (dh->dh_fc.fc_subtype == MAC_Subtype_Beacon));

  // adjust the received timestamp value by adding the receiving
  // STA's delay through its local PHY components plus time since
  // first bit of timestamp arrived at the MAC/PHY interface.
  
  // u_int64_t correction = (u_int64_t)(PHY_DELAY*1e6) + 
  // (u_int64_t)((Scheduler::instance().clock() - first_bit_time)*1e6);
  // bh->bh_bb.bb_ts += correction;
  // fprintf(stderr, "node %d's correction is %lld\n", index, correction);

  // if this beacon's timestamp is later than our own (plus
  // hysteresis), adopt its timestamp.

  tsf = tsfTimer.getTSF();
#ifdef DEBUG
  fprintf(stderr, "%d (tsf=%lld) received a beacon with tsf %lld.\n",
	  index, tsf, bh->bh_bb.bb_ts);
#endif
  if (bh->bh_bb.bb_ts > tsf + 10000) {
    // adopt the beacon
#ifdef DEBUG
    fprintf(stderr, "%d adopting timestamp %lld; used to be %lld\n",
	    index, bh->bh_bb.bb_ts, tsf);
#endif
    tsfTimer.setTSF(bh->bh_bb.bb_ts);
    tsf = tsfTimer.getTSF();

    if (mhTBTT.busy())
      mhTBTT.stop();

    // backoff to send the first atim
    if (!mhBackoff.busy())
      mhBackoff.start(cw, 0.0, is_idle(), "first atim backoff");
    got_beacon_ = true;
#ifdef DEBUG
    fprintf(stderr, "%f: %d recvBEACON got a beacon\n",
	    Scheduler::instance().clock(), index);
#endif
    double tbtt = macmib->BeaconPeriod -
	fmod((double) tsf, macmib->BeaconPeriod * 1e6) / 1e6;
#ifdef DEBUG
    fprintf(stderr, "%d's tbtt is %f seconds away (unadjusted)\n",
	    index, tbtt);
#endif
    if (tbtt < macmib->BeaconPeriod / 3) {
      mhTBTT.start(tbtt + macmib->BeaconPeriod, "tbtt recvBEACON");
    } else {
      mhTBTT.start(tbtt, "tbtt recvBEACON");
    }
  }
  mac_log(p);
}

void Mac802_11::recvATIM(Packet * p)
{
  recvd_atim_ = true;

  struct hdr_mac802_11 *dh = HDR_MAC11(p);
  u_int32_t sendaddr = ETHER_ADDR(dh->dh_sa);
  u_int32_t destaddr = ETHER_ADDR(dh->dh_da);

  if (destaddr == MAC_BROADCAST) { // broadcast ATIM
    recvd_bcast_atim_++;
#ifdef DEBUG
    fprintf(stderr, "%f: %d received a broadcast ATIM\n",
	    Scheduler::instance().clock(), index);
#endif
  } else { // unicast ATIM
    recvd_ucast_atim_++;
#if BENJIE_DEBUG > 1
    fprintf(stderr, "%f: %d received an ATIM and is acking %d %d\n",
	    Scheduler::instance().clock(), index,
	    sendaddr, recvd_ucast_atim_);
#endif
    if (!pktCTRL) {
      sendACK(sendaddr);
      if (mhSend.busy() == 0)
	tx_resume();
    } 
  }

  mac_log(p);
}

int Mac802_11::i_am_idle()
{
  if ((tx_active) || (rx_state != MAC_IDLE)) {
    return 0;
  }
  if (tx_state != MAC_IDLE)
    return 0;

  if (pktCTRL)
    return 0;

  if (!is_idle())
    return 0;

  return 1;
}

void Mac802_11::handleBackoffInterrupt()
{
  if (!mhBackoff.busy()) {
    fprintf(stderr, "fie!\n");	//DEBUG
    return;
  }

  if (index == (u_int32_t) (God::instance()->num_nodes - 1))
    return;

  Mac802_11 *nextnode =
      (Mac802_11 *) (God::instance()->maclist[index + 1]);
  if (is_idle() && (nextnode->i_am_idle())) {
    total_wasted_time += phymib->SlotTime;
  }
}
