#include <sys/types.h>
#include <netinet/in.h>
#include <assert.h>

#include <packet.h>

#include <ip.h>
#include <tcp.h>
#include <rtp.h>
#include <arp.h>

#include <cmu/marshall.h>
#include <cmu/ntr-workload.h>           // 3 ping pattern from SIGCOMM'97 paper
#include <cmu/tora/tora_packet.h>	// TORA
#include <cmu/aodv/aodv_packet.h>	// AODV
#include <cmu/dsr/hdr_sr.h>		// DSR
#include <cmu/imep/imep_spec.h>		// IMEP
#include <cmu/span/hdr_span.h>          // SPAN
#include <cmu/mac.h>
#include <cmu/mac-802_11.h>
#include <cmu/mac-wavelan1.h>
#include <cmu/mac-null.h>

#include <cmu/cmu-trace.h>

// #define LOG_POSITION

extern char* pt_names[];

static class CMUTraceClass : public TclClass {
public:
	CMUTraceClass() : TclClass("CMUTrace") { }
	TclObject* create(int argc, const char*const* argv) {
		return (new CMUTrace(argv[4], *argv[5], argv[6] ));
	}
} cmutrace_class;


CMUTrace::CMUTrace(const char *s, char t, const char *m) : Trace(t)
{
	bzero(tracename, sizeof(tracename));
	strncpy(tracename, s, MAX_ID_LEN);

        if(strcmp(tracename, "RTR") == 0) {
                tracetype = TR_ROUTER;
        }
        else if(strcmp(tracename, "TRP") == 0) {
                tracetype = TR_ROUTER;
        }
        else if(strcmp(tracename, "MAC") == 0) {
                tracetype = TR_MAC;
        }
        else if(strcmp(tracename, "IFQ") == 0) {
                tracetype = TR_IFQ;
        }
        else if(strcmp(tracename, "AGT") == 0) {
                tracetype = TR_AGENT;
        }
        else {
               fprintf(stderr,"CMU Trace Initialized with invalid type '%s'\n",
		       tracename);
                exit(1);
        }

	assert(type_ == DROP || type_ == SEND || type_ == RECV);

	// set up mactype so that format function appropriate to 
	// MAC layer being used can be called at appropriate time

        char mactypetemp[strlen(m)+1];
        strcpy(mactypetemp, m);
	
	if(strcmp(mactypetemp, "Mac/802_11") == 0) {
	  mactype = MAC_TYPE_802_11;
	}
	else if(strcmp(mactypetemp,"Mac/802_3") == 0) {
	  mactype = MAC_TYPE_802_3;
	}
	else if(strcmp(mactypetemp,"Mac/WaveLAN1") == 0) {
	  mactype = MAC_TYPE_WAVELAN1;
	}
	else if(strcmp(mactypetemp,"Mac/NULL") == 0) {
	  mactype = MAC_TYPE_NULL;
	}

	else {
          fprintf(stderr, "CMU Trace Initialized with invalid MAC type '%s'\n",
		  mactypetemp);
	       exit(1);
	}

        node_ = 0;

	bind("off_mac_", &off_mac_);
	bind("off_arp_", &off_arp_);
	bind("off_TORA_", &off_TORA_);
	bind("off_IMEP_", &off_IMEP_);
        bind("off_AODV_", &off_AODV_);
	bind("off_SR_", &off_sr_);
	bind("off_span_", &off_span_);
	bind("off_lm_", &off_lm_);
}

void
CMUTrace::format_mac_WaveLAN1(Packet *p, const char *why, int offset)
{
	struct hdr_cmn *ch = HDR_CMN(p);
	struct hdr_ip *ih = HDR_IP(p);
	struct hdr_mac802_3 *mh = HDR_MAC3(p);
	char op = (char) type_;

	if(tracetype == TR_ROUTER && type_ == SEND) {
		if(src_ != ih->src_)
			op = FWRD;
	}

#ifdef LOG_POSITION
        double x = 0.0, y = 0.0, z = 0.0;
        node_->getLoc(&x, &y, &z);
#endif

	sprintf(wrk_ + offset,
#ifdef LOG_POSITION

		"%c %.9f %d (%6.2f %6.2f) %3s %4s %d %s %d [%x %x %x %x %x] ",
#else
		"%c %.9f _%d_ %3s %4s %d %s %d [%x %x %x %x %x] ",
#endif
		op,
		Scheduler::instance().clock(),
                src_,                           // this node
#ifdef LOG_POSITION
                x,
                y,
#endif
		tracename,
		why,

                ch->uid(),                      // identifier for this event
		pt_names[ch->ptype()],
		ch->size(),
                
		0,                       //GET2BYTE(&mh->dh_fc),
		0,                       //mh->dh_duration,
		ETHER_ADDR(mh->mh_da),
		ETHER_ADDR(mh->mh_sa),
		mh->mh_type);            //GET_ETHER_TYPE(mh->dh_body));
} // end formant_mac_WaveLAN1

void
CMUTrace::format_mac_802_3(Packet *p, const char *why, int offset)
{
#if 0
        printf("formatting 802.3 mac tracer\n");
#endif
	struct hdr_cmn *ch = HDR_CMN(p);
	struct hdr_ip *ih = HDR_IP(p);
	struct hdr_mac802_3 *mh = HDR_MAC3(p);
	char op = (char) type_;

	if(tracetype == TR_ROUTER && type_ == SEND) {
		if(src_ != ih->src_)
			op = FWRD;
	}

#ifdef LOG_POSITION
        double x = 0.0, y = 0.0, z = 0.0;
        node_->getLoc(&x, &y, &z);
#endif

	sprintf(wrk_ + offset,
#ifdef LOG_POSITION
		"%c %.9f %d (%6.2f %6.2f) %3s %4s %d %s %d [%x %x %x %x %x] ",
#else
		"%c %.9f _%d_ %3s %4s %d %s %d [%x %x %x %x %x] ",
#endif
		op,
		Scheduler::instance().clock(),
                src_,                           // this node
#ifdef LOG_POSITION
                x,
                y,
#endif
		tracename,
		why,

                ch->uid(),                      // identifier for this event
		pt_names[ch->ptype()],
		ch->size(),
                
		0,                       //GET2BYTE(&mh->dh_fc),
		0,                       //mh->dh_duration,
		ETHER_ADDR(mh->mh_da),
		ETHER_ADDR(mh->mh_sa),
		mh->mh_type);            //GET_ETHER_TYPE(mh->dh_body));
}

void
CMUTrace::format_mac_802_11(Packet *p, const char *why, int offset)
{
        struct hdr_cmn *ch = HDR_CMN(p);
	struct hdr_ip *ih = HDR_IP(p);
	struct hdr_mac802_11 *mh = HDR_MAC11(p);
	char op = (char) type_;

	if(tracetype == TR_ROUTER && type_ == SEND) {
		if(src_ != ih->src_)
			op = FWRD;
	}

	MobileNode* thisnode = MobileNode::get_node_by_address(src_);
	double energy = -1;
	if (thisnode) {
	    if (thisnode->energy_model()) {
		    energy = thisnode->energy_model()->energy();
	    }
	}

	// extract the type, subtype of MAC frame this is
	char fc_type, fc_subtype;
	switch (mh->dh_fc.fc_type) {
	case MAC_Type_Management: 
	    fc_type = 'm'; 
	    switch (mh->dh_fc.fc_subtype) {
	    case MAC_Subtype_Beacon:
		fc_subtype = 'b';
		break;
	    case MAC_Subtype_ATIM:
		fc_subtype = 'a';
		break;
	    default:
		fc_subtype = '?';
		break;
	    }
	    break;
	case MAC_Type_Control: 
	    fc_type = 'c';
	    switch(mh->dh_fc.fc_subtype) {
	    case MAC_Subtype_RTS:
		fc_subtype = 'r';
		break;
	    case MAC_Subtype_CTS:
		fc_subtype = 'c';
		break;
	    case MAC_Subtype_ACK:
		fc_subtype = 'a';
		break;
	    default:
		fc_subtype = '?';
	    }
	    break;
	case MAC_Type_Data: 
	    fc_type = 'd';
	    switch (mh->dh_fc.fc_subtype) {
	    case MAC_Subtype_Data:
		fc_subtype = 'd';
		break;
	    default:
		fc_subtype = '?';
	    }
	    break;
	case MAC_Type_Reserved: 
	    fc_type = 'r'; 
	    break;
	default:
	    fc_type = '?';
	}

#ifdef LOG_POSITION
        double x = 0.0, y = 0.0, z = 0.0;
        node_->getLoc(&x, &y, &z);
#endif

	sprintf(wrk_ + offset,
#ifdef LOG_POSITION
		"%c %.9f %d (%6.2f %6.2f) %3s %4s %d %s %d [%x %x %x %x %x %c %c %.4fJ]",

#else
		"%c %.9f _%d_ %3s %4s %d %s %d [%x %x %d %d %x %c %c %.4fJ]",
#endif
		op,
		Scheduler::instance().clock(),
                src_,                           // this node
#ifdef LOG_POSITION
                x,
                y,
#endif
		tracename,
		why,

                ch->uid(),                      // identifier for this event
		pt_names[ch->ptype()],
		ch->size(),

		GET2BYTE(&mh->dh_fc),
		mh->dh_duration,
		ETHER_ADDR(mh->dh_da),
		ETHER_ADDR(mh->dh_sa),
		GET_ETHER_TYPE(mh->dh_body),
		fc_type, // 802.11 frame type
		fc_subtype, // 802.11 frame subtype
		energy);
}

void
CMUTrace::format_mac_null(Packet *p, const char *why, int offset)
{
    struct hdr_cmn *ch = HDR_CMN(p);
    struct hdr_ip *ih = HDR_IP(p);
    struct hdr_macnull  *mh = (struct hdr_macnull *)p->access(off_mac_);

    char op = (char) type_;

    if(tracetype == TR_ROUTER && type_ == SEND) {
	if(src_ != ih->src_)
		op = FWRD;
    }

#ifdef LOG_POSITION
        double x = 0.0, y = 0.0, z = 0.0;
        node_->getLoc(&x, &y, &z);
#endif

	sprintf(wrk_ + offset,
#ifdef LOG_POSITION
		"%c %.9f %d (%6.2f %6.2f) %3s %4s %d %s %d [%x %x %x %x %x] ",

#else
		"%c %.9f _%d_ %3s %4s %d %s %d [%x %x %x %x %x] ",
#endif
		op,
		Scheduler::instance().clock(),
                src_,                           // this node
#ifdef LOG_POSITION
                x,
                y,
#endif
		tracename,
		why,

                ch->uid(),                      // identifier for this event
		pt_names[ch->ptype()],
		ch->size(),

		GET2BYTE(&mh->dh_fc),
		mh->dh_duration,
		ETHER_ADDR(mh->dh_da),
		ETHER_ADDR(mh->dh_sa),
		GET_ETHER_TYPE(mh->dh_body));
}


void
CMUTrace::format_ip(Packet *p, int offset)
{
        struct hdr_cmn *ch = HDR_CMN(p);
	struct hdr_ip *ih = HDR_IP(p);

	sprintf(wrk_ + offset, "------- [%d:%d %d:%d %d %d] ",
		ih->src_, ih->sport_,
		ih->dst_, ih->dport_,
		ih->ttl_, (ch->next_hop_ < 0) ? 0 : ch->next_hop_);
}

void
CMUTrace::format_arp(Packet *p, int offset)
{
	struct hdr_arp *ah = HDR_ARP(p);

	sprintf(wrk_ + offset,
		"------- [%s %d/%d %d/%d]",
		ah->arp_op == ARPOP_REQUEST ?  "REQUEST" : "REPLY",
		ah->arp_sha,
		ah->arp_spa,
		ah->arp_tha,
		ah->arp_tpa);
}

void
CMUTrace::format_dsr(Packet *p, int offset)
{
	hdr_sr *srh = (hdr_sr*)p->access(off_sr_);
	sprintf(wrk_ + offset, 
		"%d [%d %d %d] [%d %d %d %d->%d] [%d %d %d %d->%d]",
		srh->num_addrs(),

		srh->route_request(),
		srh->rtreq_seq(),
		srh->max_propagation(),

		srh->route_reply(),
		srh->rtreq_seq(),
		srh->route_reply_len(),
		// the dest of the src route
		srh->reply_addrs()[0].addr,
		srh->reply_addrs()[srh->route_reply_len()-1].addr,

		srh->route_error(),
		srh->num_route_errors(),
		srh->down_links()[srh->num_route_errors() - 1].tell_addr,
		srh->down_links()[srh->num_route_errors() - 1].from_addr,
		srh->down_links()[srh->num_route_errors() - 1].to_addr);
}

void
CMUTrace::format_aodv(Packet *p, int offset)
{
        struct hdr_aodv *ah = HDR_AODV(p);
        struct hdr_aodv_request *rq = HDR_AODV_REQUEST(p);
        struct hdr_aodv_reply *rp = HDR_AODV_REPLY(p);

        switch(ah->ah_type) {
        case AODVTYPE_RREQ:
		sprintf(wrk_ + offset,
			"[0x%x %d %d [%d %d] [%d %d]] (REQUEST)",
			rq->rq_type,
                        rq->rq_hop_count,
                        rq->rq_bcast_id,
                        rq->rq_dst,
                        rq->rq_dst_seqno,
                        rq->rq_src,
                        rq->rq_src_seqno);
                break;

        case AODVTYPE_RREP:
        case AODVTYPE_UREP:
        case AODVTYPE_HELLO:
		sprintf(wrk_ + offset,
			"[0x%x %d [%d %d] %d] (%s)",
			rp->rp_type,
                        rp->rp_hop_count,
                        rp->rp_dst,
                        rp->rp_dst_seqno,
                        rp->rp_lifetime,
                        rp->rp_type == AODVTYPE_RREP ? "REPLY" :
                        (rp->rp_type == AODVTYPE_UREP ? "UNSOLICITED REPLY" :
                         "HELLO"));
                break;

        default:
                fprintf(stderr,
                        "%s: invalid AODV packet type\n", __FUNCTION__);
                abort();
        }
}


void
CMUTrace::format_msg(Packet *, int)
{

}

void
CMUTrace::format_imep(Packet *p, int offset)
{
	struct hdr_imep *im = HDR_IMEP(p);

#define U_INT16_T(x)	*((u_int16_t*) &(x))

	sprintf(wrk_ + offset,
		"[%c %c %c 0x%04x] ",
		(im->imep_block_flags & BLOCK_FLAG_ACK) ? 'A' : '-',
		(im->imep_block_flags & BLOCK_FLAG_HELLO) ? 'H' : '-',
		(im->imep_block_flags & BLOCK_FLAG_OBJECT) ? 'O' : '-',
		U_INT16_T(im->imep_length));

#undef U_INT16_T
}


void
CMUTrace::format_tora(Packet *p, int offset)
{
        struct hdr_tora *th = HDR_TORA(p);
	struct hdr_tora_qry *qh = HDR_TORA_QRY(p);
	struct hdr_tora_upd *uh = HDR_TORA_UPD(p);
	struct hdr_tora_clr *ch = HDR_TORA_CLR(p);

	switch(th->th_type) {

	case TORATYPE_QRY:
		sprintf(wrk_ + offset, "[0x%x %d] (QUERY)",
			qh->tq_type, qh->tq_dst);
		break;

	case TORATYPE_UPD:
		sprintf(wrk_ + offset,
			"[0x%x %d (%f %d %d %d %d)] (UPDATE)",
			uh->tu_type,
			uh->tu_dst,
			uh->tu_tau,
			uh->tu_oid,
			uh->tu_r,
			uh->tu_delta,
			uh->tu_id);
		break;

	case TORATYPE_CLR:
		sprintf(wrk_ + offset, "[0x%x %d %f %d] (CLEAR)",
			ch->tc_type,
			ch->tc_dst,
			ch->tc_tau,
			ch->tc_oid);
		break;
	}
}

void
CMUTrace::format_span(Packet *p, int offset)
{
  hdr_ip *iph = (hdr_ip *) p->access(off_ip_);
  hdr_cmn *cmnh = (hdr_cmn *) p->access(off_cmn_);
  hdr_span *spanh = (hdr_span *) p->access(off_span_);
  sprintf(wrk_ + offset, 
	  "span %d -> %d, type %d, ttl %d, via %d (coord %d)", 
	  iph->src(), iph->dst(), spanh->type, iph->ttl(), cmnh->next_hop_,
	  spanh->is_coordinator);
}

void
CMUTrace::format_tcp(Packet *p, int offset)
{
	struct hdr_cmn *ch = HDR_CMN(p);
	struct hdr_tcp *th = HDR_TCP(p);

	sprintf(wrk_ + offset,
		"[%d %d] %d %d",
		th->seqno_,
		th->ackno_,
		ch->num_forwards(),
		ch->opt_num_forwards());
}

void
CMUTrace::format_rtp(Packet *p, int offset)
{
	struct hdr_cmn *ch = HDR_CMN(p);
	struct hdr_rtp *rh = HDR_RTP(p);

	int res = snprintf(wrk_ + offset,
			   sizeof(wrk_) - offset,
			   "[%d] %d %d",
			   rh->seqno_,
			   ch->num_forwards(),
			   ch->opt_num_forwards());
	assert(res > -1);
}

void
CMUTrace::format_ntrwkld(Packet *p, int offset)
{
	struct ping_data *pd = (struct ping_data *) p->accessdata();
	char *type;
	switch (pd->type) {
	case ECHO_REQUEST:
		type = "REQ";
		break;
	case ECHO_REPLY:
		type = "REP";
		break;
	default:
		type = "unknown";
	}

	sprintf(wrk_ + offset,"%s %d %.9f", type, pd->seqno, pd->timestamp);
}


void CMUTrace::format(Packet* p, const char *why)
{
	hdr_cmn *ch = HDR_CMN(p);
	int offset = 0;

	/*
	 * Log the MAC Header
	 */
	switch(mactype) {
	  
	case MAC_TYPE_802_11:
	  format_mac_802_11(p, why, offset);
	  break;

	case MAC_TYPE_802_3:
	  format_mac_802_3(p, why, offset);
	  break;

	case MAC_TYPE_WAVELAN1:
	  format_mac_WaveLAN1(p, why, offset);
	  break;

	case MAC_TYPE_NULL:
	  format_mac_null(p,why,offset);
	  break;

	default:
	  assert("Missing MAC_TYPE `#define`" == 0); 
	}

	offset = strlen(wrk_);

	switch(ch->ptype()) {

	case PT_MAC:
		break;

	case PT_ARP:
		format_arp(p, offset);
		break;

	default:
		format_ip(p, offset);
		offset = strlen(wrk_);

		switch(ch->ptype()) {

		case PT_TORA:
			format_tora(p, offset);
			break;

		case PT_IMEP:
			format_imep(p, offset);
			break;

		case PT_SPAN:
		  	format_span(p, offset);
			break;

		case PT_DSR:
			format_dsr(p, offset);
			break;

		case PT_MESSAGE:
			format_msg(p, offset);
			break;

		case PT_TCP:
		case PT_ACK:
			format_tcp(p, offset);
			break;

		case PT_CBR:
			format_rtp(p, offset);
			break;

                case PT_AODV:
                        format_aodv(p, offset);
                        break;

		case PT_LIVE:
		        break;
			
		case PT_NTRWKLD:
		        format_ntrwkld(p, offset);
			break;

		default:
			fprintf(stderr, "%s - invalid packet type (%s).\n",
				__PRETTY_FUNCTION__, pt_names[ch->ptype()]);
			exit(1);
		}
	}
}

int
CMUTrace::command(int argc, const char*const* argv)
{
        if(argc == 3) {
                if(strcmp(argv[1], "node") == 0) {
                        node_ = (MobileNode*) TclObject::lookup(argv[2]);
                        if(node_ == 0)
                                return TCL_ERROR;
                        return TCL_OK;
                }
        }
	return Trace::command(argc, argv);
}

void
CMUTrace::recv(Packet *p, Handler *h)
{
        struct hdr_ip *ih = HDR_IP(p);
	//        struct hdr_cmn *ch = HDR_CMN(p);

        assert(initialized());
	
        /*
         * Agent Trace "stamp" the packet with the optimal route on
         * sending.
         */
        if(tracetype == TR_AGENT && type_ == SEND) {
                assert(src_ == ih->src_);
                God::instance()->stampPacket(p);
        }
#if 0
        /*
         * When the originator of a packet drops the packet, it may or may
         * not have been stamped by GOD.  Stamp it before logging the
         * information.
         */
        if(src_ == ih->src_ && type_ == DROP) {
                God::instance()->stampPacket(p);
        }
#endif
	format(p, "---");
	dump();
	if(target_ == 0)
		Packet::free(p);
	else
		send(p, h);
}

void
CMUTrace::recv(Packet *p, const char* why)
{
        assert(initialized() && type_ == DROP);
#if 0
        /*
         * When the originator of a packet drops the packet, it may or may
         * not have been stamped by GOD.  Stamp it before logging the
         * information.
         */
        if(src_ == ih->src_) {
                God::instance()->stampPacket(p);
        }
#endif
	format(p, why);
	dump();
	Packet::free(p);
}

