#include <stdlib.h>
#include <math.h>
#include <string.h>
#include "tclcl.h"
#include "packet.h"
#include "ip.h"
#include "tcp.h"
#include "tcp-sink.h"
#include "agent.h"
#include "flags.h"
#include "random.h"
#include <stdio.h>
#include <stdarg.h>
#include "template.h"
#include "timer-handler.h"
#include "mac.h"
#include <assert.h>
#include "cmu/priqueue.h"

#define NUM_ENTRIES 1000
#define HELLO 1
#define RREQ  2
#define RREP  3
#define UREP  4  // unsolicited rrep - when a broken link is detected
#define HELLO_SIZE 50
#define RREQ_SIZE 50
#define RREP_SIZE 50 
#define UREP_SIZE 50
#define NUM_ACTIVE_NEIGHBORS  NUM_ENTRIES
#define NUM_BROADCASTS        3*NUM_ENTRIES // ... HM 
#define INFINITY              NUM_ENTRIES*2
#define NUM_REPLIES           NUM_ENTRIES*3
#define ACTIVE_NEIGHBOR_TIME  300 
#define NUM_EVENTS            2000
#define DELETE_EXPIRED        2
#define UPDATE_MESSAGES       50
#define TRUE                  1 
#define FALSE                 0
#define MAX_QUEUE_LENGTH      5
#define JIT                   0.1


#define HELLO_INTERVAL       1
#define JIT_PERIODIC         0.25*HELLO_INTERVAL
#define JIT_BASE             0.75*HELLO_INTERVAL

#define ACTIVE_ROUTE_TIMEOUT  300
#define ALLOWED_HELLO_LOSS      2
#define BAD_LINK_LIFETIME       3
#define BCAST_ID_SAVE           3  // hm **********
#define NETWORK_DIAMETER      100
#define NODE_TRAVERSAL_TIME     0.4 // that would make a difference when errors are introduced
#define MY_ROUTE_TIMEOUT      600
#define REV_ROUTE_LIFE       3  
#define RREP_WAIT_TIME       3*NODE_TRAVERSAL_TIME*NETWORK_DIAMETER 
#define PREQ_RETRIES         3

static inline double jitter(double max, int make_random_) {
  return (make_random_ ? max*(((double)(random()&0xFFFF))/65536.0) : 0);
}


class AODV_Agent;

class UPDATE_Timer : public TimerHandler {
public:
  UPDATE_Timer(AODV_Agent* a) : TimerHandler()
  { a_=a; }
protected:
  virtual void expire(Event* e);
  AODV_Agent* a_;
};

struct active_neighbor{ 
  int id;
  double install_time;
};

struct routing_table {
  nsaddr_t destination;
  int seqnum;
  int hop_count;
  nsaddr_t next_hop;
  double lifetime;
  struct active_neighbor active_neighbors[NUM_ACTIVE_NEIGHBORS];
  int end_an_list;
  PacketQueue *q;
  int queue_length;
};

// saves the (broadcast_id, source) pairs so that if the agent receives a recenly rebroadcast by it request, it would 
// discard it -- that avoids routing loops of the RREQs

struct broadcast{
  int id;
  nsaddr_t dest;
  double lifetime;
};

// used to keep track of replies to an RREQ

struct reply_info{
  nsaddr_t dest;
  int num_retries;
  double install_time;
};

// handler for the HELLO messages

class UpdateHandler : public Handler {
public:
  UpdateHandler(AODV_Agent* a): a_(a) {}
  void handle(Event*);
protected:
  AODV_Agent* a_;
};

// handler for RREQ messages;  keeps track of when to retry a RREQ if a reply has not been received for a while

class RequestHandler : public Handler {
public:
  RequestHandler(AODV_Agent* a): a_(a) {}
  void handle(Event*);
protected:
  AODV_Agent* a_;
};


class AODV_Agent : public Agent {
public:
  AODV_Agent();
  struct routing_table rts[NUM_ENTRIES];
  int end_of_table;

  void recv(Packet *, Handler *);
  void route_data(Packet *);
  void send_hello();
  void send_request(nsaddr_t);
  void handle_request(Packet *);
  void handle_reply(Packet *);
  void handle_unsolicited_reply(Packet *);
  void send_unsolicited_reply(int);
  void add_to_active_neighbors_list(nsaddr_t neighbor,nsaddr_t dest);
  int add_route(nsaddr_t,int,int,nsaddr_t,double);
  int delete_route(int); 
  void link_break(Packet*);
  void update_route(int,int,int,nsaddr_t,double);
  int DropBroadcast(nsaddr_t, int);
  void process_hello(Packet*);
  int route(nsaddr_t);
  int in_reply_queue(nsaddr_t);
  void purge_queues(nsaddr_t id);
  void print_routing_table(); 
  int command(int,const char*const*);
  void timeout(int);

  //variables

  nsaddr_t node_;
  //nsaddr_t sport_;
  nsaddr_t portID_;
  double interval_;
  UPDATE_Timer update_timer_;
  UpdateHandler handle_broken_link;
  RequestHandler handle_request_retry;
  int seqnum;
  int req_id;
  struct broadcast broadcast_ids[NUM_BROADCASTS];
  int bi_last;
  int event_index[NUM_EVENTS];
  int last_event;
  int next_event; 
  struct reply_info replies[NUM_REPLIES];
  int next_reply;
  int last_reply;
  int make_random_;
  PriQueue* ifqueue;
};


void UpdateHandler::handle(Event *e){
  double now;
  now=Scheduler::instance().clock();
  
  // if route has an INFINITY metric and there was an event scheduled for it, 
  // then we have scheduled it for deletion

  if(a_->rts[a_->event_index[a_->next_event]].hop_count==INFINITY){
    a_->delete_route(a_->event_index[a_->next_event]);
  }
  // otherwise if it the route is expired: 1.  inc seq_num. because the set of 
  // neighbors has changed 2. send an unsolicited reply to the active neighbors for 
  // that route
  else if((a_->rts[a_->event_index[a_->next_event]].lifetime<now)&&
	  (a_->rts[a_->event_index[a_->next_event]].lifetime>0))
    {
      a_->seqnum++;
      a_->send_unsolicited_reply(a_->event_index[a_->next_event]);
  }
  a_->next_event++;

  if(a_->next_event==NUM_EVENTS){
    a_->next_event=0;
  }
  delete e;
}

void RequestHandler::handle(Event *e){
 
  if(e) delete e;
  // if we have gotten a reply to our RREQ or we have retried 3 times, stop retrying
  if((a_->replies[a_->next_reply].install_time>0)||
     (a_->replies[a_->next_reply].num_retries>2)){
    a_->next_reply++;
    if((a_->next_reply-a_->last_reply)==1){
      a_->next_reply=0;
      a_->last_reply=-1;
      //printf("reply queue is empty\n");
    }
    else if(a_->next_reply==NUM_REPLIES){
      a_->next_reply=0;
    }
    return;
  }
  // if we haven't gotten a reply yet and we have retired the request less than 3 times, 
  // retry again: 1. enque event for this RREQ at the end of the queue of request events
  // 2. inc retry 3. send request 4. schedule an event to "keep track of the request"
  else{
    Scheduler& s=Scheduler::instance();
    if(a_->last_reply!=a_->next_reply){
      a_->last_reply++;
      if(a_->last_reply==NUM_REPLIES){
	a_->last_reply=0;
      }
      a_->replies[a_->last_reply].dest=
	a_->replies[a_->next_reply].dest;
      a_->replies[a_->last_reply].install_time=
	a_->replies[a_->next_reply].install_time;    
      a_->replies[a_->last_reply].num_retries=1+
	a_->replies[a_->next_reply].num_retries;
      a_->next_reply++;
      if(a_->next_reply==NUM_REPLIES){
	a_->next_reply=0;
      }
    } 
      a_->send_request(a_->replies[a_->last_reply].dest);
      s.schedule(&(a_->handle_request_retry), 
		 new Event(),RREP_WAIT_TIME);
  }
}



void AODV_Agent::send_unsolicited_reply(int index) {
   nsaddr_t dest;
   double now;
   char buff[UREP_SIZE];
   int j;
   Scheduler& s=Scheduler::instance();

   now=Scheduler::instance().clock();

   // this should not occur -- there is no longrer a route for a route that 
   // expired, i.e. it was deleted in the meantime 
   if(index==-1){
     printf("index is -1 in unsolicited route reply\n");
     return;  
   }
   // you don't delete a route that you have sent a request for and you don't
   // send unsolicited replies for it either
   if(in_reply_queue(rts[index].destination)!=-1){
     //    printf("attempting to delete a RREQ-ed route3\n");
     return;
   }

   dest=rts[index].destination;
   // checks for a bug that existed at one point
   if((dest==0)||(dest==-1)) {printf("ROUTE DELETED\n"); return;}
   rts[index].hop_count=INFINITY;
   rts[index].lifetime=0;  
   // schedule the route for deletion
   s.schedule(&handle_broken_link, new Event(), BAD_LINK_LIFETIME); 
   last_event++;
   if(last_event==NUM_EVENTS){
     last_event=0;
   }
   event_index[last_event]=index;  // save the index that corresponds to the
   // event in event_index so that when the event fires up, we know what route 
   // it was for

   j=rts[index].end_an_list;
   
   // send a message to all the active neighbors,that have not expired, about 
   // the broken route 
   while(j>-1){
     // if a neighbor has expired, take it off the list
     if(rts[index].active_neighbors[j].install_time<now){
       rts[index].active_neighbors[j].id=
	 rts[index].active_neighbors[rts[index].end_an_list].id;
       rts[index].active_neighbors[j].install_time=
	 rts[index].active_neighbors[rts[index].end_an_list].install_time;
       rts[index].end_an_list--;
     }
     else {
       Packet* p=allocpkt();
       
       hdr_ip* iph=(hdr_ip *)p->access(off_ip_);
       hdr_cmn* ch=(hdr_cmn *)p->access(off_cmn_);
       iph->dst()=rts[index].active_neighbors[j].id;
       iph->dport_=0xff;      
       ch->next_hop()=rts[index].active_neighbors[j].id;
       ch->addr_type()=AF_INET;
       ch->src_=sport_;
       // UREP L=0 hop_count dest_ip dest_seqno 
       sprintf(buff,"%d %d %d %d %d", UREP, 0,65535,dest,
	       rts[index].seqnum+1);
       p->init_data(UREP_SIZE, (unsigned char *)buff);
       ch->size_=40;
       assert(ch->next_hop()!=addr_);
       //assert(ch->next_hop() != iph->src_);
       s.schedule(target_,p,jitter(JIT,make_random_));
     }
     j--;
   }
}


// check if we have a route to a particular destination.  If so return its index 
// into the routing table array, else return -1
int AODV_Agent::route(nsaddr_t dest){
  int index=-1;
  int i;
  //  double now=0;
  // now=Scheduler::instance().clock();

  for(i=0;i<end_of_table+1;i++){
    if(rts[i].destination==dest){
      return i;
    }
  }
  return index;
}

void AODV_Agent::print_routing_table(){
  int i;
  double now=now=Scheduler::instance().clock();

  printf("\nROUTING TABLE FOR NODE %d\n", addr_);
  printf("--------------------------------------------------\n");
  for(i=0;i<end_of_table+1;i++){
    printf("dest: %d, seqno: %d, hop_count: %d, next_hop: %d, lifetime is %f\n",  
	   rts[i].destination, rts[i].seqnum, rts[i].hop_count, rts[i].next_hop, rts[i].lifetime);
    //   for(j=0;j<rts[i].end_an_list+1;j++){
    // printf("\nn_id: %d, ", rts[i].active_neighbors[j].id);
    //  printf("install time: %f\n", rts[i].active_neighbors[j].install_time);
    //}
  }
}

// add a new route to the routing table
int AODV_Agent::add_route(nsaddr_t dest, int dest_seqno, 
			  int num_hops, nsaddr_t next_hop,
			  double lifetime){
  double now=Scheduler::instance().clock();
  int rindex;
  

  if(((lifetime<0)||(lifetime==0))&&(dest_seqno!=-1)){
    printf("in add_route\n");
    rindex=1/0;
  }

  
  if(dest==addr_){
    printf("trying to add myself to route table\n");
    return -1;
  }
  rindex=in_reply_queue(dest);

  // you haven't gotten a route reply but you got a route to that destination 
  // so you count it as if you've gotten a RREP
  if(rindex!=-1){
    replies[rindex].install_time=now; 
  }
  end_of_table++;
  assert(end_of_table < NUM_ENTRIES);
  rts[end_of_table].destination=dest;
  rts[end_of_table].seqnum=dest_seqno;
  rts[end_of_table].hop_count=num_hops;
  rts[end_of_table].next_hop=next_hop;
  rts[end_of_table].end_an_list=-1;
  if(lifetime>0){
    rts[end_of_table].lifetime=lifetime+now;    
  }
  else {
    rts[end_of_table].lifetime=0;    
  }
  //  print_routing_table();
  return end_of_table;  
}



int AODV_Agent::delete_route(int index){
  //int i;
  //  float now=Scheduler::instance().clock();

  if(index==-1){
    printf("trying to delete a non-existent route\n");
  }

  if(rts[index].destination<1){
    printf("%d: trying to delete a < 1 route, index is %d, hop count is %d\n", addr_, index, rts[index].hop_count);
    assert(0);
  }
  // we've sent out a request for a broken route, but in the meantime we try to delete it b/c we got 
  // an unsol reply about it; we shouldn't be deleting it so we return
  if(in_reply_queue(rts[index].destination)!=-1){
    //    printf("attempting to delete a RREQ-ed route2\n");
    return;
  }

  //  printf("%d: DELETING route to %d\n", addr_,rts[index].destination);
  //  if(end_of_table==0){
    assert(rts[index].hop_count==INFINITY);
    rts[index].destination=-1;
    rts[index].hop_count=0;
    rts[index].lifetime=0;
    rts[index].end_an_list=-1;  
  if(rts[index].q){
      delete rts[index].q;
      rts[index].q=0;
      rts[index].queue_length=0;
    }
    return end_of_table;
    //} 


  /*

  rts[index].destination=rts[end_of_table].destination;
  rts[index].seqnum=rts[end_of_table].seqnum;
  rts[index].hop_count=rts[end_of_table].hop_count;
  rts[index].next_hop=rts[end_of_table].next_hop;
  rts[index].end_an_list=rts[end_of_table].end_an_list;
  rts[index].lifetime=rts[end_of_table].lifetime;
  for(i=0;i<rts[index].end_an_list+1;i++){
    rts[index].active_neighbors[i].id=
      rts[end_of_table].active_neighbors[i].id;
    rts[index].active_neighbors[i].install_time=
      rts[end_of_table].active_neighbors[i].install_time;
  }
  if(rts[index].q){
    delete rts[index].q;
    rts[index].q=0;
    rts[index].queue_length=0;
  }
  rts[end_of_table].destination=-1;
  rts[end_of_table].hop_count=0;
  rts[end_of_table].lifetime=0;
  end_of_table--;

  */
}

void AODV_Agent::update_route(int index,int seqnum,int metric,
			      nsaddr_t next_hop,double lifetime){
  double now=Scheduler::instance().clock();
  Scheduler& s=Scheduler::instance();
  int rindex;

  rindex=in_reply_queue(rts[index].destination);
 // you haven't gotten a route reply but you got a route to that destination 
  // so you count it as if you've gotten a RREP
  if(rindex!=-1){
    replies[rindex].install_time=now;
  }
  // if we thought that route was broken and now we are updating, so its no
  // longer broken, we want to send the packets that we have queued in the 
  // meantime
  if((rts[index].hop_count==INFINITY)&&(rts[index].q)){
      while(1){
	Packet *p1;
	p1=rts[index].q->deque();
	
	if(!p1) { 
	  //printf("no more packets to deque\n");
	  delete rts[index].q;
	  rts[index].q=0;
	  rts[index].queue_length=0;
	  break;
	}
	rts[index].queue_length--;
	hdr_cmn* cmn1 = (hdr_cmn*)p1->access(off_cmn_);
	//	hdr_ip* iph1 = (hdr_ip*)p1->access(off_ip_);
	cmn1->next_hop()=next_hop;  // SH
	cmn1->addr_type()=AF_INET;
	cmn1->src_=addr_;
	assert(cmn1->next_hop()!=addr_);
	s.schedule(target_,p1,jitter(JIT, 0));
      }
  }

  // update the route
  if(rts[index].seqnum<seqnum){
    rts[index].seqnum=seqnum;
    rts[index].hop_count=metric;
    rts[index].next_hop=next_hop;
    rts[index].lifetime=now+lifetime;
  }
  else if(rts[index].seqnum==seqnum){
    if(rts[index].hop_count>metric){
      rts[index].hop_count=metric;
      rts[index].next_hop=next_hop;
    }
    if(rts[index].lifetime<now+lifetime){
      rts[index].lifetime=now+lifetime;
    }
  }
 
  else if(seqnum==-1) {
    printf("seqnum is -1 -- oopsie\n");
  }

  //  print_routing_table();
}

// this function checks if we have received the same broadcast and rebroadcasted
// it in the recent past, if so it returns true and we'll drop it

int AODV_Agent::DropBroadcast(nsaddr_t dest, int broadcast_id){
  double now=Scheduler::instance().clock();
  int i;

  // clean up the expired broadcasts first
  for(i=0;i<bi_last+1;i++){
    if(broadcast_ids[i].lifetime<now){
      broadcast_ids[i].id=broadcast_ids[bi_last].id;
      broadcast_ids[i].dest=broadcast_ids[bi_last].dest;
      broadcast_ids[i].lifetime=
	broadcast_ids[bi_last].lifetime;      
      bi_last--;
    }
  }
  // have we seen this broadcast recently ?
  for(i=0;i<bi_last+1;i++){
    if((broadcast_ids[i].dest==dest)&&
       (broadcast_ids[i].id==broadcast_id)){
      return TRUE;
    }
  }
  return FALSE;
}

 
void AODV_Agent::recv(Packet *p, Handler *h){
  hdr_cmn* ch=(hdr_cmn *)p->access(off_cmn_);
  hdr_ip* iph=(hdr_ip *)p->access(off_ip_);
  unsigned char buff[UPDATE_MESSAGES];
  int size_of_data, mesage;

  
    size_of_data=p->accessdatalen();
    if((size_of_data>0)||(ch->ptype_==10)){
      p->accessdata((unsigned char *)buff);
      sscanf((char *)buff,"%d",&mesage);
      if(mesage==HELLO){  // received a hello message 
	process_hello(p);
	return;
      }
      else if(mesage==RREP){
	handle_reply(p);
	return;
      }
      else if(mesage==RREQ){
	handle_request(p);
	return;
      }
      else if(mesage==UREP){
	handle_unsolicited_reply(p);
	return;
      }
      else printf("UNDETERMINED, i.e. sth got garbled\n");
    }
    route_data(p);  
}

// process a hello message
void AODV_Agent::process_hello(Packet* p){
  int lt,index,type, dest, dest_seqnum,hop_count;
  float lifetime;
  Scheduler& s=Scheduler::instance();
  char buff[HELLO_SIZE];
  //  hdr_ip* iph = (hdr_ip*)p->access(off_ip_);
  //  double now=0;
  //now=Scheduler::instance().clock();
  p->accessdata((unsigned char *)buff);

  sscanf(buff,"%d %d %d %d %d", &type,&dest,&dest_seqnum,&hop_count,
	 &lt); 
  lifetime=(float)lt;
  hop_count++;
  if(dest==addr_){
    // printf("got my own hello\n");  // possible with current mac layer?
    Packet::free(p);
    return;
  }
  index=route(dest);

  // we have no route to that neighbor yet
  if(index==-1){
    index=add_route(dest, dest_seqnum,hop_count,dest,lifetime);
    if(index==-1){
      Packet::free(p);
      return;
    }  
    seqnum++;  // change in neighbors => should increment own seqnum

    s.schedule(&handle_broken_link,    // sched an event to keep track of lost HELLO's
	       new Event(),lifetime);  // and signal broken link when several haven't 
    last_event++;                      // been received
    if(last_event==NUM_EVENTS){
      last_event=0;
    }
    event_index[last_event]=index;
  }
  // we have a route to the sender of the hello message, so we update it
  else {
    update_route(index,dest_seqnum, hop_count,dest,lifetime);
    s.schedule(&handle_broken_link,                          // sched an event to keep track of lost HELLO's
	       new Event(),lifetime);                        // and signal broken link when several haven't  
    last_event++;                                            // been received
    if(last_event==NUM_EVENTS){
      last_event=0;
    }
    event_index[last_event]=index;     // keeps track of what index corresponds to what event
  }
  Packet::free(p);
}

// sending a beacon
void AODV_Agent::send_hello(){
  char buff[HELLO_SIZE];
  //  double now=0;
  Scheduler& s=Scheduler::instance();
  //now=Scheduler::instance().clock(
  //printf("%d: Sending hello to %d at %f\n", addr_, IP_BROADCAST,now);

  Packet* p=allocpkt();

  hdr_ip* iph=(hdr_ip *)p->access(off_ip_);
  hdr_cmn* ch=(hdr_cmn *)p->access(off_cmn_);
  iph->dst()=IP_BROADCAST;
  ch->next_hop()=IP_BROADCAST;
  ch->addr_type()=AF_INET;
  ch->src_=addr_;

  // type node address, seqno, hop count, lifetime
  int jj=(1+ALLOWED_HELLO_LOSS)*HELLO_INTERVAL;
  sprintf(buff, "%d %d %d %d %d",HELLO,addr_,seqnum,0,jj);

  p->init_data(HELLO_SIZE,(unsigned char *)buff);

  ch->size_=40;
  assert(ch->next_hop()!=addr_);
  s.schedule(target_,p,JIT_BASE+jitter(JIT_PERIODIC,make_random_));
}


// add neighbor to the list of active neighbors for a particular destination
void AODV_Agent::add_to_active_neighbors_list(nsaddr_t neighbor,nsaddr_t dest){
  double now;
  int index,j;
  
  if(neighbor==addr_){
    printf("%d: trying to add myself as a neighbor -- hm\n", addr_);
  }

  index=route(dest);
  if(index==-1){
    // no route yet
    printf("no route yet\n");  // this can't happen
    assert(0);
    return;
  }
  else{
    now=Scheduler::instance().clock();
    j=rts[index].end_an_list;
    while(j>-1){  // if neighbor is already registered, just increase its lifetime
      if(rts[index].active_neighbors[j].id==neighbor){
	rts[index].active_neighbors[j].install_time=now+
	  ACTIVE_NEIGHBOR_TIME;
	return;
      }
      j--;
    }
    // neighbor has not been registered yet, so register it
    rts[index].end_an_list++;

    assert(rts[index].end_an_list < NUM_ENTRIES);

    rts[index].active_neighbors[rts[index].end_an_list].id=neighbor;
    rts[index].active_neighbors[rts[index].end_an_list].install_time=now+ACTIVE_NEIGHBOR_TIME;
  }
}

// purge queues from packets to a particular destination that the mac layer considers
// unreachable
void AODV_Agent::purge_queues(nsaddr_t id){

  Scheduler &s = Scheduler::instance();
  Packet *head = 0, *p, *np = 0;
  LL *ll;
  PriQueue* pq;
  int index;

  
  while((p = ifqueue->filter(id))) {
    p->next_ = head;
    head = p; 
  }
  
  /* ll = (LL*) ifqueue->target();
  assert(ll);
  
  pq = (PriQueue*) ll->sendtarget();
  assert(pq);
  
  while((p = pq->filter(id))) {
    p->next_ = head;
    head = p;
  }
  */
  for(p = head; p; p = np) {  
    np = p->next_;
    struct hdr_cmn *cmn;
    cmn=(hdr_cmn *)p->access(off_cmn_);

    if((p->accessdatalen()>0)||(cmn->ptype()==10)){  // message type packets should 
                                                   // not even be here -- they have
     printf("a broadcast in purge queues!\n");     // no callback on them
    }
    route_data(p); 
  }
}       

// callback for packets that the mac layer cannot deliver

void rt_failed_callback(Packet *p, void *arg){

   ((AODV_Agent*) arg)->link_break(p);

}

void AODV_Agent::link_break(Packet* p){
  struct hdr_cmn *cmn;
  int index;
  nsaddr_t dest;
  Scheduler& s=Scheduler::instance();

  cmn=(hdr_cmn *)p->access(off_cmn_);
  struct hdr_ip *iph = (hdr_ip*)p->access(off_ip_);

  if(cmn->next_hop() == addr_) {
    int x=1/0; 
    printf("next_hop=addr\n"); 
    drop(p);return;
  }

  dest=cmn->next_hop();
  
  // broadcasts are not queued and in fact they shouldn't even be here because 
  // they have no callback on them
  if(dest==IP_BROADCAST)
    {
      printf("broadcast in link_break\n");
      drop(p);
      return;
    }
  index=route(dest);
  // check for a bug that existed at one point
  if((index!=-1)&&(rts[index].destination<1)) {printf("dest is < 0\n"); return;}

  if(index!=-1){
    // send an unsilicited reply to all active neughbors unless you've already 
    // done so in which case hop_count will be INFINITY
    if(rts[index].hop_count!=INFINITY){
      send_unsolicited_reply(index);
    }
  }
  purge_queues(dest);
}


void AODV_Agent::route_data(Packet* p){
  hdr_cmn* cmn=(hdr_cmn *)p->access(off_cmn_);
  hdr_ip* iph=(hdr_ip *)p->access(off_ip_);
  nsaddr_t dest;
  int index;
  double now=Scheduler::instance().clock();
  Scheduler& s=Scheduler::instance();

  //printf("%d: routing data, now is %f\n", addr_, now);

  dest=iph->dst();  // extracting the destination node (i.e. w/out the agent address)
  int source=iph->src();

  index=route(dest); // route returns -1 if no route 
  
  int index3=-1;
  if((index!=-1)&&((source)!=(addr_))&&(cmn->src_!=addr_)){
    index3=route(cmn->src_);
     if(index3==-1){
       //    printf("No route to neighbor before adding it to active neighbours, so we add a route to it\n");
       add_route(cmn->src_,-1,1,cmn->src_,HELLO_INTERVAL);
     }
     else{
       rts[index3].hop_count=1;
       rts[index3].lifetime=now+HELLO_INTERVAL;
       
     }
     add_to_active_neighbors_list(cmn->src_,dest);   // we can't be the dest
  }
   int index_nexthop=0;
   if(cmn->src_==addr_){  // packet dropped at the link layer
     index_nexthop=route(cmn->next_hop());
   }

  // if we have no route, or we have a route but its broken or expired
   if((index==-1)||((index!=-1)&&((rts[index].hop_count==INFINITY)
     ||(rts[index].lifetime<now)))){     

   if((source==addr_)&&(in_reply_queue(dest)==-1)){ // if we haven't sent out a request for that route yet

     // if we are the source of the packet and we have broken, expired or no route,
     // send out a RREQ

     if(index==-1){
	 index=add_route(dest,-1,0,0,0);  // adding a route for the proper functioning of
	 if(index==-1){                   // send_request which needs an index into the route
	   Packet::free(p);               // table that it will need when scheduling the request event
	   return;                          
	 }
	 rts[index].q=new PacketQueue;    // queue packets for that destination, so that 
	 rts[index].queue_length=0;       // if and when you get a route to it, you can send 
       }                                 // them
       
	 send_request(dest);  // send req has to be after adding the fake route!
   }
      if(!rts[index].q) {
        //printf("empty queue\n"); 
        rts[index].q=new PacketQueue;
        rts[index].queue_length=1;
	rts[index].q->enque(p);
      }
      else if(rts[index].queue_length==MAX_QUEUE_LENGTH){
          drop(p);
        }
        else{
          rts[index].q->enque(p);
          rts[index].queue_length++;
        }
   
   }  // a packet undeliverable by the mac layer has come back
   else if((index_nexthop==-1)||((index_nexthop!=-1)&&((rts[index_nexthop].hop_count==INFINITY)
						       ||(rts[index_nexthop].lifetime<now)))){     
     
     if((source==addr_)&&(in_reply_queue(cmn->next_hop())==-1)){ // if we haven't sent out a request for that route
       
       // if we are the source of the packet and we have broken, expired or no route,
       // send out a RREQ
       
       if(index_nexthop==-1){
	 index_nexthop=add_route(cmn->next_hop(),-1,0,0,0);  // adding a route for the proper functioning of
	 if(index_nexthop==-1){                   // send_request which needs an index_nexthop into the route
	   Packet::free(p);               // table that it will 
	   return;
	 }
	 rts[index_nexthop].q=new PacketQueue;    // queue packets for that destination, so that 
	 rts[index_nexthop].queue_length=0;       // if and when you get a route to it, you can send 
       }                                          // them
	 send_request(cmn->next_hop());  // send req has to be after adding the fake route!
     } 
     
     if(!rts[index_nexthop].q) {  // if no queue yet, create a queue
       rts[index_nexthop].q=new PacketQueue;
       rts[index_nexthop].queue_length=1;
       rts[index_nexthop].q->enque(p);
     }
      else{
        if(rts[index_nexthop].queue_length==MAX_QUEUE_LENGTH){
          drop(p);
        }
        else{
          rts[index_nexthop].q->enque(p);
          rts[index_nexthop].queue_length++;
        }
      }
   }
   else{
     assert(index!=-1);
     assert(rts[index].lifetime>now);
     assert(rts[index].hop_count!=INFINITY);
     cmn->next_hop()=rts[index].next_hop;
     cmn->addr_type()=AF_INET;
     cmn->src_=addr_;
     cmn->xmit_failure_=rt_failed_callback;
     cmn->xmit_failure_data_=(void*)this;
     assert(cmn->next_hop()!=addr_);
     s.schedule(target_,p,jitter(JIT, 0));
   }
}

// function that handles RREQ packets

void AODV_Agent::handle_request(Packet* p){
  int index,type, num_hops,broadcast_id, dest, dest_seqno,Type, 
    src_ip, src_seqno,BroadCast;
  double lifetime;
  nsaddr_t next_hop;

  double now=Scheduler::instance().clock();
  char buff[RREQ_SIZE];
  int index1;
  Scheduler& s=Scheduler::instance();
  hdr_ip* iph=(hdr_ip *)p->access(off_ip_);
  hdr_cmn* cmn=(hdr_cmn *)p->access(off_cmn_);
  //type Type hop_count broadcast_id dest dest_seqnum src_ip src_sno
  p->accessdata((unsigned char *)buff);
  sscanf(buff,"%d %d %d %d %d %d %d %d", &type, &Type, &num_hops, 
	  &broadcast_id,&dest,&dest_seqno, &src_ip,&src_seqno);
  if(src_ip==addr_){
    // got my own request because some neighbor rebroadcast it
    drop(p);
    return;
  }
  index=route(dest);
  int index_nexthop=0;
  if(index!=-1){
    index_nexthop=route(rts[index].next_hop);
  }
  BroadCast=FALSE;

  if(bi_last>-1){
    BroadCast=DropBroadcast(src_ip,broadcast_id);
  }
  if(BroadCast==TRUE){  // we've received the same broadcast within the past BCAST_ID_SAVE seconds
    drop(p);
    return;
  }  
  // assume we are not in our own routing table
  // no route or stale route or broken route or expired route or the next_hop for that 
  // route is unreachable

  else if((index_nexthop==-1)||((index==-1)&&(dest!=(addr_)))||
	  ((index!=-1)&&(rts[index].seqnum<dest_seqno+1))||
	   ((index!=-1)&&(rts[index].hop_count==INFINITY))||
	  ((index!=-1)&&(rts[index].lifetime<now)))  
    {   //stale route message to active neighbors is suggested by the paper but is not done here -- seems like
      assert(dest!=addr_);                         // unnecessary overhead and is not specified in the paper
      num_hops++;
      next_hop=cmn->src_;
      int index3=0;

      index3=route(next_hop);
      if(index3==-1){
	add_route(next_hop, -1,1, next_hop,HELLO_INTERVAL);
      }
      else{
	rts[index3].hop_count=1;
	rts[index3].lifetime=now+HELLO_INTERVAL;
      }

      // else update shouldn't be neecessary in the presence of beacons
      index1=route(src_ip);
      if(index1==-1){
	add_route(src_ip, src_seqno,num_hops-1, next_hop,
		  REV_ROUTE_LIFE);
      }
      else {
	rts[index1].next_hop=next_hop;
	if(rts[index1].lifetime<now+REV_ROUTE_LIFE){
	  rts[index1].lifetime=now+REV_ROUTE_LIFE;
	}
      }
      iph->src()=addr_;
      sprintf(buff, "%d %d %d %d %d %d %d %d", type, 0, num_hops, 
	      broadcast_id,dest,dest_seqno, src_ip,src_seqno);
      p->init_databuf((unsigned char *)buff);
      
      iph->ttl_--;
      cmn->src_=addr_;
      assert(cmn->next_hop()!=addr_);
      s.schedule(target_,p,jitter(JIT, make_random_));
    }
  // we are the dest or we have a fresh route 
  else {
    next_hop=cmn->src_;  
    int index3;
    index3=route(next_hop);
    if(index3==-1){
      add_route(next_hop,-1,1,next_hop,HELLO_INTERVAL);
    }
    else{
      rts[index3].hop_count=1;
      rts[index3].lifetime=now+HELLO_INTERVAL;
    }
    index1=route(src_ip);
    if(index1==-1){
      add_route(src_ip, src_seqno,num_hops, next_hop,ACTIVE_ROUTE_TIMEOUT);
    }
    else {  
      rts[index1].next_hop=next_hop;
      if(rts[index1].lifetime<now+ACTIVE_ROUTE_TIMEOUT){
	rts[index1].lifetime=now+ACTIVE_ROUTE_TIMEOUT;
      }
    }

    if(dest==addr_){
      dest_seqno=seqnum;  
      lifetime=MY_ROUTE_TIMEOUT;
    }
    else {
      num_hops=num_hops+rts[index].hop_count;
      dest_seqno=rts[index].seqnum;
      lifetime=rts[index].lifetime-now; 
      assert(lifetime>0);
      int index3=route(rts[index].next_hop); // do we absolutely need a route to the next hop?
      if(index3==-1){  // we 
	// this case is now handled as if we don't have a route --
	// for new measurements tho, that might need to be printed out or sth *****************
	printf("we do\n, %d", rts[index].next_hop);  // we have a valid route but next hop has expired or unreachable
	drop(p);          
	return;
      }
      else{
	rts[index3].hop_count=1;
	rts[index3].lifetime=now+HELLO_INTERVAL;
      }

      add_to_active_neighbors_list(rts[index].next_hop,src_ip);
    }
    cmn->next_hop()=next_hop;

    cmn->addr_type()=AF_INET;  
    iph->src()=addr_; 
    iph->dst()=src_ip;
    iph->dport_=0xff;

    sprintf(buff, "%d %d %d %d %d %f ", RREP,num_hops,0, 
	    dest,dest_seqno, lifetime);
    cmn->src_=addr_;
    p->init_databuf((unsigned char *)buff);
    if(p->accessdatalen()<1) {printf("DORK"); assert(0);}
    cmn->size_=44;
    assert(cmn->next_hop()!=addr_);
    s.schedule(target_,p,jitter(JIT, make_random_));
  }
  // save the broadcast so that you don't rebroadcast it again later -- that would cause a broadcast storm
  bi_last++;
  if(bi_last==NUM_BROADCASTS){
    printf("out of space for the broadcast ids\n");
    exit(3);
  }
  broadcast_ids[bi_last].id=broadcast_id;
  broadcast_ids[bi_last].dest=src_ip;
  broadcast_ids[bi_last].lifetime=now+(double)BCAST_ID_SAVE;
}


// sends an RREQ message
void AODV_Agent::send_request(nsaddr_t dest){
  char buff[RREQ_SIZE];
  int rindex, index;
  int dest_seqno;
  Scheduler& s=Scheduler::instance();
  double now=0;
 
  Packet* p=allocpkt();
  hdr_ip* iph = (hdr_ip*)p->access(off_ip_);
  hdr_cmn* ch=(hdr_cmn *)p->access(off_cmn_);
  iph->dst()=IP_BROADCAST;
  ch->next_hop()=IP_BROADCAST;
  ch->addr_type()=AF_INET;
  ch->src_=addr_;

  rindex=in_reply_queue(dest);

  if(rindex==-1){  // this seems like it's not necessary as rindex has got to be -1 at this point
    last_reply++;
    if(last_reply==NUM_REPLIES){
      last_reply=0;
    }
    replies[last_reply].dest=dest;
    replies[last_reply].num_retries=0;
    replies[last_reply].install_time=0;
    s.schedule(&handle_request_retry, 
	       new Event(),(double)RREP_WAIT_TIME);
  }
  index=route(dest);
  if(index==-1){  // we are sending a request for a route that was deleted or which we never knew about in the past
    dest_seqno=-1;
  }
  else { // we are sending a request for an expired or broken route
    dest_seqno=rts[index].seqnum;
  }

  sprintf(buff,"%d %d %d %d %d %d %d %d", RREQ, 0, 1, req_id++,dest,
	  dest_seqno, addr_,seqnum); 

  p->init_data(RREQ_SIZE, (unsigned char *)buff);
  ch->size_=52;
  assert(ch->next_hop()!=addr_);
  s.schedule(target_,p,jitter(JIT, make_random_));
}

// this function handles RREP messages
void AODV_Agent::handle_reply(Packet* p){
  int type, num_hops, L, dest_seqno,rindex, index;
  nsaddr_t dest;
  float lifetime;
  Scheduler& s=Scheduler::instance();

  double now=Scheduler::instance().clock();
  char buff[RREP_SIZE];
  hdr_cmn* ch = (hdr_cmn*)p->access(off_cmn_);
  hdr_ip* iph = (hdr_ip*)p->access(off_ip_);

  p->accessdata((unsigned char *)buff);    
 
  sscanf(buff, "%d %d %d %d %d %f ", &type,&num_hops,&L,   
	 &dest,&dest_seqno, &lifetime);                    
  rindex=in_reply_queue(dest);
                                                           
  if(iph->dst()!=addr_){
    assert(ch->src_!=addr_);  // we'd better not have received a RREP which we just fwded on--
    int index2=route(ch->src_);
    if(index2==-1){   // register the neighbor that sent the route reply, in case you don't have it in 
      add_route(ch->src_, -1,1, ch->src_,HELLO_INTERVAL);  // the route table
    }
    else{
      rts[index2].hop_count=1;
      rts[index2].lifetime=now+HELLO_INTERVAL;
    }
    // else {
      // not necessary, will be getting a beacon -- if we remove beacons tho yeah! update_route(index2, X, 1, ch->src_

    index=route(iph->dst()); 
    // we have to fwd the reply back but we don't have a route --  REV time has expired
    if(index==-1){
      printf("REV_ROUTE expired\n");
      drop(p);  
      return;
    }
    // in case this route to the destination from the RREQ is chosen by the source/the requestor, we would need to 
    // be able to forward packets to that destination
    int backindex=route(dest);
    if(backindex==-1){
      add_route(dest, dest_seqno, num_hops,ch->src_, lifetime);
    }
    else{
      // Note to myself: 
      // we are updating the route to the destination -- promiscuous action-- is this part of the protocol?
      update_route(backindex, dest_seqno, num_hops, ch->src_, lifetime);
    }
    ch->next_hop()=rts[index].next_hop;
    ch->addr_type()=AF_INET;  // this is probably already done
    ch->src_=addr_;
    assert(ch->next_hop()!=addr_);
    s.schedule(target_,p,jitter(JIT, make_random_));
    return;
  }
  else {  // the reply is for us 
    assert((index!=-1)||(iph->dst()==addr_));
    if(rindex==-1){  // we already got another reply for that request
      index=route(dest);
      if(index!=-1){
	int index3=route(ch->src_);
	if(index3==-1){
	  add_route(ch->src_, -1,1, ch->src_,HELLO_INTERVAL);
	}
	else{
	  rts[index3].hop_count=1;
	  rts[index3].lifetime=now+HELLO_INTERVAL;
	}
	update_route(index,dest_seqno,num_hops,ch->src_,lifetime);
	
      }
      Packet::free(p);
      return;
    }
    // this is the first reply to that request that we got
    replies[rindex].install_time=now;
    // num requests is maintained by the requester,as well as time init
    assert(dest==replies[rindex].dest);
    index=route(dest);
    if(index==-1){  // this shouldn't happen since we are creating dummy routes for RREQ'sted routes
       printf("its happenin\n");
      int index3=route(ch->src_);
      if(index3==-1){
	add_route(ch->src_, -1,1, ch->src_,HELLO_INTERVAL);
      }
      else{
	rts[index3].hop_count=1;
	rts[index3].lifetime=now+HELLO_INTERVAL;
      }
      add_route(dest,dest_seqno,num_hops,ch->src_,lifetime);
    }
    else{
      // first register the neighbor in your route table or update the route that you have for it
      int index3=route(ch->src_);
      if(index3==-1){  
	add_route(ch->src_, -1,1, ch->src_,HELLO_INTERVAL);
      }
      else{
	rts[index3].hop_count=1;
	rts[index3].lifetime=now+HELLO_INTERVAL;
      }
      update_route(index,dest_seqno,num_hops,ch->src_,lifetime);

      if(rts[index].q==0) {  // if nothing on the queue for that destination return
	Packet::free(p);
	return;
      }
      Packet::free(p);
      while(1){         // there are packets on the queue -- send them
	Packet *p1;
	p1=rts[index].q->deque();

	if(!p1) { 
	  delete rts[index].q;
	  rts[index].q=0;
	  rts[index].queue_length=0;
	  return;
	}
	rts[index].queue_length--;
	hdr_cmn* cmn1 = (hdr_cmn*)p1->access(off_cmn_);
	hdr_ip* iph1 = (hdr_ip*)p1->access(off_ip_);
	cmn1->next_hop()=rts[index].next_hop;
	cmn1->addr_type()=AF_INET;
	cmn1->src_=addr_;

	assert(cmn1->next_hop()!=addr_);
	s.schedule(target_,p1,jitter(JIT, 0));
      }
    }
  }
}

// handle usolicited reply messages for broken routes

void AODV_Agent::handle_unsolicited_reply(Packet* p){
   hdr_ip* iph = (hdr_ip*)p->access(off_ip_);
   char buff[UREP_SIZE];
   int type, L, seqno, metric,index; 
   nsaddr_t dest;
 
#ifdef JOSH
   assert(strlen(p->accessdata()) < sizeof(buff) - 1);
#endif
   p->accessdata((unsigned char*)buff);
   sscanf(buff,"%d %d %d %d %d", &type, &L,&metric,&dest,&seqno);
   
   if((iph->src())==addr_){
     printf("got my own unsol reply\n");
     Packet::free(p);
     return;
   }

 if(in_reply_queue(dest)!=-1){
    //    printf("attempting to delete a RREQ-ed route\n");
    return;
  }
   index=route(dest);
   if(index==-1){   // you had a route but it got deleted in the meantime?
//printf("%d: got an unsolicited reply for a dest ( %d )I no longer have a route to\n",addr_, dest);
     Packet::free(p);
     return;
   }
   else if(rts[index].hop_count==INFINITY){  
     Packet::free(p);  // I already know the route is broken
     return;
   }
   Packet::free(p);
   if(rts[index].destination<1) printf("DEST 0 IN HANDLE_UNSOLICITED_REPLY\n");
   send_unsolicited_reply(index);
}


AODV_Agent::AODV_Agent() : Agent(PT_MESSAGE), update_timer_(this), 
  handle_broken_link(this), handle_request_retry(this), make_random_(1) { 
  interval_=HELLO_INTERVAL;
  end_of_table=-1;
  seqnum=0;
  req_id=0;
  next_event=-1;
  last_event=-1;
  next_reply=0;
  last_reply=-1;
  bi_last=-1;
  bind("node_",&node_);
  bind("portID_",&portID_);
  bind("make_random_",&make_random_);
}


void AODV_Agent::timeout(int p)
{
    send_hello();
    update_timer_.resched(interval_);
}

void UPDATE_Timer::expire(Event *e)
{ 
  a_->timeout(0);
}


static class AODVClass : public TclClass {
public:
  AODVClass() : TclClass("Agent/AODV")
  {}
  TclObject* create(int argc, const char*const* argv){
    return (new AODV_Agent());
  }
}class_aodv;

int AODV_Agent::command(int argc, const char*const* argv){

  Tcl& tcl = Tcl::instance();
  double now=0;
  if(argc>1){
    if(strcmp(argv[1],"start")==0){
      timeout(0);
      return(TCL_OK);
    }
    if (argc == 3) {
      if (strcmp(argv[1], "if-queue") == 0) {
	ifqueue = (PriQueue*)TclObject::lookup(argv[2]);
	if (ifqueue == 0) {
	  tcl.resultf("no such object %s", argv[2]);
	  return (TCL_ERROR);
	}
	return (TCL_OK);
      }
    }
  }
    return (Agent::command(argc, argv));
}


// checks if we have sent out a RREQ for destination dest
int AODV_Agent::in_reply_queue(nsaddr_t dest){
  int i;

  if(last_reply==-1){
    return -1;
  }

  if(last_reply>next_reply){
    for(i=next_reply;i<last_reply+1;i++){
      if(replies[i].dest==dest){
	return i;
      }
    }
  }
  else if(last_reply==next_reply){
      if(replies[last_reply].dest==dest){
	return last_reply;
      }
  }
  else {
    for(i=next_reply;i<NUM_REPLIES;i++){
      if(replies[i].dest==dest){
	return i;
      }
    }
    for(i=0;i<last_reply+1;i++){
      if(replies[i].dest==dest){
	return i;
      }
    }
  }
  return -1;
}


