/* -*- c++ -*-
   aodv.h
   $I`d$
   */

#ifndef __aodv_h__
#define __aodv_h__

#include <sys/types.h>
#include <cmu/list.h>

#include <agent.h>
#include <packet.h>
#include <scheduler.h>

#include <cmu/cmu-trace.h>
#include <cmu/priqueue.h>
#include <cmu/rtqueue.h>
#include <cmu/rttable.h>

/*
 * Allows AODV to use link-layer (802.11) feedback in determining when
 * links are up/down.
 */
#define AODV_LINK_LAYER_DETECTION

/*
 *  Causes AODV to apply a "smoothing" function to the link layer feedback
 *  that is generated by 802.11.  In essence, it requires that RT_MAX_ERROR
 *  errors occurs within a window of RT_MAX_ERROR_TIME before the link
 *  is considered bad.
 */
//#define AODV_USE_LL_METRIC

/*
 *  Only applies if AODV_USE_LL_METRIC is defined.
 *
 *  Causes AODV to apply omniscient knowledge to the feedback received
 *  from 802.11.  This may be flawed, because it does not account for
 *  congestion.
 */
//#define AODV_USE_GOD_FEEDBACK


class AODV;

#define AODV_HDR_LEN    64      // amount of space allocated in the pkt hdr

#define ID_NOT_FOUND    0x00
#define INFINITY        0xff

/*
 * Constants defined in draft-ietf-manet-aodv-00.txt
 */
#define ACTIVE_ROUTE_TIMEOUT    300             // seconds
// #define ALLOWED_HELLO_LOSS      2               // packets
#define ALLOWED_HELLO_LOSS      3               // packets
#define BAD_LINK_LIFETIME       3               // 3000 ms
#define BCAST_ID_SAVE           3               // 3000 ms
#define HELLO_INTERVAL          1               // 1000 ms
#define NETWORK_DIAMETER        100             // hops
#define NODE_TRAVERSAL_TIME     0.4             // 400 ms
#define MY_ROUTE_TIMEOUT        600             // seconds
#define REV_ROUTE_LIFE          3               // 3000 ms
// #define RREP_WAIT_TIME          (3 * NODE_TRAVERSAL_TIME * NETWORK_DIAMETER) // ms
#define RREP_WAIT_TIME          (2 * REV_ROUTE_LIFE)  // seconds
#define RREQ_RETRIES            3


#define MaxHelloInterval        (1.25 * HELLO_INTERVAL)
#define MinHelloInterval        (0.75 * HELLO_INTERVAL)


/* =====================================================================
   Timers (Broadcast ID, Hello, Neighbor Cache, Route Cache)
   ===================================================================== */
class BroadcastTimer : public Handler {
public:
        BroadcastTimer(AODV* a) : agent(a) {}
        void	handle(Event*);
private:
        AODV    *agent;
	Event	intr;
};

class HelloTimer : public Handler {
public:
        HelloTimer(AODV* a) : agent(a) {}
        void	handle(Event*);
private:
        AODV    *agent;
	Event	intr;
};

class NeighborTimer : public Handler {
public:
        NeighborTimer(AODV* a) : agent(a) {}
        void	handle(Event*);
private:
        AODV    *agent;
	Event	intr;
};

class RouteCacheTimer : public Handler {
public:
        RouteCacheTimer(AODV* a) : agent(a) {}
        void	handle(Event*);
private:
        AODV    *agent;
	Event	intr;
};


/* =====================================================================
   Broadcast ID Cache
   ===================================================================== */
class BroadcastID {
        friend class AODV;
 public:
        BroadcastID(nsaddr_t i, u_int32_t b) { src = i; id = b; }

 protected:
        LIST_ENTRY(BroadcastID) link;
        nsaddr_t        src;
        u_int32_t       id;
        double          expire;         // now + BCAST_ID_SAVE ms
};

LIST_HEAD(bcache, BroadcastID);


/* =====================================================================
   The Routing Agent
   ===================================================================== */
class AODV: public Agent {
        friend class rt_entry;
        friend class BroadcastTimer;
        friend class HelloTimer;
        friend class NeighborTimer;
        friend class RouteCacheTimer;
 public:
        AODV(nsaddr_t id);

        void		recv(Packet *p, Handler *);

        /*
         * HDR offsets
         */
        int             off_AODV_;

 protected:
        int             command(int, const char *const *);
        int             initialized() { return index && target_; }

        /*
         * Route Table Management
         */
        void            rt_resolve(Packet *p);
        void            rt_down(rt_entry *rt);
 public:
        void            rt_ll_failed(Packet *p);
 protected:
        void            rt_purge(void);

        void            enque(rt_entry *rt, Packet *p);
        Packet*         deque(rt_entry *rt);

        /*
         * Neighbor Management
         */
        void            nb_insert(nsaddr_t id);
        Neighbor*       nb_lookup(nsaddr_t id);
        void            nb_delete(nsaddr_t id);
        void            nb_purge(void);

        /*
         * Broadcast ID Management
         */
        void            id_insert(nsaddr_t id, u_int32_t bid);
        u_int32_t       id_lookup(nsaddr_t id);
        void            id_purge(void);

        /*
         * Packet TX Routines
         */
        void            forward(rt_entry *rt, Packet *p, int jitter);
        void            sendHello(void);
        void            sendRequest(nsaddr_t dst);
        void            sendReply(nsaddr_t ipdst, u_int32_t hop_count,
                                  nsaddr_t rpdst, u_int32_t rpseq,
                                  u_int32_t lifetime, double timestamp);
        void            sendTriggeredReply(nsaddr_t ipdst, nsaddr_t rpdst,
                                           u_int32_t rqseq);

        /*
         * Packet RX Routines
         */
        void            recvAODV(Packet *p);
        void            recvHello(Packet *p);
        void            recvRequest(Packet *p);
        void            recvReply(Packet *p);
        void            recvTriggeredReply(Packet *p);

        /* ============================================================ */

        nsaddr_t        index;                  // IP Address of this node
        int             seqno;                  // Sequence Number
        int             bid;                    // Broadcast ID

        ncache          nbhead;                 // Neighbor Cache
        bcache          bihead;                 // Broadcast ID Cache

        /*
         * Timers
         */
        BroadcastTimer  btimer;
        HelloTimer      htimer;
        NeighborTimer   ntimer;
        RouteCacheTimer rtimer;


        /*
         * Routing Table
         */
        rttable          rtable;
        /*
         *  A "drop-front" queue used by the routing layer to buffer
         *  packets to which it does not have a route.
         */
        rtqueue         rqueue;

        /*
         * A mechanism for logging the contents of the routing
         * table.
         */
        Trace           *logtarget;

        /*
         * A pointer to the network interface queue that sits
         * between the "classifier" and the "link layer".
         */
        PriQueue        *ifqueue;

        /*
         * Logging stuff
         */
        void            log_link_del(nsaddr_t dst);
        void            log_link_broke(Packet *p);
        void            log_link_kept(nsaddr_t dst);
};

#endif /* __aodv_h__ */
