
#include "libos.h"

// This code does not allocate or initialize its BSS (section for
// uninitialized data).  Thus, we force __env to be located in the
// data section by initializing it.
//
struct Env *__env = 0;

asm (".text");
asm (".global ___start");
asm ("___start:");
asm ("        call ___main");


// On entry there a trap frame is on the stack: 
//   [ 5 spare words ]
//   trap time eip
//   trap time eflags
//   trap time esp
//   trap time errcode
//   trap time va  <-- %esp 
void asm_pgfault_handler ();
asm (".text");
asm (".global _asm_pgfault_handler");
asm ("_asm_pgfault_handler:");
// caller saves
asm ("    movl %eax,20(%esp)");
asm ("    movl %edx,24(%esp)");
asm ("    movl %ecx,28(%esp)");
asm ("    call _pgfault_handler");
// copy trap time eip and eflags to trap time stack

// restore caller saves

// load esp with trap time stack - 8

// eflags and eip are waiting us there
asm ("    popf");
asm ("    ret");



void
putx (u_int number)
{
  char *hex[] = {"0","1","2","3","4","5","6","7",
		"8","9","a","b","c","d","e","f"};
  int shift;
  sys_cputs ("0x");
  for (shift = 28; shift >= 0 ; shift -= 4) {
    u_char digit = (number >> shift) & 0xf;
    sys_cputs (hex[digit]);
  }
}

/* poor man's printf */
void
print (char *prefix, int number, char *suffix)
{
  sys_cputs (prefix);
  sys_cputu (number);
  sys_cputs (suffix);
}

void
printx (char *prefix, int number, char *suffix)
{
  sys_cputs (prefix);
  putx (number);
  sys_cputs (suffix);
}

void
exit ()
{
  print ("envid ", sys_getenvid(), " exiting..\n");
  sys_env_destroy ();
  /* NOT REACHED */
  sys_cputs ("FATAL: sys_env_destroy failed.\n");
  while (1)
    ;
}

void
xpanic (char *prefix, int number, char *suffix)
{
  print (prefix, number, suffix);
  exit ();
}


// like bcopy (but doesn't handle overlapping src/dst)
void
xbcopy (const void *src, void *dst, size_t len)
{
  void *max = dst + len;
  // copy 4bytes at a time while possible
  while (dst + 3 < max)
    *((u_int *)dst)++ = *((u_int *)src)++;
  // finish remaining 0-3 bytes
  while (dst < max)
    *(char *)dst++ = *(char *)src++;
}


void
handle_cow_fault (u_int va, Pte pte)
{
}

void
pgfault_handler (u_int va, u_int err, u_int esp, u_int eflags, u_int eip)
{
#if 0
  printx ("PGFLT [va ", va, ""); 
  printx (", err ", err, ""); 
  printx (", esp ", esp, ""); 
  printx (", eflags ", eflags, ""); 
  printx (", eip ", eip, "]\n"); 
#endif



}


// Sends 'value' to 'target_envid'.
// This function does not return until the value
// has been successfully sent.  And it should xpanic
// on any error other than -E_IPC_BLOCK.
// Hint: use sys_yield() to be CPU friendly.
void
ipc_send (u_int target_envid, u_int value)
{
}



// Loops until an incoming IPC is received.
//   - returns the IPC value
//   - *from_envid is set to the sender
//
// Hint: loop something like this:
//    while (!*(volatile int *)&(__env->env_ipc_blocked))
//           sys_yield ();
//
// The volatile tells the compiler to to cache __env->env_ipc_blocked
// in a registers.  Otherwise, we'll never see when that field is set.
//
// Make sure there are no race conditions in your code.  You must
// guarantee that each IPC sent is received exactly once.
//
u_int
ipc_read (u_int *from_envid)
{
}

// an example of vpt[] and vpd[] 
// to dump the contents of the current process's 
// pgdir and page table.
//
void
dump_va_space ()
{
  int i;
  u_int va;
  Pde pde;
  Pte pte;

  // 0x800cbc, va 0x801000

  // try both of these..
  //u_int va_lim = 0xffffffff;
  u_int va_lim = UTOP; 
  for (va = 0; va < va_lim; ) {
    pde = vpd[PDENO(va)];

    if (!(pde & PG_P)) {
      va += NBPD;
      continue;
    }

    sys_cputs ("PGDIR[");
    sys_cputu (PDENO(va));
    printx ("] = ", pde, "\n");

    for (i = 0; i < NLPG; i++, va += NBPG) {
      pte = vpt[PGNO(va)];
      if (pte & PG_P) {
	sys_cputs ("    PGTBL[");
	sys_cputu (i);
	printx ("] = ", pte, "\n");
      }
    }
  }
}

// fork()
//
// RETURNS
//   on success
//      child's envid to the parent
//      0 to the child
//   on error
//      just do whatever (call xpanic, return <0) doesn't matter
//
// Hint: set __env correctly in the child.
//

int
fork (void)
{
}




// spawn a child process and return its envid
//
int
spawn (void (*child_fn) (void))
{
  int ret = fork ();
  if (ret < 0) {
    xpanic ("ERR: sys_env_fork, ret = ", ret, "\n");
  } else if (ret == 0) { // child see a 0 return value
    (*child_fn) ();
    sys_cputs ("ERR: spawn: child returned\n");
    exit ();
  }

  return ret; // return child's envid to parent
}



void
setup_xstack ()
{
  sys_cputs ("setup_xstack: commented out until sys_mem_alloc is implemented\n");
  while (1)
    ;

#if 0
  int ret;
  // allocate exception stack
  ret = sys_mem_alloc (0, UXSTACKTOP - NBPG, PG_P|PG_U|PG_W);
  if (ret < 0)
    xpanic ("Couldn't alloc exception stack, ret ", ret, "\n");
  
  // setup pgfault handler
  sys_set_pgfault_handler ((u_int)&asm_pgfault_handler, UXSTACKTOP);
#endif
}




void
exercise2 (void)
{
  int ret;
  int envid = sys_getenvid ();
  sys_cputs ("Hello from user space\n");
  sys_cputs ("My envid is: ");
  sys_cputu (envid);
  sys_cputs ("\n");
  if (envid != 2049)
    sys_cputs (" *** ERROR: expected 2049\n");


  // now test bad sys_call number
  ret = sys_call (100, 0, 0, 0);
  if (ret != -E_INVAL)
    sys_cputs (" *** ERROR: expected -E_INVAL for bad syscall number\n");


  while (1)
    ;
}

void
exercise3 (void)
{
  extern void pingpong_loop (char *);

  int envid = sys_getenvid ();
  if (envid != 2049) {
    ipc_send (2049, 0);
    pingpong_loop ("2nd init process");
  } else {
    pingpong_loop ("1st init process");
  }
}

void
exercise4_test1 (void)
{
  sys_cputs ((char *)0);
}

void
exercise4_test2 (void)
{
  sys_cputs ((char *)ULIM);
}

void
exercise4_test3 (void)
{
  sys_cputs ((char *)0xffffffff);
}

void
exercise5_test1 (void)
{
  setup_xstack ();
  // fault writing to VA 0 
  *(char *)0 = 'A';
}

void
exercise5_test2 (void)
{
  setup_xstack ();
  // fault writing to VA 0xff 
  *(char *)0xff = 'A';
}

//
//RETURNS: 1 + 2 + ... + depth
//
u_int
exercise5_test3_helper (u_int depth)
{
  if (depth == 1)
    return 1;
  else
    return depth + exercise5_test3_helper (depth - 1);
}

// When a user process starts the kernel has only allocatd
// one page (4096bytes) of stack for it.  We quickly 
// consume this with a deeply recursive call.  The 
// *USER'S* page fault handler routine should call
// sys_mem_alloc to add another page to the stack, as needed.
//
// You might as well add some policy to your page fault handler.  For
// instance, if the stack exceeds some maximum size the page fault
// handler might want to assume that there has been a programming
// error and simply exit.
//

void
exercise5_test3 (void)
{
  setup_xstack ();
  u_int x = 4000;
  u_int i = exercise5_test3_helper (x);
  if (i != x*(x + 1)/2)
    xpanic ("exercise5_test3 failed, i=", i, "\n");
}

// The kernel must be careful when putting the trap time values on 
// the user exception stack.  The user exception stack pointer is a value
// that comes from the user process, so the kernel can't trust that it 
// points at valid memory.  In fact, here the user process maliciously
// supplies a bad exception stack pointer. 
//
// The kernel should use the page_fault_mode/pfm() technique when it
// puts the trap time values on the user exception stack.  Otherwise
// the kernel might crash or corrupt memory.  
//
// Do not implement protections in sys_set_pgfault_handler, the MUST
// be done dynamically in the kernel's page fault handler.
//

void
exercise5_test4 (void)
{
  // if the kernel pushes on to the address "0", the pointer 
  // will wrap around and the kernel will overwrite the
  // top part of memory.
  sys_set_pgfault_handler ((u_int)&asm_pgfault_handler, 0);
  *(char *)0 = 'A';
}


void
exercise5_test5 (void)
{
  // try to get the kernel to overwrite the kernel stack
  sys_set_pgfault_handler ((u_int)&asm_pgfault_handler, KSTACKTOP);
  *(char *)0 = 'A';
}


void
__main (void)
{
  DEF_SYM (__envs, UENVS);
  DEF_SYM (__ppages, UPPAGES);
  DEF_SYM (vpt, UVPT);
  DEF_SYM (vpd, UVPT + SRL(UVPT, 10));
  __env = &__envs[envidx(sys_getenvid ())];

  exercise2 ();
  //exercise3 ();
  //exercise4_test1 ();
  //exercise4_test2 ();
  //exercise4_test3 ();
  //exercise5_test1 ();
  //exercise5_test2 ();
  //exercise5_test3 ();
  //exercise5_test4 ();
  //exercise5_test5 ();

  //// run these lines for exercise 6
  setup_xstack ();
  umain ();
  exit ();


}


