#include <kern/inc/pmap.h>
#include <kern/inc/mmu.h>
#include <kern/inc/trap.h>
#include <kern/inc/env.h>
#include <kern/inc/syscall.h>
#include <kern/inc/scheduler.h>
#include <kern/inc/console.h>
#include <kern/inc/printf.h>
#include <kern/inc/picirq.h>
#include <kern/inc/kclock.h>

u_int page_fault_mode;
static struct Ts ts;

/* Interrupt descriptor table.  (Must be built at run time because
 * shifted function addresses can't be represented in relocation records.)
 */
struct gate_desc idt[256] = { {0}, };
struct pseudo_desc idt_pd =
{
  0, sizeof (idt) - 1, (unsigned long) idt,
};


void
idt_init ()
{
  extern struct seg_desc gdt[];

  // Setup a TSS so that we get the right stack when we trap to the kernel.
  ts.ts_esp0 = KSTACKTOP;
  ts.ts_ss0 = GD_KD;

  // Love to put this code in the initialization of gdt, but
  // the compiler generates an error incorrectly.
  gdt[GD_TSS >> 3] = seg (STS_T32A, (u_long) (&ts), sizeof (struct Ts), 0);
  gdt[GD_TSS >> 3].sd_s = 0;

  ltr (GD_TSS);
  asm volatile ("lidt _idt_pd+2");
}


static void
print_trapframe (struct Trapframe *tf)
{
  printf ("%%%% Trap 0x%x (errcode 0x%x)\n", tf->tf_trapno, 0xffff & tf->tf_err);
  printf ("   cs:eip 0x%x:0x%x; ", 0xffff & tf->tf_cs, tf->tf_eip);
  printf ("ss:esp 0x%x:0x%x; ", 0xffff & tf->tf_ss, tf->tf_esp);
  printf ("ebp 0x%x\n", tf->tf_ebp);
  printf ("   eax 0x%x; ", tf->tf_eax);
  printf ("ebx 0x%x; ", tf->tf_ebx);
  printf ("ecx 0x%x; ", tf->tf_ecx);
  printf ("edx 0x%x;\n", tf->tf_edx);
  printf ("   eflags 0x%x; ", tf->tf_eflags);
  printf ("edi 0x%x; ", tf->tf_edi);
  printf ("esi 0x%x; ", tf->tf_esi);
  printf ("ds 0x%x; ", 0xffff & tf->tf_ds);
  printf ("es 0x%x;\n", 0xffff & tf->tf_es);
}


void
trap (struct Trapframe *tf)
{
  print_trapframe (tf);

  if (tf->tf_trapno == T_PGFLT) {
    page_fault_handler (tf);
  }
  else if (tf->tf_trapno == T_SYSCALL) {
  }
  else if (tf->tf_trapno == IRQ_OFFSET) {
    // irq 0 -- clock interrupt
  }
  else if (tf->tf_trapno >= IRQ_OFFSET && 
	   tf->tf_trapno < IRQ_OFFSET + MAX_IRQS) {
    // just ignore spurious interrupts
    u_int irq = tf->tf_trapno - IRQ_OFFSET;
    printf ("ignoring unexpected IRQ %d:", irq);
    printf (" eip 0x%x;", tf->tf_eip);
    printf (" cs = 0x%x;", tf->tf_cs & 0xffff);
    printf (" eflags = 0x%x\n", tf->tf_eflags);
  } else {
    // the user process or the kernel has a bug..
    print_trapframe (tf);
    panic ("That does it (Unhandled trap).");
  }

  panic ("in lab3 don't continue..");


}


void
page_fault_handler (struct Trapframe *tf)
{
  u_int va = rcr2 ();
#if 0
  u_int env_id = curenv ? curenv->env_id : -1;
  printf ("%%%% [0x%x] Page fault (code 0x%x) for VA 0x%x (env 0x%x)\n"
	  "   eip = 0x%x, cs = 0x%x, eflags = 0x%x\n",
	  tf->tf_trapno, 0xffff & tf->tf_err, va, env_id,
	  tf->tf_eip, 0xffff & tf->tf_cs, tf->tf_eflags);
  /* Only traps from user mode push %ss:%esp */
  if (tf->tf_err & FEC_U)
    printf ("   esp = 0x%x, ss = 0x%x\n", tf->tf_esp, 0xffff & tf->tf_ss);
#endif

}

